package com.ibm.ims.dli.converters;

/**
 * A <code>ConverterFactory</code> object instantiates a converter for a
 * specific Java data type. <code>Converter</code> objects convert an
 * input/output byte array from an IMS DL/I database into a standard Java data
 * type or vice versa.
 * <p>
 * Before using these objects, you must know the position of the target IMS
 * database field in the input/output byte array and the type of data in the
 * byte array. These converters do not provide type detection.
 * <p>
 * These converters are intended to be used as programming aids to help you
 * create type converter classes for custom DL/I data types.
 *
 */
public class ConverterFactory {

	/**
	 * Creates a <code>DoubleTypeConverter</code> object.
	 * 
	 * @return the <code>DoubleTypeConverter</code> object
	 */
	public static DoubleTypeConverter createDoubleConverter() {
		return new DoubleTypeConverterImpl();
	}

	/**
	 * Creates a <code>FloatTypeConverter</code> object.
	 * 
	 * @return the <code>FloatTypeConverter</code> object
	 */
	public static FloatTypeConverter createFloatConverter() {
		return new FloatTypeConverterImpl();
	}

	/**
	 * Creates an <code>IntegerTypeConverter</code> object.
	 * 
	 * @return the <code>IntegerTypeConverter</code> object
	 */
	public static IntegerTypeConverter createIntegerConverter() {
		return new IntegerTypeConverterImpl();
	}

	/**
	 * Creates a <code>LongTypeConverter</code> object.
	 * 
	 * @return the <code>LongTypeConverter</code> object
	 */
	public static LongTypeConverter createLongConverter() {
		return new LongTypeConverterImpl();
	}

	/**
	 * Creates a <code>PackedDecimalTypeConverter</code> object.
	 * 
	 * @param precision
	 *            the number of digits in a PackedDecimal
	 * @param scale
	 *            the number of digits to the right of the decimal point in a
	 *            PackedDecimal
	 * @param isSigned
	 *            determines if the PackedDecimal is signed or unsigned
	 * @return the <code>PackedDecimalTypeConverter</code> object
	 */
	public static PackedDecimalTypeConverter createPackedDecimalConverter(int precision, int scale, boolean isSigned) {
		return new PackedDecimalTypeConverterImpl(precision, scale, isSigned);
	}

	/**
	 * Creates a <code>ZonedDecimalTypeConverter</code> object.
	 * 
	 * @param precision
	 *            the number of digits in a ZonedDecimal
	 * @param scale
	 *            the number of digits to the right of the decimal point in a
	 *            ZonedDecimal
	 * @param isSigned
	 *            determines if the ZonedDecimal is signed or unsigned
	 * @return the <code>ZonedDecimalTypeConverter</code> object
	 */
	public static ZonedDecimalTypeConverter createZonedDecimalConverter(int precision, int scale, boolean isSigned) {
		return new ZonedDecimalTypeConverterImpl(precision, scale, isSigned);
	}

	/**
	 * Creates a <code>ShortTypeConverter</code> object.
	 * 
	 * @return the <code>ShortTypeConverter</code> object
	 */
	public static ShortTypeConverter createShortConverter() {
		return new ShortTypeConverterImpl();
	}

	/**
	 * Creates a <code>StringTypeConverter</code> object.
	 * 
	 * @param encoding
	 *            the encoding of the String
	 * @param isDBCS
	 *            if the String is part of a double byte character set
	 * @return the <code>StringTypeConverter</code> object
	 */
	public static StringTypeConverter createStringConverter(String encoding, boolean isDBCS) {
		return new StringTypeConverterImpl(encoding, isDBCS);
	}

	/**
	 * Creates a <code>UShortTypeConverter</code> object.
	 * 
	 * @return the <code>UShortTypeConverter</code> object
	 */
	public static UShortTypeConverter createUShortConverter() {
		return new UShortTypeConverterImpl();
	}

	/**
	 * Creates a <code>UIntegerTypeConverter</code> object.
	 * 
	 * @return the <code>UIntegerTypeConverter</code> object
	 */
	public static UIntegerTypeConverter createUIntegerConverter() {
		return new UIntegerTypeConverterImpl();
	}

	/**
	 * Creates a <code>ULongTypeConverter</code> object.
	 * 
	 * @return the <code>ULongTypeConverter</code> object
	 */
	public static ULongTypeConverter createULongConverter() {
		return new ULongTypeConverterImpl();
	}
}
