package com.ibm.ims.dli.tm;

import java.nio.ByteBuffer;

import com.ibm.ims.dli.DBQueryInfo;

/* (c) Copyright International Business Machines Corporation 2013. All rights reserved. */

import com.ibm.ims.dli.DLIException;
import com.ibm.ims.dli.EnvironInfo;
import com.ibm.ims.dli.FindInfo;
import com.ibm.ims.dli.MessageInfo;
import com.ibm.ims.dli.ProgramInfo;
import com.ibm.ims.jms.IMSQueueConnectionFactory;

/**
 * The <code>Application</code> interface is the starting point of a IMS transaction management application. 
 * Use this interface  to create the <code>Transaction</code>, <code>MessageQueue</code>
 * and <code>IOMessage</code> objects.
 * <p>For more details about the programming model for Java message processing (JMP) applications, see the topic
 * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_jmpprogmodels.htm" target="_blank">JMP programming models</a>" 
 * in the IMS Application Programming guide documentation.
 * <p>For more details about the programming model for Java batch processing (JBP) applications, see the topic
 * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_developingjbpapps.htm" target="_blank">Developing JBP applications with the IMS Java dependent region resource adapter</a>" 
 * in the IMS Application Programming guide documentation.
 */
public interface Application {

    /**
     * Retrieves a handle for the <code>Transaction</code>
     * object that provides support for transactions that are managed by IMS.
     * 
     * @return the <code>Transaction</code> object
     */
    public Transaction getTransaction();

    /**
     * Retrieves a handle to the <code>MessageQueue</code>
     * object that provides support for IMS message queue processing.
     * 
     * @return the <code>MessageQueue</code> object
     */
    public MessageQueue getMessageQueue();

    /**
     * Retrieves a handle to the <code>IOMessage</code>
     * object that represents messages that are coming from or going to an
     * IMS message queue.
     * <p>The following code example shows how an <code>IOMessage</code> object is retrieved:
     * <blockquote>
     * <pre>
     *    Application app = ApplicationFactory.createApplication();
     *    IOMessage msg = app.getIOMessage("class://myPackage.MyIMSFldMsgSubClass");
     * </pre>
     * </blockquote>
     * <p>For more details on defining input and output message classes, see the topic
     * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_subclassingimsfieldmsgsinput.htm" target="_blank">Defining the input and output message classes</a>" 
     * in the IMS Application Programming guide documentation.
     * @param url
     *            the URL representing a subclass of <code>com.ibm.ims.application.IMSFieldMessage</code>
     * @return the <code>IOMessage</code> object
     * @throws DLIException
     *             if an error occurs during processing
     */
    public IOMessage getIOMessage(String url) throws DLIException;

    /**
     * Retrieves a handle to the <code>AssemblerCall</code>
     * object that represents an assembler module to invoke.
     * <p>The following code example shows how an <code>AssemblerCall</code> object is retrieved and used
     * to invoke an assembler module named MYMOD with an input parameter of type String:
     * <blockquote>
     * <pre>
     *    Application app = ApplicationFactory.createApplication();
     *    AssemblerCall asCall = app.getAssemblerCall();
     *    asCall.setModuleName("MYMOD");
     *    asCall.setReturnType(AssemblerCall.RETURN_VOID);
     *    asCall.setString(1, "CharacterInput");
     *    asCall.invoke();
     * </pre>
     * </blockquote>
     * 
     * @return the <code>AssemblerCall</code> object
    */
    
    public AssemblerCall getAssemblerCall();
    
    /**
     * Retrieves a handle to the <code>IMSQueueConnectionFactory</code>
     * object that represents a Factory class used to create IMS Point-to-Point messaging 
     * domain QueueConnection objects.
     * 
     * @return the <code>IMSQueueConnectionFactory</code> object
    */
    public IMSQueueConnectionFactory getIMSQueueConnectionFactory();
   
    /**
     * Retrieves a handle to the <code>IMSCallout</code> object
     * needed to execute an IMS Callout request. IMSCallout is a Java
     * wrapper for the ICAL DL/I call.
     * 
     * @return the <code>IMSCallout</code> object
     */
    public IMSCallout createImsCallout();
    
    /**
     * Issues a DLI INQY call with the subfunction DBQUERY
     * 
     * <p>The following code example shows how to issue the call:
     * <blockquote>
     * <pre>
     *    Application app = ApplicationFactory.createApplication();
     *    DBQueryInfo info = app.inqyDBQuery();
     * </pre>
     * </blockquote>
     * <p>For more details on the INQY call and the DBQUERY subfunction, see the topic
     * "<a href="https://www.ibm.com/support/knowledgecenter/SSEPH2_15.2.0/com.ibm.ims152.doc.apr/ims_inqycalltm.htm" target="_blank">INQY call</a>" 
     *
     * @return the <code>DBQueryInfo</code> object which contains the information returned from the INQY call.
     * @throws Exception
     */
    public DBQueryInfo inqyDBQuery() throws Exception;
    
    /**
     * Issues a DLI INQY call with the subfunction ENVIRON
     * 
     * <p>The following code example shows how to issue the call:
     * <blockquote>
     * <pre>
     *    Application app = ApplicationFactory.createApplication();
     *    EnvironInfo info = app.inqyEnviron();
     * </pre>
     * </blockquote>
     * <p>For more details on the INQY call and the ENVIRON subfunction, see the topic
     * "<a href="https://www.ibm.com/support/knowledgecenter/SSEPH2_15.2.0/com.ibm.ims152.doc.apr/ims_inqycalltm.htm" target="_blank">INQY call</a>" 
     *
     * @return the <code>EnvironInfo</code> object which contains the information returned from the INQY call.
     * @throws Exception
     */
    public EnvironInfo inqyEnviron() throws Exception;
    
    /**
     * Issues a DLI INQY call with the subfunction MSGINFO
     * 
     * <p>The following code example shows how to issue the call:
     * <blockquote>
     * <pre>
     *    Application app = ApplicationFactory.createApplication();
     *    MessageInfo info = app.inqyMessageInfo();
     * </pre>
     * </blockquote>
     * <p>For more details on the INQY call and the MSGINFO subfunction, see the topic
     * "<a href="https://www.ibm.com/support/knowledgecenter/SSEPH2_15.2.0/com.ibm.ims152.doc.apr/ims_inqycalltm.htm" target="_blank">INQY call</a>" 
     *
     * @return the <code>MessageInfo</code> object which contains the information returned from the INQY call.
     * @throws Exception
     */
    public MessageInfo inqyMessageInfo() throws Exception;
    
    /**
     * Issues a DLI INQY call with the subfunction FIND
     * 
     * <p>The following code example shows how to issue the call:
     * <blockquote>
     * <pre>
     *    Application app = ApplicationFactory.createApplication();
     *    FindInfo info = app.inqyFind("PCBName");
     * </pre>
     * </blockquote>
     * <p>For more details on the INQY call and the FIND subfunction, see the topic
     * "<a href="https://www.ibm.com/support/knowledgecenter/SSEPH2_15.2.0/com.ibm.ims152.doc.apr/ims_inqycalltm.htm" target="_blank">INQY call</a>" 
     * @param PCB name
     * 		  
     * @return the <code>FindInfo</code> object which contains the information returned from the INQY call.
     * @throws Exception
     */
    public FindInfo inqyFind(String pcb) throws Exception;
    
    /**
     * Issues a DLI INQY call with the subfunction PROGRAM
     * 
     * <p>The following code example shows how to issue the call:
     * <blockquote>
     * <pre>
     *    Application app = ApplicationFactory.createApplication();
     *    ProgramInfo info = app.inqyProgram();
     * </pre>
     * </blockquote>
     * <p>For more details on the INQY call and the PROGRAM subfunction, see the topic
     * "<a href="https://www.ibm.com/support/knowledgecenter/SSEPH2_15.2.0/com.ibm.ims152.doc.apr/ims_inqycalltm.htm" target="_blank">INQY call</a>" 
     *
     * @return the <code>ProgramInfo</code> object which contains the information returned from the INQY call.
     * @throws Exception
     */
    public ProgramInfo inqyProgram() throws Exception;
    
	/**
	 * Allocates a 31-bit direct ByteBuffer. Buffers that are not required to be 31-bit
	 * should be directly allocated using ByteBuffer.allocateDirect(int capacity)
	 * 
	 * @see ByteBuffer.allocateDirect
	 * @param  capacity
     *         The new buffer's capacity, in bytes
	 * @return the <code>ByteBuffer</code> object
	 * @throws DLIException 
	 */
    public ByteBuffer get31BitDirectByteBuffer(int size) throws DLIException;
    
    /**
     * Frees a 31 bit direct ByteBuffer
     * @param byteBuffer
     * 		  The ByteBuffer object to free
     * @throws DLIException 
     */
    public void free31BitDirectByteBuffer(ByteBuffer byteBuffer) throws DLIException;
    
    /**
     * Ends the life-cycle of the <code>Application</code> instance and frees up any resources associated
     * with it.
     */
    public void end();
}
