package com.ibm.ims.dli.tm;

import java.math.BigDecimal;

import com.ibm.ims.dli.DLIException;

/**
 * The <code>IOMessage</code> interface represents a message either
 * coming from or going to an IMS message queue. It provides a mapping between
 * the data contained in the message and access functions that operate on the
 * class. Client applications can reference individual fields of the message by
 * field name utilizing a wide range of data conversion functions.
 * 
 */
public interface IOMessage {

    /**
     * Constant to indicates the end of a transaction.
     */
    public static final String END_CONVERSATION_BLANKS = "        ";

    /*
     * The transaction code rules can be specified. Two rules need to be
     * identified for IMS to be able to distinguish the transaction code from
     * the message pay load. The transaction code is a fixed eight bytes long
     * with data following immediately without any blanks or it can be less than
     * or equal to 8 bytes with a trailing blank.
     */
    // public void setTrancodeRequiresTrailingBlank(boolean yn);
    // boolean getTrancodeRequiresTrailingBlank();
    /*
     * trancodeName = blanks inidicates the end of a transaction we want to do
     * this because you can also switch to a non-conversation transaction which
     * will also end the conversation
     * http://pic.dhe.ibm.com/infocenter/dzichelp/v2r2/topic/com.ibm.ims9.doc.atm/xf450f.htm#xf450f
     */

    /**
     * Retrieves the IMS transaction name related to this message.
     * 
     * @return trancodeName the name of the IMS transaction code
     * @throws DLIException
     * @throws TMException
     *             if an error occurs during processing
     */
    public String getTransactionName() throws DLIException;

    /**
     * Sets the IMS transaction code name. You can also use this method to end a
     * conversational transaction and for deferred program switching.
     * 
     * <p>The following code example shows how to end a conversational transaction: 
     * <blockquote>
     * <pre>
     *    IOMessage ioMsg = application.getIOMessage(&quot;msg_url&quot;);
     *    ioMsg.setTrancodeName(IOMessage.END_CONVERSATION);
     * </pre>
     * </blockquote> 
     * <p>The following code example shows how to use this method for deferred program switching: 
     * <blockquote>
     * <pre>
     *    spaMessage.setTrancodeName(&quot;newTran&quot;);
     *    messageQueue.insert(spaMessage, MessageQueue.DEFAULT_DESTINATION);
     * </pre>
     * </blockquote>
     * 
     * @param transactionName
     *            the name of the IMS transaction code
     * @throws TMException
     *             if an error occurs during processing
     */
    public void setTransactionName(String transactionName) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>String</code>. 
	 * 
	 * @param fieldName
	 *            the name of the input or output message field
	 * @return the value of the field as a String
	 * @throws DLIException if the field name is not found in the segment
     */
    public String getString(String fieldName) throws DLIException;

    /**
	 * Sets the field indicated by the fieldName parameter to the specified
	 * <code>String</code> value. 
	 * 
	 * @param fieldName
	 *            the name of the input or output message field
	 * @param value
	 *            the new value for the field
	 * @throws DLIException if the field name is not found in the segment or if there
	 *               is a conversion failure
     */
    public void setString(String fieldName, String value) throws DLIException;

    /**
	 * Returns the field specified by the parameter as a <code>boolean</code>. 
	 * 
	 * @param fieldName
	 *            the name of the input or output message field
	 * @return the value of the field as a boolean
	 * @throws DLIException if the field name is not found in the segment
     */
    public boolean getBoolean(String fieldName) throws DLIException;

    /**
	 * Sets the field indicated by the fieldName parameter to the specified
	 * <code>boolean</code> value.
	 * 
	 * @param fieldName
	 *            the name of the input or output message field
	 * @param value
	 *            the new value for the field
	 * @throws DLIException if the field name is not found in the segment or if there
	 *               is a conversion failure
     */
    public void setBoolean(String fieldName, boolean value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>BigDecimal</code>.
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a <code>BigDecimal</code>
     * @throws DLIException if the field name is not found in the segment
     */
    public BigDecimal getBigDecimal(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified
     * <code>BigDecimal</code> value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setBigDecimal(String fieldName, BigDecimal value) throws DLIException;

    /**
     * Returns the field specified by the parameter as an integer. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as an <code>int</code>
     * @throws DLIException if the field name is not found in the segment
     */
    public int getInt(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified
     * integer value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setInt(String fieldName, int value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>short</code>. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a short
     * @throws DLIException if the field name is not found in the segment
     */
    public short getShort(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified
     * <code>short</code> value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setShort(String fieldName, short value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>long</code>. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a <code>long</code>
     * @throws DLIException if the field name is not found in the segment
     */
    public long getLong(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified long
     * value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setLong(String fieldName, long value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>float</code>. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a float
     * @throws DLIException if the field name is not found in the segment
     */
    public float getFloat(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified
     * <code>float</code> value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setFloat(String fieldName, float value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>double</code>. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a double
     * @throws DLIException if the field name is not found in the segment
     */
    public double getDouble(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified
     * <code>double</code> value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setDouble(String fieldName, double value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>byte</code>. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a byte
     * @throws DLIException if the field name is not found in the segment
     */
    public byte getByte(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified <code>byte</code> 
     * value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setByte(String fieldName, byte value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>byte</code> array. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a <code>byte</code> array
     * @throws DLIException if the field name is not found in the segment
     */
    public byte[] getBytes(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified
     * <code>byte</code> array value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setBytes(String fieldName, byte[] value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>java.util.Date</code>.
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a <code>java.util.Date</code>
     * @throws DLIException if the field name is not found in the segment
     */
    public java.util.Date getDate(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified
     * <code>java.util.Date</code> value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setDate(String fieldName, java.util.Date value) throws DLIException;

    /**
     * Returns the field specified by the parameter as a <code>java.sql.Date</code>.
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @return the value of the field as a <code>java.sql.Date</code>
     * @throws DLIException if the field name is not found in the segment
     */
    public java.sql.Date getSqlDate(String fieldName) throws DLIException;

    /**
     * Sets the field indicated by the fieldName parameter to the specified
     * <code>java.sql.Date</code> value. 
     * 
     * @param fieldName
     *            the name of the input or output message field
     * @param value
     *            the new value for the field
     * @throws DLIException if the field name is not found in the segment or if there
     *               is a conversion failure
     */
    public void setSqlDate(String fieldName, java.sql.Date value) throws DLIException;
    
    /**
     * Returns the actual length (LL) of the IOMessage instance which
     * may differ from the length in the metadata.
     * 
     * @return the actual length of the IOMessage
     */
    public int getActualLength() throws DLIException;
    /**
     * Resets all character data fields to blanks, and sets all non-character fields 
     * to 0x40 (if EBCDIC) or 0x20 (if ASCII). 
     * <p>Calling this method can cause potential problems for <code>ZONEDDECIMAL</code> and
     * <code>PACKEDDECIMAL</code> fields. Any getter method call to these fields will throw a
     * <code>DLIException</code> since the method will look for a sign trailing byte and encounter
     * the blank value which is invalid. To avoid this problem, ensure that following 
     * a {@link IOMessage#clear()} method call, a setter method is issued to any 
     * <code>PACKEDDECIMAL</code> or <code>ZONEDDECIMAL</code> field before a getter method is issued.
     * 
     * @throws DLIException
     */
    public void clear() throws DLIException;
}
