package com.ibm.ims.dli.tm;

import com.ibm.ims.dli.AIB;
import com.ibm.ims.dli.DLIException;

/**
 * This interface provides support for transactions that are managed internal to IMS, and do not require an 
 * external transaction manager.  Use this interface to issue commit (SYNC) and rollback (ROLB) DL/I calls, as  
 * well as symbolic checkpoint (CHKP) and restart (XRST) calls. 
 * <p>For more details and a code example for rollback in a JMP application, see the topic 
 * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_jmpapprollback.htm" target="_blank">Rolling back IMS changes in a JMP application</a>" 
 * in the IMS Application Programming guide documentation.
 * <p>For more details and a code sample for JBP symbolic checkpoint and restart, see the topic 
 * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_developingjbpapps.htm" target="_blank">Developing JBP applications with the IMS Java dependent region resource adapter</a>" 
 * in the IMS Application Programming guide documentation.
 */
public interface Transaction {

    /**
     * Commits an IMS transaction
     * 
     * @throws TMException
     *             if the operation fails
     */
    public void commit() throws DLIException;

    /**
     * Rolls back an IMS transaction
     * 
     * @throws TMException
     *             if the operation fails
     */
    public void rollback() throws DLIException;

    /**
     * Issues a checkpoint call for a Java batch message processing (JBP) application that uses 
     * symbolic checkpoints. For JBP applications that use symbolic checkpoint or restart, call 
     * this method at intervals to write checkpoints to the IMS log. 
     * <p>This version of the checkpoint method does not take any parameters.  Use this method 
     * when your application needs to take checkpoints without having to specify a 
     * checkpoint ID and without having to save any application state information. 
     * <p>If this method is used to take checkpoints, when restarting, use the 4-character constant "LAST" as 
     * the checkpoint ID for restart.<br> 
     * <p>Before taking any checkpoints, you must issue an initial {@link Transaction#restart()} call 
     * to notify IMS that your application intends to use symbolic checkpoints.</p>
     * <p>If your application needs to save application objects, use the other form of the checkpoint 
     * method: @link {Transaction#checkpoint(SaveArea)}.
     * @exception IMSException
     *                   if the checkpoint ID is invalid, or if an IMS call
     *                   returns a non-blank status code.
     */
    public void checkpoint() throws DLIException;
    
    /**
     * Issues a checkpoint call for a Java batch message processing (JBP) application that uses 
     * symbolic checkpoints. For JBP applications that use symbolic checkpoint or restart, call 
     * this method at intervals to write checkpoints to the IMS log. 
     * <p>Use this method when the application needs to take checkpoints without having to specify a checkpoint ID
     * but has to store application state information. 
     * <p>If this method is used to take checkpoints, when restarting, use the 4-character constant "LAST" as 
     * the checkpoint ID for restart.<br> 
     * <p>Before taking any checkpoints, you must issue an initial {@link Transaction#restart()} call 
     * to notify IMS that your application intends to use symbolic checkpoints.</p>
     * @param saveArea the <code>SaveArea</code> instance to store application state information.
     * @exception IMSException
     *                   if the checkpoint ID is invalid, or if an IMS call
     *                   returns a non-blank status code.
     */
    public void checkpoint(SaveArea saveArea) throws DLIException;
    
    /**
     * Issues a restart call for a Java batch message processing (JBP) application that 
     * uses symbolic checkpoints.</p>
     * <p>Use this method in one of the following cases:
     * <ul>
     * <li>At normal program start, this method tells IMS that subsequent checkpoint
     *      calls are symbolic checkpoint calls.
     * <li>The restart checkpoint ID (if any) is supplied through the
     *      CKPTID= parameter in the JBP region JCL procedure.</p>
     * </ul>
     * @return the <code>SaveArea</code> instance with the application state information to be restored.
     * @exception IMSException
     *                   if a non-blank status code is returned from the XRST call.
     */
    public SaveArea restart() throws DLIException;

    /**
     * Verifies correct repositioning of a PCB after a restart call.
     * @return boolean <code>true</code> if the PCB was correctly repositioned, <code>false</code> otherwise.  
     *    If <code>false</code>, the PCB may not be correctly repositioned, and the application must take whatever 
     *    action is required to ensure valid results before resuming processing.
     */
//    public boolean verifyPCBPosition(String pcbName) throws DLIException;
    
	/**
	 * Use this method to confirm correct repositioning of a PCB after a restart (XRST) call.
	 * Following a restart, IMS attempts to reposition its databases to the positions existing 
	 * at the time of the chosen checkpoint, using information saved in the checkpoint on the IMS log. 
	 * @param pcbArray
	 *            a <code>String</code> array of PCB reference or alias names as
	 *            defined in the Java metadata class that provides the database view.
	 * @return boolean <code>true</code> if the PCB was correctly repositioned, <code>false</code> otherwise.  
     *    If <code>false</code>, the PCB may not be correctly repositioned, and the application must take 
     *    whatever action is required to ensure valid results before resuming processing.
	 */
//	public boolean verifyPCBPosition(String[] pcbArray) throws DLIException;
	
	/**
	 * Returns the <code>AIB</code> instance associated with the most recent DL/I call. 
	 * The <code>AIB</code> contains all the data attributes of the IMS application interface block. 
	 * @return the <code>AIB</code> instance associated with the most recent DL/I call.
     */
    public AIB getAIB();

    /**
     * Stores the name of a PCB that was not reset after a symbolic restart call.
     * @param pcbName the name of the PCB
     */
//    public void setInvalidPCBs(String pcbName);
    
    /**
     * Returns the list of PCBs that were not reset correctly after a symbolic restart call was issued.
     * @return the array of PCB names
     */
//    public String[] getInvalidPCBs();

    /**
     * Returns the <code>SaveArea</code> object that is used during CHKP or XRST calls
     * to checkpoint the application data objects.
     * 
     * @return the <code>SaveArea</code> instance that stores the application state. 
     */
    public SaveArea getSaveArea();

    /**
     * Abnormally terminates the application by making a "ROLL" call. 
     */
    public void abend();
}
