package com.ibm.ims.jdbc;

/* (c) Copyright International Business Machines Corporation 2008. All rights reserved. */

import java.io.IOException;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.FileHandler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;

import com.ibm.ims.dli.IMSConnectionSpec;
import com.ibm.ims.dli.PSBImpl;
import com.ibm.ims.dli.logging.PrintWriterHandler;
import com.ibm.ims.opendb.BuildNumber;

/**
 * A factory for connections to IMS databases. This class implements the
 * <code>javax.sql.DataSource</code> interface for the IMS Universal JDBC
 * driver.
 */
public class IMSDataSource implements javax.sql.DataSource {

    private static final Logger logger = Logger.getLogger("com.ibm.ims.db.opendb.jdbc");

    /**
     * Constant indicating usage of the type 4 driver
     */
    public static final int DRIVER_TYPE_4 = IMSConnectionSpec.DRIVER_TYPE_4;

    /**
     * Constant indicating usage of the type 2 driver
     */
    public static final int DRIVER_TYPE_2 = IMSConnectionSpec.DRIVER_TYPE_2;
    
    /**
     * Constant indicating to turn on all Universal driver tracing
     */
    public static final short TRACE_ALL = -1;
    
    /**
     * Constant indicating to turn on DRDA layer tracing
     */
    public static final short TRACE_DRDA = 1;
    
    /**
     * Constant indicating to turn on DLI layer tracing
     */
    public static final short TRACE_DLI = 28;
    
    /**
     * Constant indicating to turn on JDBC layer tracing
     */
    public static final short TRACE_JDBC = 32;
    
    /**
     * Constant indicating to turn on JEE layer tracing
     */
    public static final short TRACE_JEE = 192;

    private String user;

    private String password;

    private String datastoreServer;

    private int portNumber;

    private String metadataURL;

    // private String dataSourceName;
    private String description;

    private String datastoreName;

    private int driverType;
    
    private boolean signedCompare = true;
    private int t2OutputBufferSize = 1280000;
    
	private boolean llField = false;
	private boolean expandArrayResultSet = false;
	private boolean ssaOptimization = false;
	private boolean returnResultOnStatusCodeGE = false;
	private boolean treatIncompleteFieldAsNull = true;
	
    private boolean sslConnection;
    private String sslTrustStoreLocation;
    private String sslTrustStorePassword;
    private String sslTrustMgrAlgorithm;
    private String sslKeyStoreLocation;
    private String sslKeyStorePassword;
    private String sslKeyMgrAlgorithm;
    private String keyStoreType;
	private String secureSocketProtocol;
    private String initStatusGroup;

    private int loginTimeout = 0;
    
    private PrintWriter pw;

    private PrintWriterHandler pwh;
    
    private Properties properties = null;

    /***************************************************************************
     * The default constructor.
     * 
     */
    public IMSDataSource() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "IMSDataSource()", parms);

            logger.exiting(getClass().getName(), "IMSDataSource()");
        }
    }

    /***************************************************************************
     * Attempts to establish a connection with the data source that this
     * DataSource object represents.
     * 
     * @return a connection to the data source
     */
    public Connection getConnection() throws SQLException {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getConnection()", parms);
        }

        Connection connection = null;
        
        if (properties != null) {
            String traceFile = properties.getProperty("traceFile");
            if (traceFile != null) {
                // Tracing is enabled in the URL...look at other properties
                String fullTraceFile = null;
                boolean append = false;
                
                String traceDirectory = properties.getProperty("traceDirectory");
                if (traceDirectory != null) {
                    String fileSeparator = System.getProperty("file.separator");
                    if (!traceDirectory.endsWith(fileSeparator)) {
                        traceDirectory = traceDirectory + fileSeparator;
                    }
                    fullTraceFile = traceDirectory + traceFile;
                } else {
                    fullTraceFile = traceFile;
                }
                
                String traceFileAppend = properties.getProperty("traceFileAppend");
                if (traceFileAppend != null && traceFileAppend.equalsIgnoreCase("true")) {
                    append = true;
                }
                
                String traceLevel = properties.getProperty("traceLevel");
                short traceLvl = 0;
                try {
                    traceLvl = Short.parseShort(traceLevel);
                } catch (NumberFormatException e) {
                    Object[] inserts = {traceLevel};
                    SQLException sqle = new SQLException(JDBCErrorMessages.getIMSBundle().getString(
                            "INVALID_TRACE_LEVEL", inserts));
                    sqle.initCause(e);
                    throw sqle;
                }
                
                if (traceLvl != 0) {
                    try {
                        FileHandler fh = new FileHandler(fullTraceFile, append);
                        fh.setFormatter(new SimpleFormatter());
                        fh.setLevel(Level.FINEST);
                        AtomicBoolean loggedBuildNumberAtomic = new AtomicBoolean(false);

                        if (traceLvl == -1) {
                            // Enable all loggers
                            Logger allLoggers = Logger.getLogger("com.ibm.ims.db.opendb");
                            allLoggers.setLevel(Level.FINEST);
                            allLoggers.addHandler(fh);
                            
                            // Relog Build Number to new trace file
                            Object[] inserts = {BuildNumber.BUILD_NUMBER};
                        	logger.info(JDBCErrorMessages.getIMSBundle().getString("LOG_BUILD_NUMBER", inserts));
                        } else {
                            // Check various loggers to see which are enabled
                            if ((traceLvl & 0x01) == 1) {
                                // DRDA loggers
                                Logger drdaLogger = Logger.getLogger("com.ibm.ims.db.opendb.drda");
                                drdaLogger.setLevel(Level.FINEST);
                                drdaLogger.addHandler(fh);
                                Logger drdaTIDLogger = Logger.getLogger("com.ibm.ims.db.opendb.tid");
                                drdaTIDLogger.setLevel(Level.FINEST);
                                drdaTIDLogger.addHandler(fh);
                                
                                if (loggedBuildNumberAtomic.compareAndSet(false, true)) {
                                    Object[] inserts = {BuildNumber.BUILD_NUMBER};
                                    drdaLogger.info(JDBCErrorMessages.getIMSBundle().getString("LOG_BUILD_NUMBER", inserts));
                                }
                            }
                            
                            if ((traceLvl & 0x04) == 4) {
                                // DLI logger
                                Logger dliLogger = Logger.getLogger("com.ibm.ims.db.opendb.dli");
                                dliLogger.setLevel(Level.FINEST);
                                dliLogger.addHandler(fh);
                                
                                if (loggedBuildNumberAtomic.compareAndSet(false, true)) {
                                    Object[] inserts = {BuildNumber.BUILD_NUMBER};
                                    dliLogger.info(JDBCErrorMessages.getIMSBundle().getString("LOG_BUILD_NUMBER", inserts));
                                }
                            }
     
                            if ((traceLvl & 0x08) == 8) {
                                // DLI type 2 logger
                                Logger dliT2Logger = Logger.getLogger("com.ibm.ims.db.opendb.dli.t2");
                                dliT2Logger.setLevel(Level.FINEST);
                                dliT2Logger.addHandler(fh);
                                
                                if (loggedBuildNumberAtomic.compareAndSet(false, true)) {
                                    Object[] inserts = {BuildNumber.BUILD_NUMBER};
                                    dliT2Logger.info(JDBCErrorMessages.getIMSBundle().getString("LOG_BUILD_NUMBER", inserts));
                                }
                            }

                            if ((traceLvl & 0x10) == 16) {
                                // DLI TM logger
                                Logger dliTMLogger = Logger.getLogger("com.ibm.ims.db.opendb.dli.tm");
                                dliTMLogger.setLevel(Level.FINEST);
                                dliTMLogger.addHandler(fh);
                                
                                if (loggedBuildNumberAtomic.compareAndSet(false, true)) {
                                    Object[] inserts = {BuildNumber.BUILD_NUMBER};
                                    dliTMLogger.info(JDBCErrorMessages.getIMSBundle().getString("LOG_BUILD_NUMBER", inserts));
                                }
                            }

                            if ((traceLvl & 0x20) == 32) {
                                // JDBC logger
                                Logger jdbcLogger = Logger.getLogger("com.ibm.ims.db.opendb.jdbc");
                                jdbcLogger.setLevel(Level.FINEST);
                                jdbcLogger.addHandler(fh);
                                
                                if (loggedBuildNumberAtomic.compareAndSet(false, true)) {
                                    Object[] inserts = {BuildNumber.BUILD_NUMBER};
                                    jdbcLogger.info(JDBCErrorMessages.getIMSBundle().getString("LOG_BUILD_NUMBER", inserts));
                                }
                            }
                           
                            if ((traceLvl & 0x40) == 64) {
                                // CCI logger
                                Logger cciLogger = Logger.getLogger("com.ibm.ims.db.opendb.cci");
                                cciLogger.setLevel(Level.FINEST);
                                cciLogger.addHandler(fh);
                                
                                if (loggedBuildNumberAtomic.compareAndSet(false, true)) {
                                    Object[] inserts = {BuildNumber.BUILD_NUMBER};
                                    cciLogger.info(JDBCErrorMessages.getIMSBundle().getString("LOG_BUILD_NUMBER", inserts));
                                }
                            }
            
                            if ((traceLvl & 0x80) == 128) {
                                // SPI logger
                                Logger spiLogger = Logger.getLogger("com.ibm.ims.db.opendb.spi");
                                spiLogger.setLevel(Level.FINEST);
                                spiLogger.addHandler(fh);
                                
                                if (loggedBuildNumberAtomic.compareAndSet(false, true)) {
                                    Object[] inserts = {BuildNumber.BUILD_NUMBER};
                                    spiLogger.info(JDBCErrorMessages.getIMSBundle().getString("LOG_BUILD_NUMBER", inserts));
                                }
                            }
                        } 
                    } catch (IOException e) {
                        SQLException sqlE = new SQLException(e.getMessage());
                        sqlE.initCause(e);
                        throw sqlE;
                    }
                } else {
                	Logger allLoggers = Logger.getLogger("com.ibm.ims.db.opendb");
                    allLoggers.setLevel(Level.OFF);
                }
            }
            
            // Check for dataStore property PRIOR to creating the connection
            String datastoreName = properties.getProperty("datastoreName");
            if (datastoreName != null) {
            	this.setDatastoreName(datastoreName);
            }
            // Variable Length Support - Will expose the LL fields in ResultSet and Database
            // Metadata when set to true.
            String llField = properties.getProperty("llField");
            if (llField != null && llField.equalsIgnoreCase("true")) {
                this.setLLField(true);
            }
            String initStatusGroup = properties.getProperty("initStatusGroup");
            if (initStatusGroup != null) {
                this.setInitStatusGroup(initStatusGroup.trim());
            }

			String expandArrayResultSet = properties.getProperty("expandArrayResultSet");
			if (expandArrayResultSet != null) {
				boolean booleanize = Boolean.parseBoolean(expandArrayResultSet);
				if (booleanize) {
					this.setExpandArrayResultSet(booleanize);
				}
			}
			
			String ssaOptimization = properties.getProperty("ssaOptimization");
			if (ssaOptimization != null) {
				boolean booleanize = Boolean.parseBoolean(ssaOptimization);
				if (booleanize) {
					this.setSsaOptimization(booleanize);
				}
			}
			
			String returnResultOnStatusCodeGE = properties.getProperty("returnResultOnStatusCodeGE");
			if (returnResultOnStatusCodeGE != null) {
				boolean booleanize = Boolean.parseBoolean(returnResultOnStatusCodeGE);
				if(booleanize) {
					this.setReturnResultOnStatusCodeGE(booleanize);
				}
			}
			
			String treatIncompleteFieldAsNull = properties.getProperty("treatIncompleteFieldAsNull");
			if (treatIncompleteFieldAsNull != null) {
				boolean booleanize = Boolean.parseBoolean(treatIncompleteFieldAsNull);
				if(booleanize) {
					this.setTreatIncompleteFieldAsNull(booleanize);
				}
			}
		}

        // NOT SUPPORTED
        // We are not a JDBC provider. Pooling will be handled by the SPI layer.
        // TODO Check to see if dataSourceName set and then go that route first
        // if (dataSourceName != null && dataSourceName.length() != 0) {
        // try {
        // InitialContext context = new InitialContext();
        // XADataSourceImpl xaDS =
        // (XADataSourceImpl)context.lookup(dataSourceName);
        // xaDS.setDataSourceProperties(this);
        // XAConnection xaConn = xaDS.getXAConnection();
        // connection = xaConn.getConnection();
        // } catch (NamingException e) {
        // throw new SQLException(e.toString());
        // }
        //            
        // } else {
        connection = new ConnectionImpl(this);
        // }
        
        // If Properties have been set, update the appropriate Connection Properties
        if (properties != null) {
            // Cognos...but we can document as part of URL if we want to
            String fetchSize = properties.getProperty("fetchSize");
            if (fetchSize != null) {
                try {
                    int value = Integer.parseInt(fetchSize);
                    ((ConnectionImpl)connection).setFetchSize(value);
                } catch (NumberFormatException e) {
                    Object[] inserts = {fetchSize};
                    SQLException sqle = new SQLException(JDBCErrorMessages.getIMSBundle().getString(
                            "INVALID_FETCH_SIZE", inserts));
                    sqle.initCause(e);
                    throw sqle;
                }
            }

            // If false do not do special ssa processing in where clause
            String signedCompare = properties.getProperty("signedCompare");
            if(signedCompare != null) {
                ((ConnectionImpl)connection).setSignedCompare(Boolean.valueOf(signedCompare));
            }

            // If null, then it will remain 0
            String t2outputbufsz = properties.getProperty("t2OutputBufferSize");
            if (t2outputbufsz != null) {
                try {
                    int value = Integer.parseInt(t2outputbufsz);
                    ((ConnectionImpl)connection).setT2OutputBufferSize(value);
                } catch (NumberFormatException e) {
                    Object[] inserts = {t2outputbufsz};
                    SQLException sqle = new SQLException(JDBCErrorMessages.getIMSBundle().getString(
                            "INVALID_T2OUTPUTBUFFERSIZE", inserts));
                    sqle.initCause(e);
                    throw sqle;
                }
            }
            
            // Cognos specific - they pool Connections with allocated PSBs - we need to deallocate
            // when the last Statement is closed on a Connection because will go back into the pool
            // at this point
            String dpsbOnCommit = properties.getProperty("dpsbOnCommit");
            if ((dpsbOnCommit != null && dpsbOnCommit.equalsIgnoreCase("true")) || PSBImpl.checkForCognosRuntime()) {
                ((ConnectionImpl)connection).setDPSBOnCommit(true);
                System.setProperty("attemptReconnectOnDisconnect", "true");
            } else if (PSBImpl.isManagedEnvironment) {
                System.setProperty("attemptReconnectOnDisconnect", "true");
            } else {
            	System.setProperty("attemptReconnectOnDisconnect", "false");
            }
            
            // Variable Length Support - Will expose the LL fields in ResultSet and Database
            // Metadata when set to true.
            if (this.getLLField()) {
                ((ConnectionImpl)connection).setLLField(true);
            }
            if(this.getExpandArrayResultSet()) {
            	((ConnectionImpl)connection).setExpandArrayResultSet(true);
            }
            
            if(this.getSsaOptimization()) {
            	((ConnectionImpl)connection).setSsaOptimization(true);
            }
            
            if(this.getReturnResultOnStatusCodeGE()) {
            	((ConnectionImpl)connection).setReturnResultOnStatusCodeGE(returnResultOnStatusCodeGE);
            }
            
            if(this.getTreatIncompleteFieldAsNull()) {
            	((ConnectionImpl)connection).setTreatIncompleteFieldAsNull(treatIncompleteFieldAsNull);
            }
            
            String maxRows = properties.getProperty("maxRows");
            if (maxRows != null) {
                try {
                    int value = Integer.parseInt(maxRows);
                    if (value < 0) {
                        Object[] inserts = {maxRows};
                        SQLException sqle = new SQLException(JDBCErrorMessages.getIMSBundle().getString(
                                "INVALID_MAX_ROWS", inserts));
                        throw sqle;
                    }
                    ((ConnectionImpl)connection).setMaxRows(value);
                } catch (NumberFormatException e) {
                    Object[] inserts = {maxRows};
                    SQLException sqle = new SQLException(JDBCErrorMessages.getIMSBundle().getString(
                            "INVALID_MAX_ROWS", inserts));
                    sqle.initCause(e);
                    throw sqle;
                }
            }
            
            String currentSchema = properties.getProperty("currentSchema");
            if (currentSchema != null) {
                ((ConnectionImpl)connection).setCurrentSchema(currentSchema);
            }
            
            String initStatusGroup = properties.getProperty("initStatusGroup");
            if (initStatusGroup != null) {
                ((ConnectionImpl)connection).setInitStatusGroup(initStatusGroup.trim());
            }
            
            String preloadUTC = properties.getProperty("preloadUserTypeConverters");
            if (preloadUTC != null) {
            	Boolean value = Boolean.parseBoolean(preloadUTC);
                ((ConnectionImpl)connection).setPreloadUserTypeConverters(value);
            }
        }
        
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getConnection()");
        }

        return connection;
    }

    /***************************************************************************
     * Attempts to establish a connection with the data source that this
     * DataSource object represents.
     * 
     * @param username
     *            the database user on whose behalf the connection is being made
     * @param password
     *            the user's password
     * @return a connection to the data source
     */
    public Connection getConnection(String username, String password) throws SQLException {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "username: " + username;
            logger.entering(getClass().getName(), "getConnection(String, String)", parms);
        }

        // User name and password override those in the DataSource properties
        this.user = username;
        this.password = password;

        Connection connection = null;

        connection = this.getConnection();

        // if (dataSourceName != null && dataSourceName.length() != 0) {
        // try {
        // InitialContext context = new InitialContext();
        // XADataSourceImpl xaDS =
        // (XADataSourceImpl)context.lookup(dataSourceName);
        // xaDS.setDataSourceProperties(this);
        // XAConnection xaConn = xaDS.getXAConnection();
        // connection = xaConn.getConnection();
        // } catch (NamingException e) {
        // throw new SQLException(e.toString());
        // }
        //            
        // } else {
        // connection = new ConnectionImpl(this);
        // }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getConnection(String, String)");
        }

        return connection;
    }

    // TODO Added for CCI's SQL execution
    // public Connection getConnection(PSBImpl psb) throws SQLException {
    // ConnectionImpl conn = new ConnectionImpl(psb);
    //	
    // return conn;
    // }

    /**
     * Retrieves the log writer for this DataSource object.
     * 
     * The log writer is a character output stream to which all logging and
     * tracing messages for this data source will be printed. This includes
     * messages printed by the methods of this object, messages printed by
     * methods of other objects manufactured by this object, and so on. Messages
     * printed to a data source specific log writer are not printed to the log
     * writer associated with the java.sql.Drivermanager class. When a
     * DataSource object is created, the log writer is initially null; in other
     * words, the default is for logging to be disabled.
     * 
     * @return the log writer for this data source or null if logging is
     *         disabled
     */
    public PrintWriter getLogWriter() throws SQLException {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getLogWriter()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getLogWriter()");
        }

        return this.pw;
    }

    /**
     * Retrieves the login timeout for this DataSource object
     * 
     * @return the login timeout
     */
    public int getLoginTimeout() throws SQLException {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getLoginTimeout()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getLoginTimeout()", loginTimeout);
        }

        return loginTimeout;
    }

    /***************************************************************************
     * Sets the log writer for this <code>DataSource</code> object to the
     * given <code>java.io.PrintWriter</code> object. The log writer is a
     * character output stream to which all logging and tracing messages for
     * this data source will be printed. This includes messages printed by the
     * methods of this object, messages printed by methods of other objects
     * manufactured by this object, and so on. Messages printed to a data
     * source- specific log writer are not printed to the log writer associated
     * with the <code>java.sql.Drivermanager</code> class. When a
     * <code>DataSource</code> object is created the log writer is initially
     * null; in other words, the default is for logging to be disabled.
     * 
     * @param out
     *            the new log writer; to disable logging, set to null
     */
    public void setLogWriter(PrintWriter out) throws SQLException {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "setLogWriter(PrintWriter)");
        }

        if (out == null) {
            if (this.pwh != null) {
                Logger logger = Logger.getLogger("com.ibm.ims.db.opendb");
                logger.setLevel(Level.OFF);
            }
        } else {
            if (this.pw == null) {
                this.pw = out;
                this.pwh = new PrintWriterHandler(out, new SimpleFormatter());

                Logger logger = Logger.getLogger("com.ibm.ims.db.opendb");
                logger.setLevel(Level.FINEST);
                logger.addHandler(pwh);
            }
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setLogWriter(PrintWriter)");
        }
    }

    /***************************************************************************
     * Sets the maximum time in seconds that this data source will wait while
     * attempting to connect to a database. A value of zero specifies that the
     * timeout is the default system timeout if there is one; otherwise, it
     * specifies that there is no timeout. When a <code>DataSource</code>
     * object is created, the login timeout is initially zero.
     * 
     * @param seconds
     *            the data source login time limit
     */
    public void setLoginTimeout(int seconds) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "seconds: " + seconds;
            logger.entering(getClass().getName(), "setLoginTimeout(int)", parms);
        }

        this.loginTimeout = seconds;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setLoginTimeout(int)");
        }

    }

    /**
     * Sets the user name for RACF authentication.
     * 
     * @param user
     *            User name.
     */
    public void setUser(String user) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "user: " + user;
            logger.entering(getClass().getName(), "setUser(String)", parms);
        }

        this.user = user;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setUser(String)");
        }
    }
    
	/**
	 * Sets the option to expand or not expand the ResultSet of a column with data
	 * type ARRAY.
	 * <br><br>
	 * When set to <b>true</b> the array result set expands the sub-elements of the
	 * array into 1 or more columns depending on the number of fields defined within
	 * the array.
	 * <br><br>
	 * When set to <b>false</b> (default) the array result set returns a result set
	 * of exactly 2 columns. Column 1 is the index and column 2 is a Struct object
	 * that contains accessors for fields within that array element.
	 * 
	 * @param expandArrayResultSet
	 */
    public void setExpandArrayResultSet(boolean expandArrayResultSet) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "expandArrayResultSet: " + expandArrayResultSet;
            logger.entering(getClass().getName(), "setExpandArrayResultSet(boolean)", parms);
        }

        this.expandArrayResultSet = expandArrayResultSet;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setExpandArrayResultSet(boolean)");
        }
    }
    
	/**
	 * Returns the option to expand or not expand the ResultSet of a column with data
	 * type ARRAY.
	 * <br><br>
	 * When set to <b>true</b> the array result set expands the sub-elements of the
	 * array into 1 or more columns depending on the number of fields defined within
	 * the array.
	 * <br><br>
	 * When set to <b>false</b> (default) the array result set returns a result set
	 * of exactly 2 columns. Column 1 is the index and column 2 is a Struct object
	 * that contains accessors for fields within that array element.
	 * 
	 * @param expandArrayResultSet
	 */
	public boolean getExpandArrayResultSet() {
		if (logger.isLoggable(Level.FINER)) {
			Object[] parms = new Object[2];
			parms[0] = "Hash code: " + Integer.toHexString(hashCode());
			parms[1] = "Thread ID: " + Thread.currentThread().getId();
			logger.entering(getClass().getName(), "getExpandArrayResultSet()", parms);
		}

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(getClass().getName(), "getExpandArrayResultSet()", user);
		}

		return this.expandArrayResultSet;
	}
	
	/**
	 * Sets the option to whether to optimize the SSA or not
	 * <br><br>
	 * When set to <b>true</b> the SSA will be optimized to a fully qualified key if 
	 * subfields of that key are provided.
	 * <br><br>
	 * When set to <b>false</b> (default) the SSA will not be optimized.
	 * 
	 * @param ssaOptimization
	 */
    public void setSsaOptimization(boolean ssaOptimization) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "ssaOptimization: " + ssaOptimization;
            logger.entering(getClass().getName(), "setSsaOptimization(boolean)", parms);
        }

        this.ssaOptimization = ssaOptimization;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSsaOptimization(boolean)");
        }
    }
    
	/**
	 * Returns the option to whether to optimize the SSA or not
	 * <br><br>
	 * When set to <b>true</b> the SSA will be optimized to a fully qualified key if 
	 * subfields of that key are provided.
	 * <br><br>
	 * When set to <b>false</b> (default) the SSA will not be optimized.
	 * 
	 * @param ssaOptimization
	 */
	public boolean getSsaOptimization() {
		if (logger.isLoggable(Level.FINER)) {
			Object[] parms = new Object[2];
			parms[0] = "Hash code: " + Integer.toHexString(hashCode());
			parms[1] = "Thread ID: " + Thread.currentThread().getId();
			logger.entering(getClass().getName(), "getSsaOptimization()", parms);
		}

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(getClass().getName(), "getSsaOptimization()", user);
		}

		return this.ssaOptimization;
	}
    
	/**
	 * Sets the option to return the trailing partial data retrieved from a batchRetrieve or SELECT statement that received a GE status code. The current call pattern of a batchRetrieve or SELECT statement limits the partial data returned to the last record that met the SSA qualifications. 
	 * When set to true, a retrieve call's Path or ResultSet will receive the partial data returned from a GE status code.
	 * When set to false (default), no data will be returned when a call receives a GE status code.
     * 
	 * @param returnResultOnStatusCodeGE
	 */
	public void setReturnResultOnStatusCodeGE(boolean returnResultOnStatusCodeGE) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "returnResultOnStatusCodeGE: " + returnResultOnStatusCodeGE;
            logger.entering(getClass().getName(), "setReturnResultOnStatusCodeGE(boolean)", parms);
        }

        this.returnResultOnStatusCodeGE = returnResultOnStatusCodeGE;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setReturnResultOnStatusCodeGE(boolean)");
        }
    }
    
	/**
	 * Gets the option to treat incomplete field data as null. Incomplete field data
	 * is defined as a field's full length extending past the length of the segment.
	 * When set to true, incomplete field data is treated as a null. When set to
	 * false, the fields incomplete data will be returned. True is the default
	 * option. This property only applies to Binary and String datatypes.
	 * 
	 * @return property value of treatIncompleteFieldAsNull
	 */
	public boolean getTreatIncompleteFieldAsNull() {
		if (logger.isLoggable(Level.FINER)) {
			Object[] parms = new Object[2];
			parms[0] = "Hash code: " + Integer.toHexString(hashCode());
			parms[1] = "Thread ID: " + Thread.currentThread().getId();
			logger.entering(getClass().getName(), "getTreatIncompleteFieldAsNull()", parms);
		}

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(getClass().getName(), "getTreatIncompleteFieldAsNull()", user);
		}

		return this.treatIncompleteFieldAsNull;
	}
	
	/**
	 * Sets the option to treat incomplete field data as null. Incomplete field data
	 * is defined as a field's full length extending past the length of the segment.
	 * When set to true, incomplete field data is treated as a null. When set to
	 * false, the fields incomplete data will be returned. True is the default
	 * option. This property only applies to Binary and String datatypes.
	 * 
	 * @param treatIncompleteFieldAsNull
	 */
	public void setTreatIncompleteFieldAsNull(boolean treatIncompleteFieldAsNull) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "treatIncompleteFieldAsNull: " + treatIncompleteFieldAsNull;
            logger.entering(getClass().getName(), "setTreatIncompleteFieldAsNull(boolean)", parms);
        }

        this.treatIncompleteFieldAsNull = treatIncompleteFieldAsNull;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setTreatIncompleteFieldAsNull(boolean)");
        }
    }
    
	/**
	 * Sets the option to return the trailing partial data retrieved from a batchRetrieve or SELECT statement that received a GE status code. The current call pattern of a batchRetrieve or SELECT statement limits the partial data returned to the last record that met the SSA qualifications. 
	 * When set to true, a retrieve call's Path or ResultSet will receive the partial data returned from a GE status code.
	 * When set to false (default), no data will be returned when a call receives a GE status code.
     * 
	 * @return
	 */
	public boolean getReturnResultOnStatusCodeGE() {
		if (logger.isLoggable(Level.FINER)) {
			Object[] parms = new Object[2];
			parms[0] = "Hash code: " + Integer.toHexString(hashCode());
			parms[1] = "Thread ID: " + Thread.currentThread().getId();
			logger.entering(getClass().getName(), "getReturnResultOnStatusCodeGE()", parms);
		}

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(getClass().getName(), "getReturnResultOnStatusCodeGE()", user);
		}

		return this.returnResultOnStatusCodeGE;
	}
    /**
     * Retrieves the user name for RACF authentication.
     * 
     * @return the user name for this connection.
     */
    public String getUser() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getUser()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getUser()", user);
        }

        return this.user;
    }

    /**
     * Sets the user password for RACF authentication.
     * 
     * @param password
     *            User password value
     */
    public void setPassword(String password) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "metadataURL: " + metadataURL;
            logger.entering(getClass().getName(), "setPassword(String)");
        }

        this.password = password;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setPassword(String)");
        }
    }

    /**
     * Retrieves the user password for RACF authentication.
     * 
     * @return user password value for this connection
     */
    public String getPassword() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getPassword()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getPassword()");
        }

        return this.password;
    }
    
    /**
     * Sets connection properties that are not part of the standard DataSource interface:
     * <ul>
     * <li><code>dbspOncommit</code>: Set this property to true to deallocate the PSB when a commit occurs.</li>
     * <li><code>fetchSize</code>: Gives the client a hint as to the number of rows that should be fetched from the
     *  database when more rows are needed. The number of rows specified affects only data returned using this 
     *  connection. If the value specified is zero, the hint is ignored.</li>
     * <li><code>llfield</code>: Setting this property to true exposes the LL field data as a normal column in the result set.</li>
     * <li><code>maxRows</code>: Specifies the maximum number of rows to return in a query result set.</li>
     * <li><code>traceFile</code>: Specifies the name of the trace file for the connection.</li>
     * <li><code>traceFileAppend</code>: If the specified trace file already exists, setting this property to true specifies
     *  that the trace data for the new connection should be appended to the existing trace file rather than overwriting it.</li>
     * <li><code>traceDirectory</code>: Specifies the file system directory where the trace file is located.</li>
     * <li><code>traceLevel</code>: Specifies the trace level for the connection.</li>
     * 
     * </ul>
     * <p>These properties are optional.</p>
     * 
     * @param properties
     *            Connection Properties
     */
    public void setProperties(Properties properties) throws SQLException{
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "properties: " + properties.toString();
            logger.entering(getClass().getName(), "setProperties(Properties properties)");
        }

        this.properties = properties;
        
        // Update the DataSource property fields
        // Connection property fields won't be set until getConnection() is called.
        if (this.properties != null) {
            String sslConnection = this.properties.getProperty("sslConnection");
			if (sslConnection != null) {
				if (sslConnection.equalsIgnoreCase("true")) {
					this.sslConnection = true;
				} else if (sslConnection.equalsIgnoreCase("false")){
					this.sslConnection = false;
				}
			}
    
            String sslTrustStoreLocation = this.properties.getProperty("sslTrustStoreLocation");
			if (sslTrustStoreLocation != null && !sslTrustStoreLocation.isEmpty()) {
				this.sslTrustStoreLocation = sslTrustStoreLocation;
			}
    
            String sslTrustStorePassword = this.properties.getProperty("sslTrustStorePassword");
			if (sslTrustStorePassword != null && !sslTrustStorePassword.isEmpty()) {
				this.sslTrustStorePassword = sslTrustStorePassword;
			}
    
            String sslKeyStoreLocation = this.properties.getProperty("sslKeyStoreLocation");
			if (sslKeyStoreLocation != null && !sslKeyStoreLocation.isEmpty()) {
				this.sslKeyStoreLocation = sslKeyStoreLocation;
			}
    
            String sslKeyStorePassword = this.properties.getProperty("sslKeyStorePassword");
			if (sslKeyStorePassword != null && !sslKeyStorePassword.isEmpty()) {
				this.sslKeyStorePassword = sslKeyStorePassword;
			}
    
			String keyStoreType = this.properties.getProperty("sslKeyStoreType");
			if (keyStoreType != null && !keyStoreType.isEmpty()) {
				this.keyStoreType = keyStoreType;
			}

			String secureSocketProtocol = this.properties.getProperty("sslSecureSocketProtocol");
			if (secureSocketProtocol != null && !secureSocketProtocol.isEmpty()) {
				this.secureSocketProtocol = secureSocketProtocol;
			}

            String loginTimeoutStr = properties.getProperty("loginTimeout");
            if (loginTimeoutStr != null) {
                try {
                    int loginTimeout = Integer.parseInt(loginTimeoutStr);
                    this.loginTimeout = loginTimeout;
                } catch (NumberFormatException e) {
                    Object[] inserts = {loginTimeoutStr};
                    SQLException sqle = new SQLException(JDBCErrorMessages.getIMSBundle().getString(
                            "INVALID_LOGIN_TIMEOUT", inserts));
                    sqle.initCause(e);
                    throw sqle;
                }
            }
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setProperties(Properties properties)");
        }
    }

    /**
     * Retrieves the Connection Properties.
     * 
     * @return Properties for this connection
     */
    public Properties getProperties() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getProperties()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getProperties()");
        }

        return this.properties;
    }

    /**
     * Sets the name or IP address of the datastore server (IMS Connect). You
     * can provide either the host name (for example, dev123.svl.ibm.com) or the
     * IP address (for example, 192.166.0.2).
     * <p>
     * NOTE: Only used for Type-4 connectivity. Does not apply for Type-2.
     * 
     * @param datastoreServer
     *            name or IP address of the datastore server (IMS Connect)
     */
    public void setDatastoreServer(String datastoreServer) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "datastoreServer: " + datastoreServer;
            logger.entering(getClass().getName(), "setDatastoreServer(String)", parms);
        }
        this.datastoreServer = datastoreServer;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setDatastoreServer(String)");
        }
    }

    /**
     * Retrieves the name or IP address of your datastore server (IMS Connect).
     * 
     * @return the name or IP address of the datastore server (IMS Connect)
     */
    public String getDatastoreServer() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getDatastoreServer()", parms);
        }
        String value = this.datastoreServer;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getDatastoreServer()", value);
        }
        return value;
    }

    /**
     * Sets the port number to be used to communicate with IMS Connect. The port
     * number is defined using the DRDAPORT parameter on the ODACCESS statement
     * in the integrated IMS Connect configuration PROCLIB member.
     * <p>
     * NOTE: Only used for Type-4 connectivity. Does not apply for Type-2.
     * 
     * @param portNumber
     *            port number
     */
    public void setPortNumber(int portNumber) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "portNumber: " + portNumber;
            logger.entering(getClass().getName(), "setPortNumber(int)", parms);
        }

        this.portNumber = portNumber;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setPortNumber(int)");
        }
    }

    /**
     * Retrieves the port number.
     * 
     * @return the port number for this connection.
     */
    public int getPortNumber() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getPortNumber()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getPortNumber()", portNumber);
        }

        return this.portNumber;
    }

    /**
     * Sets SSL encryption on or off.
     * 
     * @param sslConnection
     *            Set to <code>true</code> to turn SSL encryption on.
     */
    public void setSSLConnection(boolean sslConnection) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "sslConnection: " + sslConnection;
            logger.entering(getClass().getName(), "setSSLConnection(boolean)", parms);
        }
        this.sslConnection = sslConnection;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSSLConnection(boolean)");
        }
    }

    /**
     * Sets SSL trust store path.
     * 
     * @param trustStoreLocation
     *            Fully qualified path to the key store file containing the trusted certificate from IMS Connect.
     */
    public void setSSLTrustStoreLocation(String trustStoreLocation) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "trustStoreLocation: " + trustStoreLocation;
            logger.entering(getClass().getName(), "setSSLTrustStoreLocation(String trustStoreLocation)", parms);
        }
        this.sslTrustStoreLocation = trustStoreLocation;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSSLTrustStoreLocation(String trustStoreLocation)");
        }
    }

    /**
     * Sets SSL trust store password.
     * 
     * @param trustStorePassword
     *            Password set for the key store file using keytool.
     */
    public void setSSLTrustStorePassword(String trustStorePassword) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "trustStorePassword set: " + ((trustStorePassword != null) ? "Yes" : "No");
            logger.entering(getClass().getName(), "setSSLTrustStorePassword(String trustStorePassword)", parms);
        }
        this.sslTrustStorePassword = trustStorePassword;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSSLTrustStorePassword(String trustStorePassword)");
        }
    }

    /**
     * Sets SSL trust manager algorithm.
     * 
     * @param trustMgrAlgorithm
     */
    public void setSSLTrustMgrAlgorithm(String trustMgrAlgorithm) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "trustMgrAlgorithm set: " + trustMgrAlgorithm;
            logger.entering(getClass().getName(), "setSSLTrustMgrAlgorithm(String trustMgrAlgorithm)", parms);
        }
        this.sslTrustMgrAlgorithm = trustMgrAlgorithm;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSSLTrustMgrAlgorithm(String trustMgrAlgorithm)");
        }
    }

    /**
     * Sets SSL key store path.
     * 
     * @param keyStoreLocation
     *            Fully qualified path to the key store file containing the client's certificate and private key.
     */
    public void setSSLKeyStoreLocation(String keyStoreLocation) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "keyStoreLocation: " + keyStoreLocation;
            logger.entering(getClass().getName(), "setSSLKeyStoreLocation(String keyStoreLocation)", parms);
        }
        this.sslKeyStoreLocation = keyStoreLocation;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSSLKeyStoreLocation(String keyStoreLocation)");
        }
    }

    /**
     * Sets SSL key store password.
     * 
     * @param keyStorePassword
     *            Password set for the key store file using keytool.
     */
    public void setSSLKeyStorePassword(String keyStorePassword) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "keyStorePassword set: " + ((keyStorePassword != null) ? "Yes" : "No");
            logger.entering(getClass().getName(), "setSSLKeyStorePassword(String keyStorePassword)", parms);
        }
        this.sslKeyStorePassword = keyStorePassword;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSSLKeyStorePassword(String keyStorePassword)");
        }
    }

    /**
     * Sets SSL key manager algorithm.
     * 
     * @param keyMgrAlgorithm
     */
    public void setSSLKeyMgrAlgorithm(String keyMgrAlgorithm) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "keyMgrAlgorithm set: " + keyMgrAlgorithm;
            logger.entering(getClass().getName(), "setSSLKeyMgrAlgorithm(String keyMgrAlgorithm)", parms);
        }
        this.sslKeyMgrAlgorithm = keyMgrAlgorithm;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSSLKeyMgrAlgorithm(String keyMgrAlgorithm)");
        }
    }

    /**
     * Sets SSL key store type.
     * 
     * @param keyStoreType
     *            Valid values include "JKS" or "PKCS12".
     */
    public void setKeyStoreType(String keyStoreType) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "keyStoreType: " + keyStoreType;
            logger.entering(getClass().getName(), "setKeyStoreType(String keyStoreType)", parms);
        }
        this.keyStoreType = keyStoreType;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setKeyStoreType(String keyStoreType)");
        }
    }

    /**
     * Sets secure socket protocol type.
     * 
     * @param secureSocketProtocol
     *            Valid values include "SSL", "SSLv3", "TLSv1.1", "TLSv1.2", etc.
     */
    public void setSecureSocketProtocol(String secureSocketProtocol) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "secureSocketProtocol: " + secureSocketProtocol;
            logger.entering(getClass().getName(), "setSecureSocketProtocol(String secureSocketProtocol)", parms);
        }
        this.secureSocketProtocol = secureSocketProtocol;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setSecureSocketProtocol(String secureSocketProtocol)");
        }
    }

    /**
     * Retrieves the SSL connectivity.
     * 
     * @return true indicates SSL will be used, false indicates otherwise
     */
    public boolean getSSLConnection() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getSSLConnection()", parms);
        }
        boolean value = this.sslConnection;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getSSLConnection()", value);
        }
        return value;
    }
    
    /**
     * Gets SSL trust store path.
     * 
     * @return trustStore path.
     */
    public String getSSLTrustStoreLocation() {
        return this.sslTrustStoreLocation;
    }

    /**
     * Gets SSL trust store password.
     * 
     * @return trustStore password.
    */
    public String getSSLTrustStorePassword() {
        return this.sslTrustStorePassword;
    }

    /**
     * Gets SSL trust manager algorithm.
     * 
     * @return trust manager algorithm.
    */
    public String getSSLTrustMgrAlgorithm() {
        return this.sslTrustMgrAlgorithm;
    }

    /**
     * Gets SSL key store path.
     * 
     * @return keyStore path.
     */
    public String getSSLKeyStoreLocation() {
        return this.sslKeyStoreLocation;
    }

    /**
     * Gets SSL key store password.
     * 
     * @return keyStore password.
     */
    public String getSSLKeyStorePassword() {
        return this.sslKeyStorePassword;
    }

    /**
     * Gets SSL key manager algorithm.
     * 
     * @return key manager algorithm.
    */
    public String getSSLKeyMgrAlgorithm() {
        return this.sslKeyMgrAlgorithm;
    }

    /**
     * Gets SSL key store type.
     * 
     * @return keyStore type.
     */
    public String getKeyStoreType() {
        return this.keyStoreType;
    }

    /**
     * Gets SSL protocol.
     * 
     * @return SSL protocol.
     */
    public String getSecureSocketProtocol() {
        return this.secureSocketProtocol;
    }

    /**
     * Sets the location of the database metadata representing the target IMS
     * database. The metadata URL is the fully qualified name of the Java
     * metadata class generated by the IMS Explorer for Development.
     * 
     * The URL must begin with class://
     * 
     * @param metadataURL
     *            metadata url
     *            
     * @deprecated {@link #setDatabaseName(String)}
     */
    public void setMetadataURL(String metadataURL) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "metadataURL: " + metadataURL;
            logger.entering(getClass().getName(), "setMetadataURL(String)", parms);
        }
        this.metadataURL = metadataURL;
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setMetadataURL(String)");
        }
    }

    /**
     * Retrieves the URL of the database metadata representing the target IMS
     * database.
     * 
     * @return the metadata URL
     * 
     * @deprecated {@link #getDatabaseName()}
     */
    public String getMetadataURL() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getMetadataURL()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getMetadataURL()", metadataURL);
        }

        return this.metadataURL;  // This variable is used in getDatabaseName
    }

    /**
     * Sets the description of this DataSource.
     * 
     * @param description
     *            the description
     */
    public void setDescription(String description) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "description: " + description;
            logger.entering(getClass().getName(), "setDescription(String)", parms);
        }

        this.description = description;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setDescription(String)");
        }
    }

    /**
     * Retrieves the description of this DataSource.
     * 
     * @return the description
     */
    public String getDescription() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getDescription()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getDescription()", description);
        }

        return this.description;
    }

    // IMS specific properties

    /**
     * Sets the name of the IMS datastore to access. The datastore parameter
     * must match either the name of the datastore defined to ODBM or be blank.
     * The datastore name is defined in the ODBM CSLDCxxx PROCLIB member using
     * either the DATASTORE (NAME=name) or ALIAS (NAME=aliasname) parameter. If
     * the datastore value is left blank (or not supplied), IMS Connect will
     * connect to any available instance of ODBM as it is assumed all datastores
     * defined to ODBM are data shared.
     * <p>
     * For Type-2 connectivity: The datastore parameter refers to the 1 to 4
     * character identifier of the DRA startup table member. The member name
     * structure is DFSxxxx0 where xxxx is the member identifier. Example:
     * datastore parameter = SYS1 refers to DRA startup table member DFSSYS10
     * The ODBA startup table is a user created member that contains the IMS
     * subsystem connection properties which includes the IMS subsystem id. This
     * is not required to be set for the Java Dependent Region runtime.
     * 
     * @param datastoreName
     *            IMS datastore name
     */
    public void setDatastoreName(String datastoreName) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "datastoreName: " + datastoreName;
            logger.entering(getClass().getName(), "setDatastoreName(String)", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setDatastoreName(String)");
        }

        this.datastoreName = datastoreName;
    }

    /**
     * Retrieves name of the IMS datastore this DataSource is configured to
     * access.
     * 
     * @return the datastore name
     */
    public String getDatastoreName() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getDescription()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getDescription()", datastoreName);
        }

        return this.datastoreName;
    }

    /**
     * Sets the type of driver to use to connect to the database. Type-4:
     * Distributed access over TCP/IP. Type-2: Local access (no TCP/IP).
     * 
     * @param driverType
     *            Supported values are <code>IMSDataSource.DRIVER_TYPE_4</code>
     *            and <code>IMSDataSource.DRIVER_TYPE_2</code>
     * 
     */
    public void setDriverType(int driverType) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "driverType: " + driverType;
            logger.entering(getClass().getName(), "setDriverType(int)", parms);
        }

        this.driverType = driverType;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setDriverType(int)");
        }
    }

    /**
     * Retrieves the database connectivity method.
     * 
     * @return the database connectivity method
     */
    public int getDriverType() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getDriverType()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getDriverType()", driverType);
        }

        return this.driverType;
    }
    
    /**
     * Sets the name of the target IMS database to be accessed. 
     * 
     * If the metadata repository is the IMS catalog then the database name is
     * the name of the PSB (program specification block) containing the database(s)
     * to be accessed.
     * 
     * If the metadata repository is the file system then the database name
     * is the fully qualified name of the Java metadata class generated
     * by the IMS Explorer for Development.  In this case the name must begin with class://
     * 
     * @param databaseName
     *            database name
     */
    public void setDatabaseName(String databaseName) {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[3];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            parms[2] = "databaseName: " + databaseName;
            logger.entering(getClass().getName(), "setDatabaseName(String)", parms);
        }
        
        this.metadataURL = databaseName;  // This variable is used in setMetadataURL
        
        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "setDatabaseName(String)");
        }
    }
    
    /**
     * Retrieves the database name this DataSource is configured to access.
     * 
     * @return the database name
     */
    public String getDatabaseName() {
        if (logger.isLoggable(Level.FINER)) {
            Object[] parms = new Object[2];
            parms[0] = "Hash code: " + Integer.toHexString(hashCode());
            parms[1] = "Thread ID: " + Thread.currentThread().getId();
            logger.entering(getClass().getName(), "getDatabaseName()", parms);
        }

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(getClass().getName(), "getDatabaseName()", this.metadataURL);
        }
        
        return this.metadataURL;  // This variable is used in getMetadataURL
    }
        
	protected boolean getLLField() {
		return this.llField ;
	}

	protected void setLLField(boolean llField) {
		this.llField = llField;
	}
	public String getInitStatusGroup() {
		return initStatusGroup;
	}

	public void setInitStatusGroup(String initStatusGroup) {
		this.initStatusGroup = initStatusGroup;
	}
	public String toString() {
        StringBuffer value = new StringBuffer("IMSDataSource:");
        value.append(" Datastore Name = ").append(this.datastoreName);
        value.append(" Datastore Server = ").append(this.datastoreServer);
        value.append(" Database Name = ").append(this.metadataURL);
        value.append(" Port = ").append(this.portNumber);
        value.append(" Driver Type = ").append(this.driverType);
        value.append(" Description = ").append(this.description);
        value.append(" UserID = ").append(this.user);
        value.append(" SSL Connection = ").append(this.sslConnection);

        return value.toString();
    }

    /**
     * This function is not supported.
     * 
     * Returns true if this either implements the interface argument 
     * or is directly or indirectly a wrapper for an object that does. 
     * Returns false otherwise. If this implements the interface then 
     * return true, else if this is a wrapper then return the result 
     * of recursively calling <code>isWrapperFor</code> on the wrapped object. If 
     * this does not implement the interface and is not a wrapper, 
     * return false. This method should be implemented as a low-cost 
     * operation compared to <code>unwrap</code> so that callers can use this method 
     * to avoid expensive <code>unwrap</code> calls that may fail. If this method 
     * returns true then calling <code>unwrap</code> with the same argument should 
     * succeed. 
     * 
     * @param iface
     *            a Class defining an interface
     * @throws SQLException
     *            if an error occurs while determining whether this is a wrapper for an object with the given interface
     * @return true if this implements the interface or directly or indirectly wraps an object that does
     * @since 1.6
     */
    //Start JDK6
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        Object[] inserts = {"ResultSetMetaData.isWrapperFor(Class<?> iface)"};
        throw new SQLException(JDBCErrorMessages.getIMSBundle().getString("FUNCTION_NOT_SUPPORTED", inserts));
    }
    //End JDK6

    /**
     * This function is no supported.
     * 
     * Returns an object that implements the given interface to allow 
     * access to non-standard methods, or standard methods not exposed 
     * by the proxy. If the receiver implements the interface then the 
     * result is the receiver or a proxy for the receiver. If the 
     * receiver is a wrapper and the wrapped object implements the 
     * interface then the result is the wrapped object or a proxy for 
     * the wrapped object. Otherwise return the the result of calling 
     * <code>unwrap</code> recursively on the wrapped object or a proxy for that 
     * result. If the receiver is not a wrapper and does not implement 
     * the interface, then an <code>SQLException</code> is thrown. 
     * 
     * @param iface
     *            a Class defining an interface that the result must implement
     * @throws SQLException
     *            If no object found that implements the interface
     * @return an object the implements the interface. May be a proxy for the actual implementing object.
     * @since 1.6
     */
    //Start JDK6
    public <T> T unwrap(Class<T> iface) throws SQLException {
        Object[] inserts = {"ResultSetMetaData.isWrapperFor(Class<?> iface)"};
        throw new SQLException(JDBCErrorMessages.getIMSBundle().getString("FUNCTION_NOT_SUPPORTED", inserts));
    }
    //End JDK6
    
    public void setServerName(String serverName) {
    	setDatastoreServer(serverName);
    }
    
    public void setHost(String host) {
    	setDatastoreServer(host);
    }

	public Logger getParentLogger() throws SQLFeatureNotSupportedException {
		Logger parentLogger = logger.getParent();
		if (parentLogger == null) {
			parentLogger = logger;
		}
		return parentLogger;
	}
}
