package com.ibm.ims.jms;

/* (c) Copyright International Business Machines Corporation 2013. All rights reserved. */

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;

import com.ibm.ims.dli.DLIException;
import com.ibm.ims.dli.tm.TMDLICall;
import com.ibm.ims.dli.tm.TMErrorMessages;

/**
 * Factory class used to create IMS Point-to-Point messaging domain QueueConnection objects.
 * 
 * <p>The following code fragment shows how to use <code>IMSQueueConnectionFactory</code>
 * to obtain and use a <code>QueueConnection</code> object.
 * 
 * <pre>
 * IMSQueueConnectionFactory jmsConnectionFactory = new IMSQueueConnectionFactory();
 * QueueConnection jmsConnection = null;
 * QueueSession jmsQueueSession = null;
 * Queue jmsQueue = null;
 * QueueRequestor jmsQueueRequestor = null;
 *        
 * try {
 *     jmsConnectionFactory.setTimeout(1000);    //Set 10 second timeout
 *     jmsConnectionFactory.setResponseAreaLength(128000);    //Allocate 128K response area
 *     jmsConnection = jmsConnectionFactory.createQueueConnection();
 *     jmsQueueSession = jmsConnection.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
 *     jmsQueue = jmsQueueSession.createQueue("OTMDEST1");
 *           
 *     jmsQueueRequestor = new QueueRequestor(jmsQueueSession, jmsQueue);
 *     TextMessage sendMsg = jmsQueueSession.createTextMessage();
 *           
 *     String msg = "My Message";
 *     sendMsg.setText(msg);
 *     TextMessage replyMsg = (TextMessage)jmsQueueRequestor.request(sendMsg);
 *     System.out.println("Received message: "+replyMsg.getText());
 * } catch (Exception e) {
 *     e.printStackTrace();
 * }
 *
 * </pre>
 * 
 */
public class IMSQueueConnectionFactory implements QueueConnectionFactory {

    private static final Logger logger = Logger.getLogger("com.ibm.ims.db.opendb.dli.tm");
    final static int API_AIB_ICAL = 4;
    
    private int responseAreaLength = -1;
    private int timeout = 0;
    protected TMDLICall tmDLICall = null;
    
    public IMSQueueConnectionFactory() {
        try {
			this.tmDLICall = new TMDLICall();
		} catch (DLIException e) {
			throw new RuntimeException(e);
		}
	}

	public QueueConnection createQueueConnection() throws JMSException {
        if(logger.isLoggable(Level.FINER)){
            logger.entering(this.getClass().getName(), "createQueueConnection()");
        }
        
        QueueConnectionImpl conn = new QueueConnectionImpl(this);
        
        if(logger.isLoggable(Level.FINER)){
            logger.exiting(this.getClass().getName(), "createQueueConnection()");
        }

        return conn;
    }

    public QueueConnection createQueueConnection(String arg0, String arg1) throws JMSException {
        throw new JMSException(TMErrorMessages.getIMSBundle().getString("UNSUPPORTED_FUNCTION_SC"));
    }

    public Connection createConnection() throws JMSException {
        throw new JMSException(TMErrorMessages.getIMSBundle().getString("UNSUPPORTED_FUNCTION_SC"));
    }

    public Connection createConnection(String arg0, String arg1) throws JMSException {
        throw new JMSException(TMErrorMessages.getIMSBundle().getString("UNSUPPORTED_FUNCTION_SC"));
    }

    /**
     * Sets the timeout value. This value represents the number of 0.01 second units to wait for a response.
     * <p>By default, the value is set in the OTMA descriptor. If the value is set here to zero it will
     * automatically be changed to 1000, which indicates a 10 second timeout value. The range of acceptable
     * values is 0 - 999999.
     * 
     * @param timeout
     */
    public void setTimeout(int timeout) throws JMSException {
        if(timeout < 0 || timeout > 999999){
            Object[] inserts = new Object[1];
            inserts[0] = "" + timeout;
            throw new JMSException(TMErrorMessages.getIMSBundle().getString("INVALID_TIMEOUT_SC", inserts));
        }
        this.timeout = timeout;
    }
    
    /**
     * Sets the responseAreaLength value. This value represents the size of the byte array that you
     * want to have allocated to hold the response byte stream.
     * 
     * @param responseAreaLength
     */
    public void setResponseAreaLength(int responseAreaLength) throws JMSException{
        if(responseAreaLength < 0){
            Object[] inserts = new Object[1];
            inserts[0] = "" + responseAreaLength;
            throw new JMSException(TMErrorMessages.getIMSBundle().getString("INVALID_RESPONSEAREALEN_SC", inserts));
        }
        
        this.responseAreaLength = responseAreaLength;
    }
    
    public int getResponseAreaLength(){
        return responseAreaLength;
    }

    public int getTimeout() {
        return timeout;
    }
}
