package com.ibm.ims.base;

import java.io.Serializable;

import com.ibm.ims.base.DBType;
import com.ibm.ims.dli.DLIException;

/**
 * A DLITypeInfo object is used to provide the layout of a segment's fields.
 * With this metadata, the segment is able to provide access to all of the
 * fields in any segment using either the defined field name or the 1-based
 * index of the DLITypeInfo object in the array that is registered for each
 * segment type. The defined field name can be an alias for the defined search
 * field name in the DBD source file. When referring to the field, you can use
 * the field alias instead of the search or key field name, as these field names
 * are limited to only 8 characters. Note that the field names are case
 * sensitive and must be written exactly as defined in the DLITypeInfo objects.
 * When defining a key or search field, you must use the
 * <code>DLITypeInfo(String, int, int, int, String)</code> constructor, which
 * allows you to provide the alias and actual name of the search or key field.
 */
public class DLITypeInfo implements Serializable {

	String fieldName = null;
	String searchFieldName = null;
	String typeQualifier = null;
	String xmlSchema = null;
	int fieldType;
	int offset;
	int length;
	int keyType;

	// necessary data for PACKED and ZONED decimal
	String expandedString = null;
	int scale;
	boolean containsDecimal;
	boolean signed; // needed for support of 'S' in packed/zoned decimals
	int numberDigits;
	DBType dbType = null;
	boolean isRoot = false; // true if field is a unique key of a direct access
							// database (DEDB, HDAM, PHDAM) root segment

	public static final int TINYINT = 1;
	public static final int INTEGER = 2;
	public static final int CHAR = 3;
	public static final int DOUBLE = 4;
	public static final int FLOAT = 5;
	public static final int BIT = 6;
	public static final int BLOB = 17; // synonym for BINARY
	public static final int BIGINT = 8;
	public static final int SMALLINT = 9;
	public static final int VARCHAR = 10;
	public static final int PACKEDDECIMAL = 11;
	public static final int ZONEDDECIMAL = 12;
	public static final int DATE = 13;
	public static final int TIME = 14;
	public static final int TIMESTAMP = 15;
	public static final int TYPELIST = 16;
	public static final int BINARY = 17;
	public static final int SECONDARY_INDEX = 18;
	public static final int CLOB = 19;
	public static final int DBCS = 20;
	public static final int XML = 21;
	public static final int USMALLINT = 22;
	public static final int UINTEGER = 23;
	public static final int UBIGINT = 24;
	public static final int UTINYINT = 25;
	public static final int STRUCT = 26;
	public static final int ARRAY = 27;
	public static final int OTHER = 28;

	// Key field information (NOT_KEY is default)
	public static final int NOT_KEY = 2200;
	public static final int UNIQUE_KEY = 2201;
	public static final int NON_UNIQUE_KEY = 2202;

	/**
	 * Constructs a DLITypeInfo object. This constructor is only to be used with
	 * non-key and non-search fields. It cannot be used for zoned decmial,
	 * packed decimal, date, time, and timestamp fields. Use the
	 * <code>DLITypeInfo(String, String, int, int, int)</code> or
	 * <code>DLITypeInfo(String, String, int, int, int, String)</code>
	 * constructors for these types.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>CHAR</code>,
	 * <code>INTEGER</code>, <code>DOUBLE</code>, <code>FLOAT</code>,
	 * <code>BIT</code>, <code>BLOB</code>, <code>BIGINT</code>,
	 * <code>TINYINT</code>, <code>BINARY</code>, and <code>SMALLINT</code>.
	 *
	 * Certain types have predefined lengths and cannot be changed. The
	 * <code>INTEGER</code> and <code>FLOAT</code> types are always 4 bytes
	 * long. The <code>DOUBLE</code> and <code>BIGINT</code> types are always 8
	 * bytes long. The <code>SMALLINT</code> type is always 2 bytes long. The
	 * <code>TINYINT</code> and <code>BIT</code> types are always 1 byte long.
	 * If another length value is specified for one of these types, an exception
	 * will be thrown.
	 *
	 * @param fieldName
	 *            The name of the field.
	 * @param type
	 *            the type of the field
	 * @param startingOffset
	 *            the starting offset in the I/O area of this field, beginning
	 *            at offset 1
	 * @param length
	 *            the length, in bytes, of this field
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, int type, int startingOffset, int length) {

		// Copy Strings to remove potential Cross-Heap References
		String copyFieldName = new String(fieldName);

		if (startingOffset < 1) {
			Object[] inserts = { copyFieldName, Integer.toString(startingOffset) };
			throw new IllegalArgumentException(
					IMSErrorMessages.getIMSBundle().getString("INVALID_STARTING_OFFSET", inserts));
		}

		if (length < 1) {
			Object[] inserts = { copyFieldName, Integer.toString(length) };
			throw new IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("INVALID_LENGTH", inserts));
		}

		this.fieldName = copyFieldName;
		this.fieldType = type;
		this.offset = startingOffset - 1; // subtract 1 as arrays are zero-based

		switch (this.fieldType) {
		case INTEGER:
		case UINTEGER:
		case FLOAT:
			if (length != 4) {
				Object[] inserts = { copyFieldName, Integer.toString(length) };
				throw new IllegalArgumentException(
						IMSErrorMessages.getIMSBundle().getString("INVALID_LENGTH", inserts));
			}
			this.length = 4;
			break;
		case DOUBLE:
		case BIGINT:
		case UBIGINT:
			if (length != 8) {
				Object[] inserts = { copyFieldName, Integer.toString(length) };
				throw new IllegalArgumentException(
						IMSErrorMessages.getIMSBundle().getString("INVALID_LENGTH", inserts));
			}
			this.length = 8;
			break;
		case SMALLINT:
		case USMALLINT:
			if (length != 2) {
				Object[] inserts = { copyFieldName, Integer.toString(length) };
				throw new IllegalArgumentException(
						IMSErrorMessages.getIMSBundle().getString("INVALID_LENGTH", inserts));
			}
			this.length = 2;
			break;
		case CHAR:
		case VARCHAR:
		case DBCS:
			this.length = length;
			break;
		case TINYINT:
		case UTINYINT:
		case BIT:
			if (length != 1) {
				Object[] inserts = { copyFieldName, Integer.toString(length) };
				throw new IllegalArgumentException(
						IMSErrorMessages.getIMSBundle().getString("INVALID_LENGTH", inserts));
			}
			this.length = 1;
			break;
		case TYPELIST:
			this.length = length;
			break;
		case BINARY: // also BLOB, as it is a synonym
			this.length = length;
			break;
		default: {
			Object[] inserts = { copyFieldName };
			throw new IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("UNSUPPORTED_TYPE", inserts));
		}
		}

		// Initialize key field to no
		this.keyType = DLITypeInfo.NOT_KEY;
	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is only to be used with
	 * non-key and non-search fields. It cannot be used for zoned decmial,
	 * packed decimal, date, time, and timestamp fields. Use the
	 * <code>DLITypeInfo(String, String, int, int, int)</code> or
	 * <code>DLITypeInfo(String, String, int, int, int, String)</code>
	 * constructors for these types.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>CHAR</code>,
	 * <code>INTEGER</code>, <code>DOUBLE</code>, <code>FLOAT</code>,
	 * <code>BIT</code>, <code>BLOB</code>, <code>BIGINT</code>,
	 * <code>TINYINT</code>, <code>BINARY</code>, and <code>SMALLINT</code>.
	 *
	 * Certain types have predefined lengths and cannot be changed. The
	 * <code>INTEGER</code> and <code>FLOAT</code> types are always 4 bytes
	 * long. The <code>DOUBLE</code> and <code>BIGINT</code> types are always 8
	 * bytes long. The <code>SMALLINT</code> type is always 2 bytes long. The
	 * <code>TINYINT</code> and <code>BIT</code> types are always 1 byte long.
	 * If another length value is specified for one of these types, an exception
	 * will be thrown.
	 *
	 * @param fieldName
	 *            The name of the field.
	 * @param type
	 *            the type of the field
	 * @param startingOffset
	 *            the starting offset in the I/O area of this field, beginning
	 *            at offset 1
	 * @param length
	 *            the length, in bytes, of this field
	 * @param dbType
	 *            the database type
	 * @param isRoot
	 *            true if field is a unique key of a direct access database
	 *            (DEDB, HDAM, PHDAM) root segment
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, int type, int startingOffset, int length, DBType dbType, boolean isRoot) {
		this(fieldName, type, startingOffset, length);
		this.dbType = dbType;
		this.isRoot = isRoot;
	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is to be used with all
	 * key and search fields. It cannot be used for zoned decmial, packed
	 * decimal, date, time, and timestamp fields. Use the
	 * <code>DLITypeInfo(String, String, int, int, int)</code> or
	 * <code>DLITypeInfo(String, String, int, int, int, String)</code>
	 * constructors for these types. It provides functionality to provide an
	 * alias for the key and search fields. This alias name is not limited to 8
	 * characters.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>CHAR</code>,
	 * <code>INTEGER</code>, <code>DOUBLE</code>, <code>FLOAT</code>,
	 * <code>BIT</code>, <code>BLOB</code>, <code>BIGINT</code>,
	 * <code>TINYINT</code>, <code>BINARY</code>, and <code>SMALLINT</code>.
	 *
	 * Certain types have predefined lengths and cannot be changed. The
	 * <code>INTEGER</code> and <code>FLOAT</code> types are always 4 bytes
	 * long. The <code>DOUBLE</code> and <code>BIGINT</code> types are always 8
	 * bytes long. The <code>SMALLINT</code> type is always 2 bytes long. The
	 * <code>TINYINT</code> and <code>BIT</code> types are always 1 byte long.
	 * If another length value is specified for one of these types, an exception
	 * will be thrown.
	 *
	 * @param fieldName
	 *            The name of the field. This name can be an alias that maps to
	 *            the the actual name as defined in the DBD source file, which
	 *            is given by the <code>searchFieldName</code> parameter.
	 * @param type
	 *            the type of the field
	 * @param startingOffset
	 *            the starting offset in the I/O area of this field, beginning
	 *            at offset 1
	 * @param length
	 *            the length, in bytes, of this field
	 * @param searchFieldName
	 *            the name of the search or key field exactly as defined in the
	 *            DBD source file
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, int type, int startingOffset, int length, String searchFieldName) {

		// Copy Strings to remove potential Cross-Heap References
		// fieldName is copied in DLITypeInfo(String, int, int, int);
		this(fieldName, type, startingOffset, length);
		if (searchFieldName != null) {
			this.searchFieldName = new String(searchFieldName);
		}
	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is to be used with all
	 * key and search fields. It cannot be used for zoned decmial, packed
	 * decimal, date, time, and timestamp fields. Use the
	 * <code>DLITypeInfo(String, String, int, int, int)</code> or
	 * <code>DLITypeInfo(String, String, int, int, int, String)</code>
	 * constructors for these types. It provides functionality to provide an
	 * alias for the key and search fields. This alias name is not limited to 8
	 * characters.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>CHAR</code>,
	 * <code>INTEGER</code>, <code>DOUBLE</code>, <code>FLOAT</code>,
	 * <code>BIT</code>, <code>BLOB</code>, <code>BIGINT</code>,
	 * <code>TINYINT</code>, <code>BINARY</code>, and <code>SMALLINT</code>.
	 *
	 * Certain types have predefined lengths and cannot be changed. The
	 * <code>INTEGER</code> and <code>FLOAT</code> types are always 4 bytes
	 * long. The <code>DOUBLE</code> and <code>BIGINT</code> types are always 8
	 * bytes long. The <code>SMALLINT</code> type is always 2 bytes long. The
	 * <code>TINYINT</code> and <code>BIT</code> types are always 1 byte long.
	 * If another length value is specified for one of these types, an exception
	 * will be thrown.
	 *
	 * @param fieldName
	 *            The name of the field. This name can be an alias that maps to
	 *            the the actual name as defined in the DBD source file, which
	 *            is given by the <code>searchFieldName</code> parameter.
	 * @param type
	 *            the type of the field
	 * @param startingOffset
	 *            the starting offset in the I/O area of this field, beginning
	 *            at offset 1
	 * @param length
	 *            the length, in bytes, of this field
	 * @param searchFieldName
	 *            the name of the search or key field exactly as defined in the
	 *            DBD source file
	 * @param dbType
	 *            the database type
	 * @param isRoot
	 *            true if field is a unique key of a direct access database
	 *            (DEDB, HDAM, PHDAM) root segment
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, int type, int startingOffset, int length, String searchFieldName,
			DBType dbType, boolean isRoot) {
		this(fieldName, type, startingOffset, length, searchFieldName);
		this.dbType = dbType;
		this.isRoot = isRoot;
	}

	/**
	 * Constructs a DLITypeInfo object, adding the ability to specify a key
	 * type. Values include: NOT_KEY indicates field is not a key field
	 * (default) UNIQUE_KEY indicates field is a (or part of) a unique key
	 * NON_UNIQUE_KEY indicates field is a (or part of) a non-unique key
	 */
	public DLITypeInfo(String fieldName, int type, int startingOffset, int length, String searchFieldName,
			int keyType) {
		this(fieldName, type, startingOffset, length, searchFieldName);
		this.keyType = keyType;
	}

	/**
	 * Constructs a DLITypeInfo object, adding the ability to specify a key type
	 * and db type. Key values include: NOT_KEY indicates field is not a key
	 * field (default) UNIQUE_KEY indicates field is a (or part of) a unique key
	 * NON_UNIQUE_KEY indicates field is a (or part of) a non-unique key
	 * 
	 * @param dbType
	 *            the database type
	 * @param isRoot
	 *            true if field is a unique key of a direct access database
	 *            (DEDB, HDAM, PHDAM) root segment
	 */
	public DLITypeInfo(String fieldName, int type, int startingOffset, int length, String searchFieldName, int keyType,
			DBType dbType, boolean isRoot) {
		this(fieldName, type, startingOffset, length, searchFieldName, keyType);
		this.dbType = dbType;
		this.isRoot = isRoot;
	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is to be used with
	 * non-key and non-search fields of zoned or packed decimal,
	 * Date/Time/Timestamp data, or clob strings. Use the
	 * <code>DLITypeInfo(String, String, int, int, int)</code> or
	 * <code>DLITypeInfo(String, String, int, int, int, String)</code>
	 * constructors for all other types. The uses of the type qualifier are as
	 * follows:
	 *
	 * If the field type is either packed or zoned decimal, the type qualifier
	 * is the PICTURE string representing the layout of the field. All COBOL
	 * PICTURE strings containing valid combinations of 9s, Vs, and Ss are
	 * supported.
	 *
	 * If the field contains Date/Time/Timestamp data, the type qualifier
	 * specifies the format of the data. For example, a type qualifier of
	 * ddMMyyyy indicates that the data is formatted as follows: 02011999 is
	 * January 2, 1999.
	 *
	 * For DATE and TIME all formatting opions in the java.text.SimpleDateFormat
	 * class are supported. For the TIMESTAMP type an additional formatting
	 * option 'f' has been provided for nanoseconds. TIMESTAMP can contain up to
	 * 9 'f's and replaces the 'S' options for milliseconds; instead, 'fff'
	 * indicates milliseconds of precision. An example TIMESTAMP format is
	 * yyyy-mm-dd hh:mm:ss.fffffffff
	 *
	 * If the field type is a CLOB the type qualifier is the name of the child
	 * overflow segment. The child overflow segment is used to store any
	 * overflow characters after the first length-4 characters which will be
	 * stored in this field. This base field will only hold length-4 characters
	 * since the CLOB field type reserves the first 4 bytes for version info and
	 * length.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>CLOB</code>,
	 * <code>DATE</code>, <code>TIME</code>, <code>TIMESTAMP</code>,
	 * <code>ZONEDDECIMAL</code> and <code>PACKEDDECIMAL</code>.
	 *
	 * @param fieldName
	 *            The name of the field.
	 * @param typeQualifier
	 *            the type qualifier for the field, or the child overflow
	 *            segment name, as described above
	 * @param type
	 *            the type of the field
	 * @param startingOffset
	 *            the starting offset in the I/O area of this field, beginning
	 *            at offset 1
	 * @param length
	 *            the length of the field for a PACKEDDECIMAL number, the length
	 *            can be calculated with the following function: ceiling((number
	 *            of digits + 1)/2) for a ZONEDDECIMAL number, the length can be
	 *            calculated with the following function: number of digits + 1
	 *            (if the decimal is stored in memory) number of digits (if the
	 *            decimal is not stored in memory)
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, String typeQualifier, int type, int startingOffset, int length) {

		// Copy Strings to remove potential Cross-Heap References
		String copyFieldName = new String(fieldName);
		String copyTypeQualifier = new String(typeQualifier);

		if (startingOffset < 1) {
			Object[] inserts = { copyFieldName, Integer.toString(startingOffset) };
			throw new IllegalArgumentException(
					IMSErrorMessages.getIMSBundle().getString("INVALID_STARTING_OFFSET", inserts));
		}

		if (length < 1) {
			Object[] inserts = { copyFieldName, Integer.toString(length) };
			throw new IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("INVALID_LENGTH", inserts));
		}

		if (typeQualifier == null) {
			Object[] inserts = { copyFieldName };
			throw new IllegalArgumentException(
					IMSErrorMessages.getIMSBundle().getString("NULL_TYPE_QUALIFIER", inserts));
		}

		this.fieldName = copyFieldName;
		this.typeQualifier = copyTypeQualifier;
		this.fieldType = type;
		this.offset = startingOffset - 1; // subtract 1 as arrays are zero-based
		this.length = length;

		switch (this.fieldType) {
		case PACKEDDECIMAL:
		case ZONEDDECIMAL:
		case BINARY:
			this.scale = this.determineScale();
			break;
		case DATE:
		case TIME:
		case TIMESTAMP:
			break;
		case CLOB:
			break;
		default: {
			Object[] inserts = { copyFieldName };
			throw new IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("UNSUPPORTED_TYPE", inserts));
		}
		}

	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is to be used for
	 * virtual XML columns.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>XML</code>,
	 *
	 * @param fieldName
	 *            The name of the field.
	 * @param xmlSchema
	 *            The xmlSchema name
	 * @param type
	 *            the type of the field
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, String xmlSchema, int type) {

		// Copy Strings to remove potential Cross-Heap References
		String copyFieldName = new String(fieldName);
		String copyXmlSchema = new String(xmlSchema);

		this.fieldName = copyFieldName;
		this.xmlSchema = copyXmlSchema;
		this.fieldType = type;
		// this.offset = - 1;
		// this.length = -1;

		switch (this.fieldType) {
		case XML:
			break;
		default: {
			Object[] inserts = { copyFieldName };
			throw new IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("UNSUPPORTED_TYPE", inserts));
		}
		}

	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is to be used for
	 * virtual XML columns for intact data.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>XML</code>,
	 *
	 * @param fieldName
	 *            The name of the field.
	 * @param overflowSegment
	 *            Child overflow segment name, as described above
	 * @param xmlSchema
	 *            The xmlSchema name      
	 * @param type
	 *            the type of the field
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, String overflowSegment, String xmlSchema, int type, int startingOffset,
			int length) {

		// Copy Strings to remove potential Cross-Heap References
		String copyFieldName = new String(fieldName);
		String copyTypeQualifier = new String(overflowSegment);
		String copyXmlSchema = new String(xmlSchema);

		if (startingOffset < 1) {
			Object[] inserts = { copyFieldName, Integer.toString(startingOffset) };
			throw new IllegalArgumentException(
					IMSErrorMessages.getIMSBundle().getString("INVALID_STARTING_OFFSET", inserts));
		}

		if (length < 1) {
			Object[] inserts = { copyFieldName, Integer.toString(length) };
			throw new IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("INVALID_LENGTH", inserts));
		}

		if (overflowSegment == null) {
			Object[] inserts = { copyFieldName };
			throw new IllegalArgumentException(
					IMSErrorMessages.getIMSBundle().getString("NULL_OVERFLOWSEGMENT", inserts));
		}

		this.fieldName = copyFieldName;
		this.typeQualifier = copyTypeQualifier;
		this.xmlSchema = copyXmlSchema;
		this.fieldType = type;
		this.offset = startingOffset - 1; // subtract 1 as arrays are zero-based
		this.length = length;

		switch (this.fieldType) {
		case XML:
			break;
		default: {
			Object[] inserts = { copyFieldName };
			throw new IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("UNSUPPORTED_TYPE", inserts));
		}
		}

	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is to be used with
	 * non-key and non-search fields of zoned or packed decimal,
	 * Date/Time/Timestamp data, or clob strings. Use the
	 * <code>DLITypeInfo(String, String, int, int, int)</code> or
	 * <code>DLITypeInfo(String, String, int, int, int, String)</code>
	 * constructors for all other types. The uses of the type qualifier are as
	 * follows:
	 *
	 * If the field type is either packed or zoned decimal, the type qualifier
	 * is the PICTURE string representing the layout of the field. All COBOL
	 * PICTURE strings containing valid combinations of 9s, Vs, and Ss are
	 * supported.
	 *
	 * If the field contains Date/Time/Timestamp data, the type qualifier
	 * specifies the format of the data. For example, a type qualifier of
	 * ddMMyyyy indicates that the data is formatted as follows: 02011999 is
	 * January 2, 1999.
	 *
	 * For DATE and TIME all formatting opions in the java.text.SimpleDateFormat
	 * class are supported. For the TIMESTAMP type an additional formatting
	 * option 'f' has been provided for nanoseconds. TIMESTAMP can contain up to
	 * 9 'f's and replaces the 'S' options for milliseconds; instead, 'fff'
	 * indicates milliseconds of precision. An example TIMESTAMP format is
	 * yyyy-mm-dd hh:mm:ss.fffffffff
	 *
	 * If the field type is a CLOB the type qualifier is the name of the child
	 * overflow segment. The child overflow segment is used to store any
	 * overflow characters after the first length-4 characters which will be
	 * stored in this field. This base field will only hold length-4 characters
	 * since the CLOB field type reserves the first 4 bytes for version info and
	 * length.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>CLOB</code>,
	 * <code>DATE</code>, <code>TIME</code>, <code>TIMESTAMP</code>,
	 * <code>ZONEDDECIMAL</code> and <code>PACKEDDECIMAL</code>.
	 *
	 * @param fieldName
	 *            The name of the field.
	 * @param typeQualifier
	 *            the type qualifier for the field, or the child overflow
	 *            segment name, as described above
	 * @param type
	 *            the type of the field
	 * @param startingOffset
	 *            the starting offset in the I/O area of this field, beginning
	 *            at offset 1
	 * @param length
	 *            the length of the field for a PACKEDDECIMAL number, the length
	 *            can be calculated with the following function: ceiling((number
	 *            of digits + 1)/2) for a ZONEDDECIMAL number, the length can be
	 *            calculated with the following function: number of digits + 1
	 *            (if the decimal is stored in memory) number of digits (if the
	 *            decimal is not stored in memory)
	 * @param dbType
	 *            the database type
	 * @param isRoot
	 *            true if field is a unique key of a direct access database
	 *            (DEDB, HDAM, PHDAM) root segment
	 * 
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, String typeQualifier, int type, int startingOffset, int length, DBType dbType,
			boolean isRoot) {
		this(fieldName, typeQualifier, type, startingOffset, length);
		this.dbType = dbType;
		this.isRoot = isRoot;
	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is to be used with all
	 * search fields of zoned or packed decimal, Date/Time/Timestamp data, or
	 * clob strings. Use the
	 * <code>DLITypeInfo(String, String, int, int, int)</code> or
	 * <code>DLITypeInfo(String, String, int, int, int, String, int)</code>
	 * constructors for all other types. This constructor provides functionality
	 * to provide an alias for the key and search fields. This alias name is not
	 * limited to 8 characters. The uses of the type qualifier are as follows:
	 *
	 * If the field type is either packed or zoned decimal, the type qualifier
	 * is the PICTURE string representing the layout of the field. All COBOL
	 * PICTURE strings containing valid combinations of 9s, Vs, and Ss are
	 * supported.
	 *
	 * If the field contains Date/Time/Timestamp data, the type qualifier
	 * specifies the format of the data. For example, a type qualifier of
	 * ddMMyyyy indicates that the data is formatted as follows: 02011999 is
	 * January 2, 1999.
	 *
	 * For DATE and TIME all formatting opions in the java.text.SimpleDateFormat
	 * class are supported. For the TIMESTAMP type an additional formatting
	 * option 'f' has been provided for nanoseconds. TIMESTAMP can contain up to
	 * 9 'f's and replaces the 'S' options for milliseconds; instead, 'fff'
	 * indicates milliseconds of precision. An example TIMESTAMP format is
	 * yyyy-mm-dd hh:mm:ss.fffffffff
	 *
	 * If the field type is a CLOB the type qualifier is the name of the child
	 * overflow segment. The child overflow segment is used to store any
	 * overflow characters after the first length-4 characters which will be
	 * stored in this field. This base field will only hold length-4 characters
	 * since the CLOB field type reserves the first 4 bytes for version info and
	 * length.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>CLOB</code>,
	 * <code>DATE</code>, <code>TIME</code>, <code>TIMESTAMP</code>,
	 * <code>ZONEDDECIMAL</code> and <code>PACKEDDECIMAL</code>.
	 *
	 * @param fieldName
	 *            The name of the field. This name can be an alias that maps to
	 *            the the actual name as defined in the DBD source file, which
	 *            is given by the <code>searchFieldName</code> parameter.
	 * @param typeQualifier
	 *            the type qualifier for the field
	 * @param type
	 *            the type of the field
	 * @param startingOffset
	 *            the starting offset in the I/O area of this field, beginning
	 *            at offset 1
	 * @param length
	 *            the length of the field
	 * @param searchFieldName
	 *            the name of the search or key field exactly as defined in the
	 *            DBD source file
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, String typeQualifier, int type, int startingOffset, int length,
			String searchFieldName) {
		// Copy Strings to remove potential Cross-Heap References
		// fieldName and typeQualifier are copied in DLITypeInfo(String, String,
		// int, int, int);
		this(fieldName, typeQualifier, type, startingOffset, length);
		if (searchFieldName != null) {
			this.searchFieldName = new String(searchFieldName);
		}
	}

	/**
	 * Constructs a DLITypeInfo object. This constructor is to be used with all
	 * search fields of zoned or packed decimal, Date/Time/Timestamp data, or
	 * clob strings. Use the
	 * <code>DLITypeInfo(String, String, int, int, int)</code> or
	 * <code>DLITypeInfo(String, String, int, int, int, String, int)</code>
	 * constructors for all other types. This constructor provides functionality
	 * to provide an alias for the key and search fields. This alias name is not
	 * limited to 8 characters. The uses of the type qualifier are as follows:
	 *
	 * If the field type is either packed or zoned decimal, the type qualifier
	 * is the PICTURE string representing the layout of the field. All COBOL
	 * PICTURE strings containing valid combinations of 9s, Vs, and Ss are
	 * supported.
	 *
	 * If the field contains Date/Time/Timestamp data, the type qualifier
	 * specifies the format of the data. For example, a type qualifier of
	 * ddMMyyyy indicates that the data is formatted as follows: 02011999 is
	 * January 2, 1999.
	 *
	 * For DATE and TIME all formatting opions in the java.text.SimpleDateFormat
	 * class are supported. For the TIMESTAMP type an additional formatting
	 * option 'f' has been provided for nanoseconds. TIMESTAMP can contain up to
	 * 9 'f's and replaces the 'S' options for milliseconds; instead, 'fff'
	 * indicates milliseconds of precision. An example TIMESTAMP format is
	 * yyyy-mm-dd hh:mm:ss.fffffffff
	 *
	 * If the field type is a CLOB the type qualifier is the name of the child
	 * overflow segment. The child overflow segment is used to store any
	 * overflow characters after the first length-4 characters which will be
	 * stored in this field. This base field will only hold length-4 characters
	 * since the CLOB field type reserves the first 4 bytes for version info and
	 * length.
	 *
	 * <p>
	 * The <code>DLITypeInfo</code> class defines several constants that can be
	 * supplied as values for the <code>type</code> argument: <code>CLOB</code>,
	 * <code>DATE</code>, <code>TIME</code>, <code>TIMESTAMP</code>,
	 * <code>ZONEDDECIMAL</code> and <code>PACKEDDECIMAL</code>.
	 *
	 * @param fieldName
	 *            The name of the field. This name can be an alias that maps to
	 *            the the actual name as defined in the DBD source file, which
	 *            is given by the <code>searchFieldName</code> parameter.
	 * @param typeQualifier
	 *            the type qualifier for the field
	 * @param type
	 *            the type of the field
	 * @param startingOffset
	 *            the starting offset in the I/O area of this field, beginning
	 *            at offset 1
	 * @param length
	 *            the length of the field
	 * @param searchFieldName
	 *            the name of the search or key field exactly as defined in the
	 *            DBD source file
	 * @param dbType
	 *            the database type
	 * @param isRoot
	 *            true if field is a unique key of a direct access database
	 *            (DEDB, HDAM, PHDAM) root segment
	 *
	 * @exception IllegalArgumentException
	 *                if the starting offset is less than zero, an unsupported
	 *                type is given, or an invalid length is given
	 */
	public DLITypeInfo(String fieldName, String typeQualifier, int type, int startingOffset, int length,
			String searchFieldName, DBType dbType, boolean isRoot) {
		this(fieldName, typeQualifier, type, startingOffset, length, searchFieldName);
		this.dbType = dbType;
		this.isRoot = isRoot;
	}

	/**
	 * Constructs a DLITypeInfo object, adding the ability to specify a key
	 * type. Values include: NOT_KEY indicates field is not a key field
	 * (default) UNIQUE_KEY indicates field is a (or part of) a unique key
	 * NON_UNIQUE_KEY indicates field is a (or part of) a non-unique key
	 */
	public DLITypeInfo(String fieldName, String typeQualifier, int type, int startingOffset, int length,
			String searchFieldName, int keyType) {
		this(fieldName, typeQualifier, type, startingOffset, length, searchFieldName);
		this.keyType = keyType;
	}

	/**
	 * Constructs a DLITypeInfo object, adding the ability to specify a key type
	 * and db type. Key values include: NOT_KEY indicates field is not a key
	 * field (default) UNIQUE_KEY indicates field is a (or part of) a unique key
	 * NON_UNIQUE_KEY indicates field is a (or part of) a non-unique key
	 * 
	 * @param dbType
	 *            the database type
	 * @param isRoot
	 *            true if field is a unique key of a direct access database
	 *            (DEDB, HDAM, PHDAM) root segment
	 */
	public DLITypeInfo(String fieldName, String typeQualifier, int type, int startingOffset, int length,
			String searchFieldName, int keyType, DBType dbType, boolean isRoot) {
		this(fieldName, typeQualifier, type, startingOffset, length, searchFieldName, keyType);
		this.dbType = dbType;
		this.isRoot = isRoot;
	}

	// What is this used for?
	protected DLITypeInfo(String fieldName, int type, String searchFieldName, int length) {

		this.fieldName = fieldName;
		this.fieldType = type;
		this.searchFieldName = searchFieldName;
		this.length = length;
		// Initialize key field to no
		this.keyType = DLITypeInfo.NOT_KEY;
	}

	/**
	 * Insert the method's description here. Creation date: (02/27/2000 2:20:41
	 * PM)
	 * 
	 * @return int
	 */
	private int determineScale() {

		int scale = 0;
		int index = -1;

		this.expandedString = this.expandStringAndGetNumDigits(this.typeQualifier);

		switch (this.fieldType) {
		case PACKEDDECIMAL: {
			index = this.expandedString.indexOf('V');
			if (index == -1) {
				index = this.expandedString.indexOf('v');
			}
			if (index != -1) {
				scale = this.expandedString.length() - index - 1;
			} else if ((this.expandedString.startsWith("P")) || (this.expandedString.startsWith("p"))) {
				scale = this.expandedString.length();
			} else if ((this.expandedString.endsWith("P")) || (this.expandedString.endsWith("p"))) {
				scale = 0;
			}
			break;
		}
		case ZONEDDECIMAL: {
			index = this.expandedString.indexOf('V');
			if (index == -1) {
				index = this.expandedString.indexOf('v');
			}
			if (index != -1) {
				scale = this.expandedString.length() - index - 1;
				break;
			}
			index = this.expandedString.indexOf('.');
			if (index != -1) {
				this.containsDecimal = true;
				scale = this.expandedString.length() - index - 1;
				break;
			} else if ((this.expandedString.startsWith("P")) || (this.expandedString.startsWith("p"))) {
				scale = this.expandedString.length();
			} else if ((this.expandedString.endsWith("P")) || (this.expandedString.endsWith("p"))) {
				scale = 0;
			}

			break;
		}
		}

		return scale;
	}

	/**
	 * Insert the method's description here. Creation date: (02/27/2000 1:44:22
	 * PM)
	 */
	private String expandStringAndGetNumDigits(String pictureString) {

		StringBuffer expandedString = new StringBuffer();
		int startingIndex = -1;
		int index = 0;

		int expandCount = 0;
		char expandChar = 0;

		if ((pictureString.startsWith("S")) || (pictureString.startsWith("s"))) {
			pictureString = pictureString.substring(1); // remove the leading
														// 'S'
			signed = true; // leading 'S' signifies the number is signed
		} else {
			signed = false;
		}

		while ((index = pictureString.indexOf('(', startingIndex)) != -1) {
			expandedString.append(pictureString.substring(startingIndex + 1, index));
			startingIndex = pictureString.indexOf(')', index);
			expandChar = pictureString.charAt(index - 1);
			expandCount = Integer.parseInt(pictureString.substring(index + 1, startingIndex));
			for (int i = 0; i < expandCount - 1; i++) {
				expandedString.append(expandChar);
			}
		}

		expandedString.append(pictureString.substring(startingIndex + 1, pictureString.length()));

		// determine the number of digits in the PICTURE string
		int count = 0;
		int length = expandedString.length();
		for (int i = 0; i < length; i++) {
			if (String.valueOf(expandedString.charAt(i)).equals("9")) {
				count++;
			}
		}

		this.numberDigits = count;

		return expandedString.toString();
	}
	/**
	 * Sets the length of the field's value in the segment.
	 * 
	 * @param int
	 *            the length, in bytes
	 */
	/*
	 * public void setFieldLength(int length) { this.length = length; }
	 */

	/**
	 * Returns the length of the field's value in the segment.
	 * 
	 * @return int the length, in bytes
	 */
	public int getFieldLength() {
		return length;
	}

	/**
	 * Returns the field name given to this field in the segment.
	 * 
	 * @return String the name of the field
	 */
	public String getFieldName() {
		return fieldName;
	}

	/**
	 * Returns the offset of the field in the segment's I/O area.
	 * 
	 * @return int the zero-based offset
	 */
	public int getFieldOffset() {
		return offset;
	}

	/**
	 * Returns the XML Schema assigned to a field's value in the segment.
	 * 
	 * @return String the XML Schema associated
	 */
	public String getXmlSchema() {
		return xmlSchema;
	}

	/**
	 * Returns the type assigned to a field's value in the segment.
	 * 
	 * @return int the type
	 */
	public int getFieldType() {
		return fieldType;
	}

	/**
	 * Returns the type of key the field is: DLITypeInfo.NOT_KEY
	 * DLITypeInfo.UNIQUE_KEY DLITypeInfo.NON_UNIQUE_KEY
	 */
	public int getKeyType() {
		return keyType;
	}

	/**
	 * Returns the search field name given to this field in the segment, which
	 * must be exactly as defined in the DBD source file.
	 *
	 * @return String the name of the search field as defined in the DBD source
	 *         file
	 */
	public String getSearchFieldName() {
		return searchFieldName;
	}

	/**
	 * Returns the type qualifier for this type.
	 * 
	 * @return the type qualifier
	 */
	public String getTypeQualifier() {
		return this.typeQualifier;
	}

	public DBType getDBType() {
		return this.dbType;
	}

	public boolean getIsRoot() {
		return this.isRoot;
	}

	/**
	 * Returns the scale of a BigDecimal
	 * 
	 * @return the type qualifier
	 */
	public int getScale() throws DLIException {
		if (this.fieldType != DLITypeInfo.ZONEDDECIMAL && this.fieldType != DLITypeInfo.PACKEDDECIMAL) {
			Object[] inserts = { "com.ibm.ims.base.DLITypeInfo.getScale()", this.getDataTypeName(this.fieldType),
					this.fieldName };
			throw new DLIException(IMSErrorMessages.getIMSBundle().getString("DATA_TYPE_NOT_BIGDECIMAL", inserts));
		}
		return this.scale;
	}

	/**
	 * Returns the precision of a BigDecimal
	 * 
	 * @return the type qualifier
	 */
	public int getPrecision() throws DLIException {
		if (this.fieldType != DLITypeInfo.ZONEDDECIMAL && this.fieldType != DLITypeInfo.PACKEDDECIMAL) {
			Object[] inserts = { "com.ibm.ims.base.DLITypeInfo.getPrecision()", this.getDataTypeName(this.fieldType),
					this.fieldName };
			throw new DLIException(IMSErrorMessages.getIMSBundle().getString("DATA_TYPE_NOT_BIGDECIMAL", inserts));
		}

		int precision = 0;

		int index = -1;
		index = this.expandedString.indexOf('P');
		if (index == -1) {
			index = this.expandedString.indexOf('p');
		}

		int count = 0;
		if (index != -1) {
			// determine the number of Ps in the PICTURE string
			int length = expandedString.length();
			for (int i = 0; i < length; i++) {
				if (String.valueOf(expandedString.charAt(i)).equalsIgnoreCase("P")) {
					count++;
				}
			}
		}

		precision = count + this.numberDigits;

		return precision;
	}

	private String getDataTypeName(int fieldType) {
		switch (fieldType) {
		case DLITypeInfo.CHAR:
			return "CHAR";
		case DLITypeInfo.BIT:
			return "BIT";
		case DLITypeInfo.BLOB:
			return "BLOB";
		case DLITypeInfo.VARCHAR:
			return "VARCHAR";
		case DLITypeInfo.DBCS:
			return "DBCS";
		case DLITypeInfo.DATE:
			return "DATE";
		case DLITypeInfo.TIME:
			return "TIME";
		case DLITypeInfo.TIMESTAMP:
			return "TIMESTAMP";
		case DLITypeInfo.TYPELIST:
			return "TYPELIST";
		case DLITypeInfo.CLOB:
			return "CLOB";
		case DLITypeInfo.TINYINT:
			return "TINYINT";
		case DLITypeInfo.SMALLINT:
			return "SMALLINT";
		case DLITypeInfo.INTEGER:
			return "INTEGER";
		case DLITypeInfo.BIGINT:
			return "BIGINT";
		case DLITypeInfo.UTINYINT:
			return "UTINYINT";
		case DLITypeInfo.USMALLINT:
			return "USMALLINT";
		case DLITypeInfo.UINTEGER:
			return "UINTEGER";
		case DLITypeInfo.UBIGINT:
			return "UBIGINT";
		case DLITypeInfo.DOUBLE:
			return "DOUBLE";
		case DLITypeInfo.FLOAT:
			return "FLOAT";
		case DLITypeInfo.PACKEDDECIMAL:
			return "PACKEDDECIMAL";
		case DLITypeInfo.ZONEDDECIMAL:
			return "ZONEDDECIMAL";
		case DLITypeInfo.SECONDARY_INDEX:
			return "SECONDARY_INDEX";
		default:
			return "UNKNOWN FIELD TYPE";
		}
	}

	/**
	 * Indicates whether this type is signed.
	 * 
	 */
	public boolean isSigned() {

		switch (fieldType) {
		case DLITypeInfo.TINYINT:
		case DLITypeInfo.SMALLINT:
		case DLITypeInfo.INTEGER:
		case DLITypeInfo.BIGINT:
		case DLITypeInfo.DOUBLE:
		case DLITypeInfo.FLOAT:
			return true;
		case DLITypeInfo.PACKEDDECIMAL:
		case DLITypeInfo.ZONEDDECIMAL:
			return this.signed;
		default:
			return false;
		}
	}
}
