package com.ibm.ims.db.cci;

/* (c) Copyright International Business Machines Corporation 2008. All rights reserved. */

import java.util.HashMap;
import java.util.Iterator;

import javax.resource.ResourceException;

/**
 * The <code>SSAListHelper</code> class is used to construct a segment search argument (SSA) qualification statement
 * string. You can set DL/I command codes for the SSAs, with the exception of lock classes and the C command code.
 * 
 * <p>
 * The following code example shows how to set the SSA qualification statement string using the
 * <code>SSAListHelper</code> to return return the last patient admitted to all of the  
 * wards with more than five doctors and less than three nurses in hospital "ALEXANDRIA".* <last occurance"):
 * 
 * <pre>
 * SSAListHelper sh = new SSAListHelper();
 * sh.addInitialQualification(&quot;Hospital&quot;, &quot;HospName&quot;, SSAListHelper.EQUALS, &quot;ALEXANDRIA&quot;);
 * sh.appendQualification(&quot;Ward&quot;, SSAListHelper.AND, &quot;Doccount&quot;, SSAListHelper.GREATER_THAN, 5);
 * sh.appendQualification(&quot;Ward&quot;, SSAListHelper.OR, &quot;Nurcount&quot;, SSAListHelper.LESS_THAN, 3);
 * sh.addCommandCode(&quot;Patient&quot;, SSAListHelper.CC_L);
 **iSpec.setSSAList(sh.toString());
 * </pre>
 */
public class SSAListHelper {

	HashMap<String, SSAHelper> ssas = new HashMap<String, SSAHelper>();

	/**
	 * Constant indicating the relational operator for equals (=) in an SSA qualification statement
	 */
	public static final short EQUALS = (short) 0xC5D8; /* EQ */

	/**
	 * Constant indicating the relational operator for greater than or equal to (>=) in an SSA qualification statement
	 */
	public static final short GREATER_OR_EQUAL = (short) 0xC7C5; /* GE */

	/**
	 * Constant indicating the relational operator for less than or equal to (<=) in an SSA qualification statement
	 */
	public static final short LESS_OR_EQUAL = (short) 0xD3C5; /* LE */

	/**
	 * Constant indicating the relational operator for greater than (>) in an SSA qualification statement
	 */
	public static final short GREATER_THAN = (short) 0xC7E3; /* GT */

	/**
	 * Constant indicating the relational operator for less than (<) in an SSA qualification statement
	 */
	public static final short LESS_THAN = (short) 0xD3E3; /* LT */

	/**
	 * Constant indicating the relational operator for not equal to (!=) in an SSA qualification statement
	 */
	public static final short NOT_EQUAL = (short) 0xD5C5; /* NE */

	// Boolean Operator Constants

	/**
	 * Constant indicating the boolean operator <code>AND</code> in an SSA qualification statement
	 */
	public static final byte AND = (byte) 0x50; /* & */

	/**
	 * Constant indicating the boolean operator <code>OR</code> in an SSA qualification statement
	 */
	public static final byte OR = (byte) 0x4E; /* + */

	/**
	 * Constant indicating the boolean operator <code>INDEPENDENT AND</code> in an SSA qualification statement
	 */
	public static final byte INDEPENDENT_AND = (byte) 0x7B; /* # */

	// Command Code Constants

	/**
	 * Constant indicating the C command code (concatenated key)
	 */
	static final byte CC_C = (byte) 0xC3; /* C */

	/**
	 * Constant indicating the D command code (path call)
	 */
	public static final byte CC_D = (byte) 0xC4; /* D */

	/**
	 * Constant indicating the L command code (last occurrence)
	 */
	public static final byte CC_L = (byte) 0xD3; /* L */


	// TODO
	// The command codes F, N, P, U, V & Q are not supported
	/**
	 * Constant indicating the F command code (first occurrence)
	 */
	static final byte CC_F = (byte) 0xC6; /* F */

	/**
	 * Constant indicating the N command code (path call ignore)
	 */
	static final byte CC_N = (byte) 0xD5; /* N */

	/**
	 * Constant indicating the P command code (set parentage)
	 */
	static final byte CC_P = (byte) 0xD7; /* P */

	/**
	 * Constant indicating the U command code (maintain position at this level)
	 */
	static final byte CC_U = (byte) 0xE4; /* U */

	/**
	 * Constant indicating the V command code (maintain position at this level and all superior levels)
	 */
	static final byte CC_V = (byte) 0xE5; /* V */

	static final byte CC_Q = (byte) 0xD8; /* Q */

	/**
	 * Add a command code to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code>
	 * @param commandCode
	 *            the command code to add
	 * @throws ResourceException
	 *             if an error occurs during processing
	 */
	public void addCommandCode(String segmentName, byte commandCode) throws ResourceException {
		if (segmentName == null) {
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("SEG_NAME_NULL"));
		}

		segmentName = segmentName.toUpperCase();

		SSAHelper ssa = null;
		if (ssas.containsKey(segmentName)) {
			ssa = ssas.get(segmentName);
		}

		if (ssa == null) {
			ssa = new SSAHelper();
		}

		ssa.addCommandCode(commandCode);

		ssas.put(segmentName, ssa);
	}

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the <code>SSAList</code> constants
	 * @param value
	 *            the value of the field as a String
	 * @throws ResourceException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, String value)
			throws ResourceException {

		if (segmentName == null) {
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("SEG_NAME_NULL"));
		}

		if (fieldName == null) {
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("FLD_NAME_NULL"));
		}

		if (value == null) {
			Object[] inserts = {
				value
			};
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_SSA_QUAL_VALUE", inserts));
		}

		segmentName = segmentName.toUpperCase();

		fieldName = fieldName.toUpperCase();

		StringBuffer ssaQual = new StringBuffer(fieldName);

		switch (relationalOp) {
			case SSAListHelper.EQUALS:
				ssaQual.append(" = ");
				break;
			case SSAListHelper.NOT_EQUAL:
				ssaQual.append(" != ");
				break;
			case SSAListHelper.GREATER_OR_EQUAL:
				ssaQual.append(" >= ");
				break;
			case SSAListHelper.GREATER_THAN:
				ssaQual.append("  > ");
				break;
			case SSAListHelper.LESS_OR_EQUAL:
				ssaQual.append(" <= ");
				break;
			case SSAListHelper.LESS_THAN:
				ssaQual.append(" < ");
				break;
			default:
				Object[] inserts = {
					relationalOp
				};
				throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_RELATIONAL_OP", inserts));
		}

		ssaQual.append("'").append(value).append("'");

		SSAHelper ssa = null;

		if (ssas.containsKey(segmentName)) {
			ssa = ssas.get(segmentName);
		}

		if (ssa == null) {
			ssa = new SSAHelper();
		}

		ssa.addInitialQualificationStatement(ssaQual);

		ssas.put(segmentName, ssa);

	}

	/**
	 * Append a qualifiation statement to the specified SSA in the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to the previous - one of the
	 *            <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the <code>SSAList</code> constants
	 * @param value
	 *            the value of the field as a String
	 * @throws ResourceException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			String value) throws ResourceException {

		if (segmentName == null) {
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("SEG_NAME_NULL"));
		}

		if (fieldName == null) {
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("FLD_NAME_NULL"));
		}

		if (value == null) {
			Object[] inserts = {
				value
			};
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_SSA_QUAL_VALUE", inserts));
		}

		StringBuffer ssaQual = new StringBuffer();

		segmentName = segmentName.toUpperCase();

		fieldName = fieldName.toUpperCase();

		switch (booleanOp) {
			case SSAListHelper.AND:
				ssaQual.append(" & ");
				break;
			case SSAListHelper.OR:
				ssaQual.append(" | ");
				break;
			case SSAListHelper.INDEPENDENT_AND:
				ssaQual.append(" # ");
				break;
			default:
				Object[] inserts = {
					booleanOp
				};
				throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_BOOLEAN_OP", inserts));
		}

		ssaQual.append(fieldName);

		switch (relationalOp) {
			case SSAListHelper.EQUALS:
				ssaQual.append("=");
				break;
			case SSAListHelper.NOT_EQUAL:
				ssaQual.append(" != ");
				break;
			case SSAListHelper.GREATER_OR_EQUAL:
				ssaQual.append(" >= ");
				break;
			case SSAListHelper.GREATER_THAN:
				ssaQual.append(" > ");
				break;
			case SSAListHelper.LESS_OR_EQUAL:
				ssaQual.append(" <= ");
				break;
			case SSAListHelper.LESS_THAN:
				ssaQual.append(" < ");
				break;
			default:
				Object[] inserts = {
					relationalOp
				};
				throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_RELATIONAL_OP", inserts));
		}

		ssaQual.append("'").append(value).append("'");

		SSAHelper ssa = null;
		if (ssas.containsKey(segmentName)) {
			ssa = ssas.get(segmentName);
		}

		if (ssa == null) {
			ssa = new SSAHelper();
		}

		ssa.appendQualificationStatement(ssaQual);

		ssas.put(segmentName, ssa);

	}

	/**
	 * Add an initial unqualified statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the qualification to
	 * @throws ResourceException
	 */
	public void addSSA(String segmentName) throws ResourceException {
		if (segmentName == null) {
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("SEG_NAME_NULL"));
		}

		segmentName = segmentName.toUpperCase();

		SSAHelper ssa = null;
		if (ssas.containsKey(segmentName)) {
			ssa = ssas.get(segmentName);
		}

		if (ssa != null) {
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("SSA_LEVEL_ALREADY_EXISTS"));
		}

		ssas.put(segmentName, null);
	}

	/**
	 * Returns the <code>SSAList</code> in the form of a string.
	 * 
	 * @return the ssaList <code>String</code>.
	 */
	public String toString() {
		StringBuffer ssaList = new StringBuffer();

		Iterator<String> iter = ssas.keySet().iterator();

		while (iter.hasNext()) {
			String segmentName = iter.next();
			ssaList.append(segmentName).append(" ");

			SSAHelper ssa = ssas.get(segmentName);

			if (ssa != null) {
				StringBuffer ccs = ssa.commandCodes;
				if (ccs.length() > 0) {
					ssaList.append("*");
					ssaList.append(ccs);
					ssaList.append(" ");
				}

				StringBuffer qStats = ssa.qualificationStatements;
				if (qStats != null && qStats.length() > 0) {
					ssaList.append("(");
					ssaList.append(qStats);
					ssaList.append(") ");
				}
			}
		}

		return ssaList.toString();
	}

	class SSAHelper {
		StringBuffer commandCodes = new StringBuffer();
		StringBuffer qualificationStatements = null;

		void addCommandCode(byte commandCode) throws ResourceException {
			Object[] inserts = new Object[1];
			switch (commandCode) {
				case SSAListHelper.CC_D:
					commandCodes.append("D");
					break;
				case SSAListHelper.CC_L:
					commandCodes.append("L");
					break;
				case SSAListHelper.CC_F:
					inserts[0] = "F";
					throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_CMD_CODE", inserts));
//					commandCodes.append("F");
//					break;
				case SSAListHelper.CC_N:
					inserts[0] = "N";
					throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_CMD_CODE", inserts));
//					commandCodes.append("N");
//					break;
				case SSAListHelper.CC_P:
					inserts[0] = "P";
					throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_CMD_CODE", inserts));
//					commandCodes.append("P");
//					break;
				case SSAListHelper.CC_U:
					inserts[0] = "U";
					throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_CMD_CODE", inserts));
//					commandCodes.append("U");
//					break;
				case SSAListHelper.CC_V:
					inserts[0] = "V";
					throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_CMD_CODE", inserts));
//					commandCodes.append("V");
//					break;
				case SSAListHelper.CC_Q:
					inserts[0] = "Q";
					throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_CMD_CODE", inserts));
//					commandCodes.append("Q");
//					break;
				case SSAListHelper.CC_C:
					inserts[0] = "C";
					throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_CMD_CODE", inserts));
//					commandCodes.append("C");
//					break;
				default:
					inserts[0] = commandCode;
					throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_CMD_CODE", inserts));
			}
		}

		void addInitialQualificationStatement(StringBuffer qualStat) {
			qualificationStatements = new StringBuffer(qualStat);
		}

		void appendQualificationStatement(StringBuffer qualStat) throws ResourceException {
			if (qualificationStatements == null) {
				throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("NEED_ADD_BEFORE_APPEND_SSA"));
			}
			qualificationStatements.append(qualStat);
		}

	}

}
