package com.ibm.ims.dli;

/**
 * Use this interface to issue the following DL/I calls programmatically from
 * your Java application when using type-2 connectivity and running in an IMS
 * Java Dependent Region (JDR).
 * 
 * @see PSB#prepareIMSRegionDLICall(DLICall.Function)
 * @see PSB#prepareIMSRegionDLICall(DLICallProperty)
 */
public interface DLICall {

	/**
	 * The constants that represent the IMS DL/I call functions.
	 * 
	 */
	public enum Function {

		/**
		 * 
		 * An Authorization (AUTH) call verifies each user's security
		 * authorization. It determines whether a user is authorized to access
		 * the resources specified on the AUTH call.
		 * <p>
		 * The following code example shows how to issue an AUTH call:
		 * <blockquote> <CODE><br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.AUTH);<br>
		 *   props.put(DLICallProperty.Property.CLASSNAME, DLICallProperty.Classname.FIELD);<br>
		 *   props.put(DLICallProperty.Property.RESOURCE, "abcd");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   AUTHCallResult result = (AUTHCallResult) dliCall.execute();<br>
		 *   short feedback = result.getFeedback());<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I AUTH call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_authcall.htm"
		 * target="_blank">AUTH call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		AUTH,

		/**
		 * 
		 * The Dequeue (DEQ) call is used to release a segment that is retrieved
		 * using the Q command code.
		 * <p>
		 * The following code example shows how to issue a DEQ call:
		 * <blockquote> <CODE> <br>
		 * try { <br>
		 *    props = new DLICallProperty(DLICall.Function.DEQ);<br>
		 *    props.put(DLICallProperty.Property.LOCK_CLASS, DLICallProperty.LockClass.A);
		 *    DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *    DLICallResult result = dliCall.execute();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I DEQ call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_deqcall.htm"
		 * target="_blank">DEQ call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		DEQ,

		/**
		 * 
		 * A Get Message (GMSG) call is used in an automated operator (AO)
		 * application program to retrieve a message from AO exit routine
		 * DFSAOE00.
		 * <p>
		 * The following code example shows how to issue a GMSG call:
		 * <blockquote> <CODE> <br>
		 * try { <br>
		 *   props = new DLICallProperty(DLICall.Function.GMSG);<br>
		 *   props.put(DLICallProperty.Property.AIB_SUB_FUNCTION_CODE, DLICallProperty.Subfunction.WAITAOI);<br>
		 *   props.put(DLICallProperty.Property.AOI_TOKEN, "XYZ");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   GMSGCallResult result = (GMSGCallResult) dliCall.execute();<br>
		 *   String message = result.getMessage();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I GMSG call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_gmsgcalltm.htm"
		 * target="_blank">GMSG call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		GMSG,

		/**
		 * 
		 * An Issue Command (ICMD) call lets an automated operator (AO)
		 * application program issue an IMS command and retrieve the first
		 * command response segment.
		 * <p>
		 * The following code example shows how to issue a ICMD call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.ICMD);<br>
		 *   props.put(DLICallProperty.Property.COMMAND, "/DISPLAY ACTIVE REGION.");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   ICMDCallResult result = (ICMDCallResult) dliCall.execute();<br>
		 *   String response = result.getResponseSegment();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I ICMD call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_icmdcalltm.htm"
		 * target="_blank">ICMD call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		ICMD,

		/**
		 * 
		 * An Initialize (INIT) call allows the application to receive data
		 * availability status codes by checking each DB PCB for data
		 * availability.
		 * <p>
		 * The following code example shows how to issue a INIT call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.INIT);<br>
		 *   props.put(DLICallProperty.Property.AIB_SUB_FUNCTION_CODE, DLICallProperty.Subfunction.DBQUERY);<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   INITCallResult result = (INITCallResult) dliCall.execute();<br>
		 *   Iterator&lt;DBPCB&gt; dbPCBs = result.getDBPCBs().iterator(); <br>
		 *   while (dbPCBs.hasNext()) {<br>
		 *      DBPCB dbpcb = dbPCBs.next();<br>
		 *      String dbName = dbpcb.getDBName();<br>
		 *      short sc = dbpcb.getStatusCode();<br> 
		 *   }<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I INIT call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_initcalltm.htm"
		 * target="_blank">INIT call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		INIT,

		/**
		 * 
		 * The Inquiry (INQY) call is used to request information regarding
		 * execution environment, destination type and status, and session
		 * status. INQY is valid only when using the AIBTDLI interface.
		 * <p>
		 * The following code example shows how to issue a INQY call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.INQY);<br>
		 *   props.put(DLICallProperty.Property.AIB_SUB_FUNCTION_CODE, DLICallProperty.Subfunction.FIND);<br>
		 *   props.put(DLICallProperty.Property.PCB_NAME, "PCB1");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   INQYCallResult result = (INQYCallResult) dliCall.execute();<br>
		 *   DBPCB dbPCB = result.getDBPCBs().firstElement();<br>
		 *   String pcbName = dbPCB.getDBName();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I INQY call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_inqycall.htm"
		 * target="_blank">INQY call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		INQY,

		/**
		 * 
		 * The Field (FLD) call is used to access a field within a segment for
		 * MSDBs or DEDBs.
		 * <p>
		 * The following code example shows how to issue a INQY call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.FLD);<br>
		 *   props.put(Property.PCB_NAME, "PCB1");<br>
		 *   FSA fsa = props.getFSA();<br>
		 *   fsa.setFieldName("field1");<br>
		 *   fsa.setOpCode(FLD_FSA_OPCODE.ADD);<br>
		 *   fsa.setOperand("value1");<br>
		 *   props.put(Property.FSA, fsa);<br>
		 *   
		 *   DLICall call = psb.prepareIMSRegionDLICall(props);<br>
		 *   
		 *   FLDCallResult result = (FLDCallResult) call.execute();<br>
		 *   FLD_STATUSCODE statusCode = result.getStatusCode();<br>
		 *   if (statusCode != FLD_STATUSCODE.b) {<br> 
		 *   // Call failed<br> 
		 *   }<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <P>
		 * 
		 * For more details about the usage of the DL/I FLD call, see the topic
		 * <a href="
		 * http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_fldcall.htm"
		 * target="_blank">FLD call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		FLD,

		/**
		 * 
		 * The Log (LOG) call is used to send and write information to the IMS
		 * system log.
		 * <p>
		 * The following code example shows how to issue a LOG call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.LOG);<br>
		 *   props.put(DLICallProperty.Property.LOG_CODE_BYTE, (byte) 0xA0);<br>
		 *   props.put(DLICallProperty.Property.LOG_TEXT, "LOGGING FROM JAVA");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   DLICallResult result = dliCall.execute();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I LOG call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_logcalltm.htm"
		 * target="_blank">LOG call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		LOG,

		/**
		 * 
		 * A Retrieve Command (RCMD) call enables an automated operator (AO)
		 * application program retrieve the second and subsequent command
		 * response segments after an ICMD call.
		 * <p>
		 * The following code example shows how to issue a RCMD call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(DLICall.Function.RCMD);<br>
		 *   RCMDCallResult rcmdResult = (RCMDCallResult) dliCall.execute();<br>
		 *   String response = rcmdResult.getResponseSegment();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I RCMD call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_rcmdcall.htm"
		 * target="_blank">RCMD call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		RCMD,

		/**
		 * 
		 * The Release Locks (RLSE) call is used to release all locks held for
		 * unmodified data.
		 * <p>
		 * The following code example shows how to issue a RLSE call:
		 * <blockquote> <CODE> <br>
		 * try {
		 *   props = new DLICallProperty(DLICall.Function.RLSE);<br>
		 *   props.put(DLICallProperty.Property.PCB_NAME, "PCB1");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   dliCall.execute();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I RLSE call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_rlsecall.htm"
		 * target="_blank">RLSE call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		RLSE,

		/**
		 * 
		 * The Roll (ROLL) call backs out output messages sent by a
		 * conversational application program and terminates the conversation.
		 * <p>
		 * The following code example shows how to issue a ROLL call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(DLICall.Function.ROLL);<br>
		 *   dliCall.execute();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I ROLL call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_rollcalltm.htm"
		 * target="_blank">ROLL call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		ROLL,

		/**
		 * 
		 * The Roll Back to SETS/SETU (ROLS) call returns message queue
		 * positions to sync points established by the SETS/SETU call.
		 * <p>
		 * The following code example shows how to issue a ROLS call:
		 * <blockquote> <CODE> <br>
		 * try {
		 *   props = new DLICallProperty(DLICall.Function.ROLS);<br>
		 *   props.put(DLICallProperty.Property.TOKEN, "1234");<br>
		 *   props.put(DLICallProperty.Property.AIB_OUTPUT_AREA_LENGTH, 100);<br>
		 *   props.put(DLICallProperty.Property.PCB_NAME, "PCB1");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   ROLSCallResult rolsResult = (ROLSCallResult) dliCall.execute();<br>
		 *   String savedData = rolsResult.getData();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I ROLS call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_rolscalltm.htm"
		 * target="_blank">ROLS call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		ROLS,

		/**
		 * 
		 * The Set Backout Point (SETS) call is used to set an intermediate
		 * backout point or to cancel all existing backout points.
		 * <p>
		 * The following code example shows how to issue a SETS call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.SETS);<br>
		 *   props.put(DLICallProperty.Property.TOKEN, "1234");<br>
		 *   props.put(DLICallProperty.Property.SETDATA, "ZYX");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   dliCall.execute();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I SETS call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_setsetucalltm.htm"
		 * target="_blank">SETS call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		SETS,

		/**
		 * 
		 * The Set Unconditional (SETU) call operates like the SETS call except
		 * that the SETU call isn't rejected if unsupported PCBs are in the PSB
		 * or if the program uses an external subsystem.
		 * <p>
		 * The following code example shows how to issue a SETU call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.SETU);<br>
		 *   props.put(DLICallProperty.Property.TOKEN, "1234");<br>
		 *   props.put(DLICallProperty.Property.SETDATA, "XYZ");<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   dliCall.execute();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I SETU call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_setsetucalltm.htm"
		 * target="_blank">SETU call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		SETU,

		/**
		 * 
		 * The Statistics (STAT) call is used in a CICS, IMS online, or batch
		 * program to obtain database statistics that might be useful for
		 * performance monitoring.
		 * <p>
		 * The following code example shows how to issue a STAT call:
		 * <blockquote> <CODE> <br>
		 * try {<br>
		 *   props = new DLICallProperty(DLICall.Function.STAT);<br>
		 *   props.put(DLICallProperty.Property.PCB_NAME, "PCB1");<br>
		 *   props.put(DLICallProperty.Property.STAT_TYPE, DLICallProperty.STAT_Option.TYPE_DBAS);<br>
		 *   props.put(DLICallProperty.Property.STAT_FORMAT, DLICallProperty.STAT_Option.FORMAT_F);<br>
		 *   props.put(DLICallProperty.Property.CALL_TYPE, DLICallProperty.STAT_Option.STAT_NORMAL);<br>
		 *   DLICall dliCall = psb.prepareIMSRegionDLICall(props);<br>
		 *   STATCallResult result = (STATCallResult) dliCall.execute();<br>
		 *   String stats = result.getStatistics();<br>
		 * } catch (DLIException e) {<br>
		 *   e.printStackTrace();<br>
		 * }<br>
		 * </CODE> </blockquote>
		 * <p>
		 * For more details about the usage of the DL/I STAT call, see the topic
		 * <a href=
		 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_statcall.htm"
		 * target="_blank">STAT call</a> in the IMS Application Programming APIs
		 * reference documentation.
		 */
		STAT
	}

	/**
	 * Issues the DL/I call to IMS.
	 * 
	 * @return a <code>DLICallResult</code> object instance that contains the
	 *         data produced by the given DLI call; <code>null</code> if the DLI
	 *         call failed.
	 * 
	 * @throws DLIException
	 *             when the DL/I call fails.
	 */
	public DLICallResult execute() throws DLIException;

}
