package com.ibm.ims.dli;

import java.nio.ByteBuffer;

/**
 * Object that represents a connection to a GSAM PCB.
 *
 */
public interface GSAMPCB {

	/**
	 * Opens a connection to a GSAM dataset.
	 * 
	 */
	public void open();

	/**
	 * Close the connection to a GSAM dataset.
	 * 
	 * @exception DLIException
	 *                if an error occurs while closing the GSAM dataset.
	 */
	public void close();

	/**
	 * Retrieves a record in a GSAM database using random access.
	 * 
	 * @param recordSearchArg
	 *            the record search argument (<code>RSA</code>) object that
	 *            contains the address of a unique GSAM database record.
	 * @return a <code>Path</code> object that contains the GSAM database record
	 *         data or <code>null</code> if no records are found
	 * @exception DLIException
	 *                if a GSAM database error occurs.
	 */
	public Path getUnique(RSA recordSearchArg) throws DLIException;

	
	/**
	 * Retrieves a record in a GSAM database using random access.
	 * 
	 * @param recordSearchArg the record search argument (<code>RSA</code>) object
	 *                        that contains the address of a unique GSAM database
	 *                        record.
	 * @param outputBuffer    The output buffer (<code>ByteBuffer</code>) object
	 *                        that will contain the GSAM database record.
	 * @throws DLIException if a GSAM database error occurs.
	 */
	public void getUnique(RSA recordSearchArg, ByteBuffer outputBuffer) throws DLIException;

	
	/**
	 * Retrieves a record in a GSAM database using sequential access. After
	 * calling <code>getNext</code>, the cursor moves to the next record in the
	 * database.
	 * 
	 * @return the <code>Path</code> object that contains the GSAM database
	 *         record data or <code>null</code> if no records are found
	 * @exception DLIException
	 *                if a GSAM database error occurs.
	 */
	public Path getNext() throws DLIException;
	
	/**
	 * Retrieves a record in a GSAM database using sequential access. After calling
	 * <code>getNext</code>, the cursor moves to the next record in the database.
	 * 
	 * @param recordSearchArg the record search argument (<code>RSA</code>) object
	 *                        that contains the address of a unique GSAM database
	 *                        record.
	 * @param outputBuffer    The output buffer (<code>ByteBuffer</code>) object
	 *                        that will contain the GSAM database record.
	 * @throws DLIException if a GSAM database error occurs.
	 */
	public void getNext(ByteBuffer output) throws DLIException;

	/**
	 * Inserts a new record into a GSAM database. New records are appended to
	 * the end of the database.
	 * 
	 * @param gsamRecord
	 *            the <code>Path</code> object that contains the GSAM database
	 *            record data to be inserted.
	 * @exception DLIException
	 *                if a GSAM database error occurs.
	 */
	public RSA insert(Path gsamRecord) throws DLIException;

	/**
	 * Inserts a new record into a GSAM database. New records are appended to the
	 * end of the database.
	 * 
	 * @param gsamRecordBuffer The input buffer (<code>ByteBuffer</code>) object
	 *                         contains the GSAM database record data to be
	 *                         inserted. The <code>ByteBuffer</code> limit will
	 *                         determine the maximum number of bytes to be inserted.
	 * @exception DLIException if a GSAM database error occurs.
	 */
	public RSA insert(ByteBuffer gsamRecordBuffer) throws DLIException;

	/**
	 * Retrieves the application interface block (<code>AIB</code>) object for
	 * the GSAM PCB.
	 * 
	 * @return the Application Interface Block (AIB).
	 */
	public AIB getAIB();

	/**
	 * Retrieves a <code>Path</code> that contains the GSAM record used for an
	 * insert operation. Use this method together with the
	 * {@link GSAMPCB#insert(Path)} method to insert new records in the GSAM
	 * database.
	 * <p>
	 * 
	 * @see GSAMPCB#insert(Path)
	 * @return a <code>Path</code> object that contains the GSAM database record
	 *         specified as input
	 */
	public Path getPathForInsert();
}
