package com.ibm.ims.dli;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;

import com.ibm.ims.dli.t2.INQYCallImpl;
import com.ibm.ims.dli.t2.T2ErrorMessages;

/**
 * Object that represents the output from an Inquiry (INQY) call that is issued
 * with the MSGINFO subfunction.
 * <p>
 * For more details about the values returned by the <code>MessageInfo</code>
 * methods, see the topic <a href=
 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_inqycall.htm"
 * target="_blank">INQY call</a> in the IMS Application Programming APIs
 * reference documentation.
 * 
 */
public class MessageInfo {

	private String imsID;
	private StringBuilder versionNumber = new StringBuilder();
	private INQYCallImpl inqy;
	private long ioBufferAddress = 0;
	private final static Charset EBCDIC = Charset.forName("CP1047");
	
	private StringBuffer info = null;
	byte[] ioArea;

	/**
	 * <code>MessageInfo</code> constructor.
	 * 
	 * @param ioArea
	 *            a byte array that represents the data output area.
	 * 
	 * @throws UnsupportedEncodingException
	 */
	public MessageInfo(byte[] ioArea, long ioBufferAddress) throws UnsupportedEncodingException {
		
		this.ioArea = ioArea;
		this.info = new StringBuffer(new String(ioArea, "Cp1047"));
		this.ioBufferAddress = ioBufferAddress;
		
	}
	
	public MessageInfo(byte[] ioArea, INQYCallImpl inqyCallImpl) throws UnsupportedEncodingException {
		
		this.ioArea = ioArea;
		this.info = new StringBuffer(new String(ioArea, "Cp1047"));
		
		this.inqy =  inqyCallImpl;
		this.ioBufferAddress = this.inqy.getIOBufferAddress();
	}

	/**
	 * Retrieves the output response version.
	 * 
	 * @return the version number
	 */
	public String getVersionNumber() {
		//Ensures string is empty before appending
		versionNumber.setLength(0);
		
		for (int i = 0; i < 4; i++) {
			versionNumber.append(ioArea[i]);
		}
		
		return versionNumber.toString();
	}
	
	/**
	 * Retrieves the IMS identifier from which the input message originated.
	 * 
	 * @return IMS identifier.
	 */
	public String getIMSID() {
		imsID = info.substring(4, 8).trim();
		return imsID;
	}
	
	/**
	 * Retrieves the distributed network user ID.
	 * 
	 * @return user ID address
	 * @throws DLIException 
	 */
	public String getUserID() throws DLIException {
		int startOffset = 12;
		String userID = readAddressDataFromIO(startOffset);

		return userID;
	}
	
	/**
	 * Retrieves the distributed network session ID.
	 * 
	 * @return session ID
	 * @throws DLIException 
	 */
	public String getSessionID() throws DLIException {
		int startOffset = 16;
		String sessionID = readAddressDataFromIO(startOffset);

		return sessionID;
	}
	
	/**
	 * Gets the reason code produced by the MSGINFO INQY DLI call.
	 * 
	 * @return the reason code of the INQY DLI call
	 */
	public int getReasonCode() {
		return this.inqy.getAIB().getReasonCode();
	}
	
	/**
	 * Gets the reason code produced by the MSGINFO INQY DLI call
	 * as a hexadecimal string.
	 * 
	 * @return the reason code of the INQY DLI call
	 */
	public String getReasonCodeHex() {
		return this.inqy.getAIB().getReasonCodeHex();
	}
	
	/**
	 * Gets the return code produced by the MSGINFO INQY DLI call.
	 * 
	 * @return the return code of the INQY DLI call
	 */
	public int getReturnCode() {
		return this.inqy.getAIB().getReturnCode();
	}
	
	/**
	 * Gets the return code produced by the MSGINFO INQY DLI call
	 * as a hexadecimal string.
	 * 
	 * @return the return code of the INQY DLI call
	 */
	public String getReturnCodeHex() {
		return this.inqy.getAIB().getReturnCodeHex();
	}
	
	/**
	 * Gets the IMS status code of the INQY call from the IOPCB.
	 * 
	 * @return the IMS status code of the INQY DLI call.
	 */
	public int getStatusCode() {
		return this.inqy.getAIB().getIOPCB().getStatusCode();
	}
	
	/**
	 * Gets the IMS status code of the INQY call from the IOPCB
	 * as a character string.
	 * 
	 * @return the IMS status code of the INQY DLI call.
	 */
	public String getStatusCodeChars() {
		return this.inqy.getAIB().getIOPCB().getStatusCodeChars();
	}
	
	/**
	 * This method retrieves the address at the given offset from the IO Area
	 * and then subtracts the ioBufferAddress from the retrieved address to 
	 * get the offset of where the data is located and retrieves the data
	 * 
	 * @param offset
	 * @return
	 * @throws DLIException
	 */
	private String readAddressDataFromIO(int offset) throws DLIException {
		byte[] dataByteArray;
		String data;
		if (ioBufferAddress == 0) {
			return null;
		}

		ByteBuffer buffer = ByteBuffer.wrap(ioArea);
		buffer.position(offset);

		int address = buffer.getInt();

		if (address == 0) {
			return null;
		}

		offset = (int) (address - ioBufferAddress);

		int llLength = 2;
		if (offset < 0 || offset >= buffer.capacity() - llLength) {
			Object[] inserts = {offset, 2, buffer.capacity(), ioBufferAddress};
			throw new DLIException(T2ErrorMessages.getIMSBundle().getString("BUFFER_OUT_OF_BOUNDS", inserts));
		}

		buffer.position(offset);
		int dataLength = buffer.getShort();

		//check to see if offset + LL length + dataLength is larger than the buffer capacity
		if (offset + llLength + dataLength > buffer.capacity()) {
			Object[] inserts = {offset, dataLength, buffer.capacity(), ioBufferAddress};
			throw new DLIException(T2ErrorMessages.getIMSBundle().getString("BUFFER_OUT_OF_BOUNDS", inserts));
		}

		dataByteArray = new byte[dataLength];
		buffer.get(dataByteArray, 0 , dataLength);

		data = new String(dataByteArray, EBCDIC);
		return data;
	}
}
