package com.ibm.ims.dli;

/*
 * (c) Copyright International Business Machines Corporation 2008. All rights
 * reserved.
 */
import java.io.PrintWriter;

import com.ibm.ims.dli.DLICall.Function;

/**
 * A Program Specification Block (PSB) contains one or more PCB objects. Use
 * this interface to obtain a handle to any PCB contained in the PSB. The PCB
 * handle is used to access the particular database referenced by the PCB.
 * <p>
 * The code fragment below shows how to create and use <code>PSB</code> to
 * obtain a PCB handle. First, create an instance of
 * <code>IMSConnectionSpec</code> using the
 * <code>IMSConnectionSpecFactory</code>. Next, pass the connection request
 * properties to the <code>PSBFactory</code> to create the <code>PSB</code>.
 * When the <code>PSB</code> is created, a connection is made to the database.
 * Finally, call the <code>getPCB</code> method, using the PCB name as a
 * parameter, to obtain a <code>PCB</code> handle to access the database.
 * <blockquote>
 * 
 * <pre>
 * IMSConnectionSpec connSpec = IMSConnectionSpecFactory.createIMSConnectionSpec();
 * connSpec.setDatastoreName(&quot;SYS1&quot;);
 * connSpec.setDatastoreServer(&quot;9.876.543.21&quot;);
 * connSpec.setPortNumber(50);
 * connSpec.setDatabaseName(&quot;class://testdb.jdbo.HospitalDatabaseView&quot;);
 * connSpec.setDriverType(IMSConnectionSpec.DRIVER_TYPE_4);
 * connSpec.setUser(&quot;usr&quot;);
 * connSpec.setPassword(&quot;usrpwd&quot;);
 * PSB psb = PSBFactory.createPSB(connSpec);
 * PCB pcb = psb.getPCB(&quot;Pcb01&quot;);
 * </pre>
 * 
 * </blockquote>
 * <p>
 * <code>PSB</code> provides support for local transaction processing. A local
 * transaction consists of a unit of work with several units of recovery. In the
 * IMS Universal DL/I driver, the local transaction is scoped to the PSB. No
 * explicit call is needed to begin a local transaction. A unit of recovery
 * consists of a sequence of DL/I calls followed by either a call to commit the
 * current unit of recovery with {@link PSB#commit}, or a call to undo the
 * changes up to the last unit of recovery with {@link PSB#rollback}. If
 * <code>commit</code> or <code>rollback</code> is called and the PSB is not
 * deallocated, a new unit of recovery is started. By default, the local
 * transaction ends when the PSB is deallocated. To persist changes made to the
 * database, your application must always call <code>commit</code> prior to
 * PSB.deallocate or PSB.close, otherwise the changes are rolled back to the
 * last point <code>commit</code> was called.
 * </p>
 * <p>
 * The following code fragment illustrates how to create and use
 * <code>PSB</code> in a local transaction. <blockquote>
 * 
 * <pre>
 *   PSB psb = PSBFactory.createPSB(imsConnectionSpec);
 *   PCB pcb = psb.getPCB(&quot;PCb01&quot;);
 *    
 *   // perform an insert operation
 *   SSAList ssa = pcb.getSSAList(&quot;HOSPITAL&quot;);
 *   Path path = ssa.getPathForInsert(&quot;HOSPITAL&quot;);
 *   path.setString(&quot;HOSPCODE&quot;, &quot;R1210020000A&quot;);
 *   path.setString(&quot;HOSPNAME&quot;, &quot;SANTA TERESA&quot;);
 *   pcb.insert(path);
 *   
 *   // commit the changes or use psb.rollback() to undo the insert
 *   psb.commit(); 
 *   ...
 *   psb.deallocate();
 *   psb.close();
 * </pre>
 * 
 * </blockquote>
 * 
 * @see PCB
 * @see PSBFactory#createPSB(IMSConnectionSpec)
 */
public interface PSB {

	/**
	 * FINE indicates basic tracing.
	 */
	public static final int FINE = 500;

	/**
	 * FINER indicates fairly detailed tracing.
	 */
	public static final int FINER = 400;

	/**
	 * FINEST indicates highly detailed tracing.
	 */
	public static final int FINEST = 300;

	/**
	 * ALL indicates that all messages should be logged.
	 */
	public static final int ALL = 1;

	/**
	 * OFF indicates that no messages should be logged.
	 */
	public static final int OFF = 0;

	/**
	 * This method returns a <code>PCB</code> object that represents an actual
	 * PCB contained in the PSB. Use <code>PCB</code> to access the database it
	 * references.
	 * 
	 * @param pcbName
	 *            the name of a PCB within the PSB
	 * @return the specified <code>PCB</code> object
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public PCB getPCB(String pcbName) throws DLIException;

	/**
	 * This method returns a <code>GSAMPCB</code> object that represents an
	 * actual GSAM PCB contained in the PSB. Use <code>GSAMPCB</code> to access
	 * the GSAM database it references.
	 * 
	 * @param pcbName
	 *            the name of a GSAM PCB within the PSB
	 * @return the specified <code>GSAMPCB</code> object
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public GSAMPCB getGSAMPCB(String pcbName) throws DLIException;

	/**
	 * This method returns a <code>PCB</code> object that represents an IOPCB
	 * Use <code>PCB</code> to access the database it references.
	 * 
	 * @return the specified <code>PCB</code> object
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public PCB getIOPCB() throws DLIException;

	/**
	 * Returns the name of the PSB as defined to IMS
	 * 
	 * @return the name of the PSB
	 */
	public String getIMSName();

	/**
	 * Allocates the <code>PSB</code>, making all databases referenced by the
	 * contained PCBs accessible.
	 * <p>
	 * <b>Note:</b> This method will be called internally by the libraries. It
	 * is not necessary for client applications to call this method.
	 * 
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void allocate() throws DLIException;

	/**
	 * Deallocates the <code>PSB</code> and free up all IMS resources associated
	 * with the PSB.
	 * 
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void deallocate() throws DLIException;

	/**
	 * Closes any physical resources associated with this <code>PSB</code>. If
	 * the actual PSB was never deallocated, this <code>close</code> method will
	 * also deallocate the PSB.
	 * 
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void close() throws DLIException;

	/**
	 * Opens any physical resources associated with this <code>PSB</code>.
	 * Calling the <code>PSB.open</code> method from a distributed environment
	 * will create a socket connection and allocate the PSB. Calling this method
	 * from local non-IMS environments will simply allocate the PSB.
	 * 
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void open() throws DLIException;

	/**
	 * Indicates whether or not the <code>PSB</code> is currently allocated.
	 * 
	 * @return true if allocated, false otherwise
	 */
	public boolean isAllocated();

	/**
	 * Indicates whether or not the <code>PSB</code> is currently open (has a
	 * physical connection to the host system).
	 * 
	 * @return true if open, false otherwise
	 */
	public boolean isOpen();

	/**
	 * Sets the log writer for this PSB instance. All tracing will be logged
	 * using this PrintWriter.
	 * 
	 * @param writer
	 *            the PrintWriter to use for logging
	 */
	public void setLogWriter(PrintWriter writer);

	/**
	 * Sets the trace level for this PSB instance.
	 * 
	 * @param level
	 *            the trace level
	 */
	public void setLogWriterTraceLevel(int level);

	/** ******************************************** */
	/* Basic non-XA local tx support */
	/** ******************************************** */
	/**
	 * Commits the current local transaction and releases all locks and database
	 * position held by the underlying EIS instance.
	 * 
	 * @exception NotSupportedException
	 *                if the operation is not supported
	 * @exception DLIException
	 *                if the operation fails
	 */
	public void commit() throws DLIException, NotSupportedException;

	/**
	 * Rolls back the current local transaction
	 * 
	 * @exception NotSupportedException
	 *                if the operation is not supported
	 * @exception DLIException
	 *                if the operation fails
	 */
	public void rollback() throws DLIException, NotSupportedException;

	/**
	 * Creates an instance of <code>DLICall</code> for programmatically issuing
	 * DL/I calls to IMS from your Java application.
	 * 
	 * @param properties
	 *            a <code>DLICallProperty</code> instance that represents input
	 *            parameters for the DL/I call. See {@link DLICall} for examples
	 *            of how to set the properties..
	 * @return DLICall the <code>DLICall</code> instance
	 * @throws DLIException
	 * @see DLICall
	 */
	public DLICall prepareIMSRegionDLICall(DLICallProperty properties) throws DLIException;

	/**
	 * Creates an instance of <code>DLICall</code> for programmatically issuing
	 * DL/I calls to IMS from your Java application.
	 * 
	 * @param function
	 *            the function name of the DL/I call. Use the constants defined
	 *            in <code>DLICall</code> to specify this parameter.
	 * @return DLICall the <code>DLICall</code> instance
	 * @see DLICall
	 */
	public DLICall prepareIMSRegionDLICall(Function function) throws DLIException;

	// /**
	// * Factory method for creating <code>DBStruct</code> objects.
	// *
	// * @param typeName the type name of the structured type that this
	// <code>DBStruct</code> object maps to. The typeName is the name of a
	// * defined type that has been defined for this database. It needs to be in
	// the following format:
	// * <pcbName>.<segmentName>.<structName>
	// * It is the value returned by <code>DBStruct.getTypeName</code>.
	// * @param attributes the attributes that populate the returned object
	// * @return a <code>DBStruct</code> object that maps to the given type and
	// is populated with the given attributes
	// * @throws DLIException if the typeName is null or invalid
	// */
	// public DBStruct createStruct(String typeName, Object[] attributes) throws
	// DLIException;

	/**
	 * Factory method for creating <code>DBStruct</code> objects.
	 * 
	 * @param typeName
	 *            the type name of the structured type that this
	 *            <code>DBStruct</code> object maps to. The typeName is the name
	 *            of a defined type that has been defined in the DBD for this
	 *            database. It needs to be in the following format:
	 *            <code>pcbName.segmentName.structName</code> It is the value
	 *            returned by <code>DBStruct.getTypeName</code>.
	 * @return a <code>DBStruct</code> object that maps to the given type
	 * @throws DLIException
	 *             if the typeName is null or invalid
	 * @see DBStruct
	 */
	public DBStruct createStruct(String typeName) throws DLIException;

	// /**
	// * Factory method for creating <code>DBArray</code> objects.
	// *
	// * @param typeName the name of the type the elements of the array map to.
	// The typeName is the name of a
	// * defined type that has been defined for this database. It needs to be in
	// the following format:
	// * <pcbName>.<segmentName>.<arrayName>
	// * It is the value returned by <code>DBArray.getTypeName</code>.
	// * @param elements the elements that populate the returned object
	// * @return a <code>DBArray</code> object that maps to the given type and
	// is populated with the given elements
	// * @throws DLIException if the typeName is null or invalid
	// */
	// public DBArray createArrayOf(String typeName, Object[] elements) throws
	// DLIException;

	/**
	 * Factory method for creating <code>DBArray</code> objects.
	 * 
	 * @param typeName
	 *            the name of the type the elements of the array map to. The
	 *            typeName is the name of a defined type that has been defined
	 *            in the DBD for this database. It needs to be in the following
	 *            format: <code>pcbName.segmentName.arrayName</code> It is the
	 *            value returned by <code>DBArray.getTypeName</code>.
	 * @return a <code>DBArray</code> object that maps to the given type
	 * @throws DLIException
	 *             if the typeName is null or invalid
	 * @see DBArray
	 */
	public DBArray createArrayOf(String typeName) throws DLIException;
	
	/**
	 * To avoid abendu3303, use INIT STATUS GROUPA. 
	 * IMS will give you a status code for unavailable databases. Example: BA
	 * 
	 * @param properties
	 *            a <code>DLICallProperty</code> instance that represents input
	 *            parameters for the DL/I call. See {@link DLICall} for examples
	 *            of how to set the properties..
	 * @return the status code returned by IMS
	 * @throws DLIException
	 */
	public short initStatusGroupA() throws DLIException;
	
	/**
	 * To avoid deadlock conditions(Abend u777), use INIT STATUS GROUPB. 
	 * IMS will give you a status code for unavailable databases. Example: BC
	 * 
	 * @return the status code returned by IMS
	 * @throws DLIException
	 */
	public short initStatusGroupB() throws DLIException;
}
