package com.ibm.ims.dli;

import java.math.BigDecimal;

/* (c) Copyright International Business Machines Corporation 2008. All rights reserved. */

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Hashtable;
import java.util.Vector;

/**
 * The <code>Path</code> interface represents a list of database segments for
 * the purpose of a DL/I retrieval, insert, delete, or update operation. The
 * <code>Path</code> can be viewed as the concatenation of all of the segment
 * instances in a specific database hierarchic path, starting from the highest
 * level segment that is nearest the root segment to the lowest level segment.
 * Use the <code>Path</code> interface to set or retrieve the value of any
 * segment field located in the hierarchic path.
 * 
 * The following code fragment shows how to create and use <code>Path</code> to
 * get the value of the ILLNAME field in the ILLNESS segment and return it as a
 * <code>String</code>. <blockquote>
 * 
 * <pre>
 * Path path = ssaList.getPathForRetrieveReplace();
 * if (pcb.getUnique(path, ssaList, false)) {
 * 	String fieldValue = path.getString("ILLNESS", "ILLNAME");
 * }
 * </pre>
 * 
 * </blockquote>
 * 
 * @see PathSet#next()
 * @see PathSet#next(Path)
 * @see PCB
 * @see SSAList#getPathForBatchUpdate(String)
 * @see SSAList#getPathForInsert(String)
 * @see SSAList#getPathForRetrieveReplace()
 */
public interface Path {

	/**
	 * Returns the names of the segment(s) contained in this <code>Path</code>
	 * as defined by the database metadata. If there is more than one segment,
	 * the segment names are delimited with periods.
	 * <p>
	 * <B>Note:</B> Can be used when making unqualified DL/I calls with no
	 * SSAList to determine what type of segment was returned in order to make
	 * the proper getXXX (where 'XXX' is the data type; for example, getString)
	 * calls.
	 * <p>
	 * The following code fragment shows how the <code>getName</code> method is
	 * used. <blockquote>
	 * 
	 * <PRE>
	 * String value = null;
	 * Path path = pcb.getNext(false);
	 * String name = path.getName(); // returns the name of the segment
	 * if (name.equals("PATIENT")) {
	 * 	value = path.getString("PATNAME");
	 * } else if (name.equals("DOCTOR")) {
	 * 	value = path.getString("DOCNAME");
	 * }
	 * </PRE>
	 * 
	 * </blockquote>
	 * 
	 * @return names of the segment(s) contained in this <code>Path</code>.
	 */
	public String getName();

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>com.ibm.ims.dli.DBArray</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         false
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>boolean</code>
	 * 
	 * @see DBArray
	 */
	public DBArray getArray(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>com.ibm.ims.dli.DBArray</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>boolean</code>
	 * 
	 * @see DBArray
	 */
	public void setArray(String segmentName, String fieldName, DBArray value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>boolean</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         false
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>boolean</code>
	 */
	public boolean getBoolean(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>boolean</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>boolean</code>
	 */
	public void setBoolean(String segmentName, String fieldName, boolean value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>byte</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>byte</code>
	 */
	public byte getByte(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>byte</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>byte</code>
	 */
	public void setByte(String segmentName, String fieldName, byte value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as an <code>int</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into an
	 *                <code>int</code>
	 */
	public int getInt(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as an <code>int</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into an
	 *                <code>int</code>
	 */
	public void setInt(String segmentName, String fieldName, int value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>short</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>short</code>
	 */
	public short getShort(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>short</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>short</code>
	 */
	public void setShort(String segmentName, String fieldName, short value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>long</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>long</code>
	 */
	public long getLong(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>long</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>long</code>
	 */
	public void setLong(String segmentName, String fieldName, long value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>float</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>float</code>
	 */
	public float getFloat(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>float</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>float</code>
	 */
	public void setFloat(String segmentName, String fieldName, float value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>double</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>double</code>
	 */
	public double getDouble(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>double</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>double</code>
	 */
	public void setDouble(String segmentName, String fieldName, double value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the segment of this
	 * <code>Path</code> object as a <code>byte</code> array.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>byte</code> array
	 */
	public byte[] getBytes(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the segment of this
	 * <code>Path</code> object as a <code>byte</code> array.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>byte</code> array
	 */
	public void setBytes(String segmentName, String fieldName, byte[] value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>java.lang.String</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.lang.String</code>.
	 */
	public String getString(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as an <code>java.lang.String</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.lang.String</code>.
	 */
	public void setString(String segmentName, String fieldName, String value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>java.math.BigDecimal</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.math.BigDecimal</code>.
	 */
	public BigDecimal getBigDecimal(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the segment of this
	 * <code>Path</code> object as a <code>java.math.BigDecimal</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.math.BigDecimal</code>.
	 */
	public void setBigDecimal(String segmentName, String fieldName, BigDecimal value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>java.sql.Date</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.Date</code>
	 */
	public Date getDate(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>java.sql.Date</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.Date</code>
	 */
	public void setDate(String segmentName, String fieldName, Date value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>java.sql.Time</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.Time</code>
	 */
	public Time getTime(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>java.sql.Time</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.Time</code>
	 */
	public void setTime(String segmentName, String fieldName, Time value) throws DLIException;

	/**
	 * Retrieves the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>java.sql.TimeStamp</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.TimeStamp</code>
	 */
	public Timestamp getTimestamp(String segmentName, String fieldName) throws DLIException;

	/**
	 * Retrieves the first warning reported by calls on this Path object.
	 * Subsequent warnings on this Path object will be chained to the DLIWarning
	 * object that this method returns.
	 * 
	 * The warning chain is automatically cleared each time a new row is read.
	 * 
	 * @return The first DLIWarning object reported or null if no warning
	 *         situations occurred.
	 */
	public DLIWarning getWarnings();

	/**
	 * Clears all warnings reported on this Path object. After this method is
	 * called, the method getWarnings returns null until a new warning is
	 * reported for this Path object.
	 */
	public void clearWarnings();

	/**
	 * Sets the value of the field in the specified segment of this
	 * <code>Path</code> object as a <code>java.sql.TimeStamp</code>.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.TimeStamp</code>
	 */
	public void setTimestamp(String segmentName, String fieldName, Timestamp value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as a <code><code>java.lang.Object</code></code>. <br>
	 * This method will return the value of the given field as a Java object.
	 * The type of the Java object will be the default Java object type
	 * corresponding to the field's actual type.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval.
	 */
	public Object getObject(String segmentName, String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object
	 * with the <code>java.lang.Object</code> value.
	 * 
	 * @param segmentName
	 *            name of the segment
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval.
	 */
	public void setObject(String segmentName, String fieldName, Object value) throws DLIException;

	/***********************************************************/
	/* Getters/Setters without segment name specified */
	/***********************************************************/

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as a <code>com.ibm.ims.dli.DBArray</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         false
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>boolean</code>
	 * 
	 * @see DBArray
	 */
	public DBArray getArray(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>com.ibm.ims.dli.DBArray</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>boolean</code>
	 * 
	 * @see DBArray
	 */
	public void setArray(String fieldName, DBArray value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as a <code>boolean</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         false
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>boolean</code>
	 */
	public boolean getBoolean(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>boolean</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>boolean</code>
	 */
	public void setBoolean(String fieldName, boolean value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the segment of this
	 * <code>Path</code> object as a <code>byte</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>byte</code>
	 */
	public byte getByte(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the segment of this
	 * <code>Path</code> object as a <code>byte</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>byte</code>
	 */
	public void setByte(String fieldName, byte value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the segment of this
	 * <code>Path</code> object as an <code>int</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into an
	 *                <code>int</code>
	 */
	public int getInt(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * an <code>int</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>int</code>
	 */
	public void setInt(String fieldName, int value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as a <code>short</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>short</code>
	 */
	public short getShort(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the s <code>Path</code> object
	 * as a <code>short</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>short</code>
	 */
	public void setShort(String fieldName, short value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as a <code>long</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field name is not found in the database or if the
	 *                field value was not able to be converted into a
	 *                <code>long</code>
	 */
	public long getLong(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>long</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>long</code>
	 */
	public void setLong(String fieldName, long value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as a <code>float</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>float</code>
	 */
	public float getFloat(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>float</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>float</code>
	 */
	public void setFloat(String fieldName, float value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as a <code>double</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is 0
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>double</code>
	 */
	public double getDouble(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>double</code> . <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>double</code>
	 */
	public void setDouble(String fieldName, double value) throws DLIException;

	/**
	 * Gets the value of the designated field in the <code>Path</code> object as
	 * a <code>byte</code> array. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>byte</code> array
	 */
	public byte[] getBytes(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>byte</code> array. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>byte</code> array
	 */
	public void setBytes(String fieldName, byte[] value) throws DLIException;

	/**
	 * Gets the value of the designated field in the <code>Path</code> object as
	 * a <code>java.lang.String</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.lang.String</code>.
	 */
	public String getString(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>java.lang.String</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.lang.String</code>.
	 */
	public void setString(String fieldName, String value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as a <code><code>java.math.BigDecimal</code></code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.math.BigDecimal</code>
	 */
	public BigDecimal getBigDecimal(String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the <code>Path</code> object as a
	 * <code>java.math.BigDecimal</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.math.BigDecimal</code>
	 */
	public void setBigDecimal(String fieldName, BigDecimal value) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>java.sql.Date</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.Date</code>
	 */
	public Date getDate(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>java.sql.Date</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.Date</code>
	 */
	public void setDate(String fieldName, Date value) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>java.sql.Time</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.Time</code>
	 */
	public Time getTime(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>java.sql.Time</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.Time</code>
	 */
	public void setTime(String fieldName, Time value) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>java.sql.TimeStamp</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.TimeStamp</code>
	 */
	public Timestamp getTimestamp(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object as
	 * a <code>java.sql.TimeStamp</code>. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.sql.TimeStamp</code>.
	 */
	public void setTimestamp(String fieldName, Timestamp value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>Path</code>
	 * object as an <code><code>Object</code></code> in the Java Programming
	 * Language. <br>
	 * This method will return the value of the given field as a Java object.
	 * The type of the Java object will be the default Java object type
	 * corresponding to the field's actual type. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @return value of the field; if the value is NULL, the value returned is
	 *         null
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value was not able to be converted into a
	 *                <code>java.math.BigDecimal</code>
	 */
	public Object getObject(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>Path</code> object
	 * with the <code>Object</code> value. <br>
	 * This method is used when the field name is unique within the
	 * <code>Path</code>. Otherwise use the method that has the segment name
	 * parameter.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if the field or segment name is not found in the current
	 *                <code>Path</code> because the segment or field does not
	 *                exist in the database or was not marked for retrieval, or
	 *                if the field value could not be converted into a
	 *                <code>java.sql.TimeStamp</code>.
	 */
	public void setObject(String fieldName, Object value) throws DLIException;

	/**
	 * Returns the AIB (Application Interface Block) associated with the most
	 * recent DL/I call.
	 * 
	 * @return the <code>AIB</code>
	 * 
	 */
	public AIB getAIB();

	/**
	 * Reports whether the last field read had a value of NULL. You must first
	 * call one of the getter methods on a field to try to read its value before
	 * you call this method to see if the value read was NULL.
	 * 
	 * @return true if the last column value read was NULL and false otherwise
	 */
	public boolean wasNull();

	/**
	 * Returns a collection of fields in this path. The key is the name of a
	 * segment and the values are a Vector of fields names in that segment.
	 * 
	 * @return the collection if this Path contains fields (e,g,; was used to
	 *         retrieve fields) and null otherwise
	 */
	public Hashtable<String, Vector<String>> getFields();

	/**
	 * Retrieves the number, types and properties of the fields of this
	 * <code>Path</code> object.
	 * 
	 * @return the description of the fields of this <code>Path</code> object
	 * @throws DLIException
	 *             if a processing error occurs
	 */
	public PathMetaData getMetaData() throws DLIException;

	/*
	 * public short getStatusCode(); // This needs to be returned from ODBM on
	 * every call for each IOArea public int getReturnCode(); // This will be in
	 * SQLCA public String getReturnCodeHex(); public int getReasonCode(); //
	 * This will be in SQLCA public String getReasonCodeHex(); public int
	 * getErrorCodeExtension(); // This will be in SQLCA //public String
	 * getErrorCodeExtensionHex(); public byte[] getKeyFeedbackArea(); // This
	 * will be in SQLCA TODO - always? Or is this a property set? If not, how do
	 * we get it later?
	 */
}
