package com.ibm.ims.dli;

/* (c) Copyright International Business Machines Corporation 2008. All rights reserved. */

/**
 * Interface used for storing and setting a prepared value. This is useful when
 * when the value to set in an <code>SSA</code> is not known until runtime. The
 * application can create a <code>SSAList</code> initially and then set the
 * <code>PreparedValue</code> with the appropriate value once it is known. This
 * also results in the ability to reuse an <code>SSA</code> (and therefore an
 * <code>SSAList</code>) several times throughout the course of an application
 * by simply changing the value in the <code>PreparedValue</code> object.
 * <P>
 * The following code fragment, in which ssaList is a valid <code>SSAList</code>
 * object, illustrates how to create and use a <code>PreparedValue</code>.
 * <blockquote>
 * 
 * <PRE>
 * // create a placeholder for the HOSPCODE field
 * PreparedValue pv = ssaList.createPreparedValue("HOSPCODE");  
 *   
 * // insert the placeholder in the qualification
 * ssaList.addInitialQualification("HOSPITAL", "HOSPCODE", SSAList.EQUALS, pv);  
 * ...
 * // insert the value 123 for HOSPCODE in the SSA list
 * pv.setValue("123");  
 * Path path = ssaList.getPathForRetrieveReplace();
 *   
 * // issue the DLI call
 * pcb.getUnique(path, ssaList, false);
 * </PRE>
 * 
 * </blockquote>
 * 
 * @see SSAList#createPreparedValue(String)
 * @see SSAList#getPreparedValue(String)
 * @see SSAList#getPreparedValues()
 */
public interface PreparedValue {

	/**
	 * Retrieves the name of the prepared value. Typically the name is the name
	 * of the field the prepared value parameter is referring to.
	 * 
	 * @return the name of the prepared value
	 */
	public String getName();

	/**
	 * Sets the value of the prepared value parameter using the given object.
	 * 
	 * @param value
	 *            the value of the designated prepared value parameter
	 */
	public void setValue(Object value);

	/**
	 * Gets the value of the prepared value parameter as an
	 * <code>java.lang.Object</code>.
	 * 
	 * @return the value of the prepared value parameter
	 */
	public Object getValue();

}
