package com.ibm.ims.dli.tm;

import com.ibm.ims.dli.DLIException;

/**
 * The <code>AssemblerCall</code> interface is used to invoke an assembler module.  The assembler module
 * can have up to 10 input arguments of varying types.  The module can also have one of eight different
 * return types, including void.
 * <p>The following code example shows how an <code>AssemblerCall</code> object is retrieved and used
 * to invoke an assembler module named MYMOD with an input parameter of type String:
 * <blockquote>
 * <pre>
 *    Application app = ApplicationFactory.createApplication();
 *    AssemblerCall asCall = app.getAssemblerCall();
 *    asCall.setModuleName("MYMOD");
 *    asCall.setReturnType(AssemblerCall.RETURN_VOID);
 *    asCall.setString(1, "CharacterInput");
 *    asCall.invoke();
 * </pre>
 * </blockquote>
 * 
 *
 */
public interface AssemblerCall {

    /**
     * Constant indicating the return type for the exit is void
     */
    public static final int RETURN_VOID = 1;

    /**
     * Constant indicating the return type for the exit is a String
     */
    public static final int RETURN_STRING = 2;

    /**
     * Constant indicating the return type for the exit is a byte array
     */
   public static final int RETURN_BINARY = 3;

   /**
    * Constant indicating the return type for the exit is a short
    */
    public static final int RETURN_SHORT = 4;

    /**
     * Constant indicating the return type for the exit is an integer
     */
    public static final int RETURN_INT = 5;

    /**
     * Constant indicating the return type for the exit is a long
     */
    public static final int RETURN_LONG = 6;

    /**
     * Constant indicating the return type for the exit is a float
     */
    public static final int RETURN_FLOAT = 7;

    /**
     * Constant indicating the return type for the exit is a double
     */
    public static final int RETURN_DOUBLE = 8;

    /**
     * Sets the name of the assembler module to invoke.
     * 
     * @param moduleName the name of the assembler module to invoke
     */
    public void setModuleName(String moduleName);

    /**
     * Retrieves the name of the assembler module
     * 
     */
    public String getModuleName();

    /**
     * Sets the character encoding to use for all character data passed to and returned from
     * the exit routine.
     * 
     * The default character encoding is Cp1047.
     * 
     * @param encoding the character encoding (e.g.; Cp1047)
     */
    public void setCharacterEncoding(String encoding);
    
    /**
     * Retrieves the character encoding
     * 
     */
    public String getCharacterEncoding();

    /**
     * Sets the return type of the exit routine.  The return type must be one of the
     * constants defined in this class.
     * 
     * @param returnType the return type of the exit routine
     */
    public void setReturnType(int returnType) throws DLIException;

    /**
     * Retrieves the return type of the exit routine.
     * 
     */
    public int getReturnType();
    
    /**
     * Sets the designated argument with a <code>java.lang.String</code> value.
     *
     * @param argument the index (1-10) of the argument to set
     * @param value the value of the argument
     */
    public void setString(int argument, String value) throws DLIException;

    /**
     * Sets the designated argument with a <code>short</code> value.
     *
     * @param argument the index (1-10) of the argument to set
     * @param value the value of the argument
     */
    public void setShort(int argument, short value) throws DLIException;

    /**
     * Sets the designated argument with an <code>int</code> value.
     *
     * @param argument the index (1-10) of the argument to set
     * @param value the value of the argument
     */
    public void setInt(int argument, int value) throws DLIException;

    /**
     * Sets the designated argument with a <code>long</code> value.
     *
     * @param argument the index (1-10) of the argument to set
     * @param value the value of the argument
     */
    public void setLong(int argument, long value) throws DLIException;

    /**
     * Sets the designated argument with a <code>float</code> value.
     *
     * @param argument the index (1-10) of the argument to set
     * @param value the value of the argument
     */
    public void setFloat(int argument, float value) throws DLIException;

    /**
     * Sets the designated argument with a <code>double</code> value.
     *
     * @param argument the index (1-10) of the argument to set
     * @param value the value of the argument
     */
    public void setDouble(int argument, double value) throws DLIException;

    /**
     * Sets the designated argument with a <code>byte[]</code> value.
     *
     * @param argument the index (1-10) of the argument to set
     * @param value the value of the argument
     */
    public void setBytes(int argument, byte[] value) throws DLIException;
    
    /**
     * Invokes the exit routine.
     * 
     * @return an Object representing the return type of the exit routine.
     */
    public Object invoke() throws DLIException;

}
