package com.ibm.ims.dli.tm;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import com.ibm.ims.dli.DLIException;

/**
 * The <code>IMSCallout</code> interface is the starting point of a IMSCallout
 * (ICAL) DL/I request.
 * 
 * The following code shows an example of the implementation of this interface.
 * 
 * <pre>
 * <code>
 * icalRequestArea = app.get31BitByteBuffer(100);
 * icalResponseArea = app.get31BitByteBuffer(100);
 *     
 * IMSCallout imsCallout = app.createImsCallout();
 *     
 * imsCallout.setOtmaDescriptor("SAMPLE")
 *           .setRequestArea(icalRequestArea)
 *           .setResponseArea(icalResponseArea)
 *           .setSubFunction(IMSCallout.SubFunction.SENDRECV)
 *           .setTimeout(10000)
 *           .execute();
 *           
 * System.out.println(imsCallout.getReturnCode());
 * System.out.println(imsCallout.getReasonCode());
 * System.out.println(imsCallout.getErrorExtensionCode());
 * 
 * </code>
 * </pre>
 * 
 * @see Application#createImsCallout()
 * 
 */
public interface IMSCallout {
	
	/**
	 * <b><code>SENDRECV</code></b>
	 * <br>
	 * This is a The IMS application program uses this subfunction to send a message
	 * and wait for the response. This subfunction is used for synchronous
	 * program-to-program communication.
	 * <p>
	 * <b><code>RECEIVE</code></b>
	 * <br>
	 * The IMS application program uses this subfunction to retrieve the
	 * complete response data from a previous incomplete ICAL call. If a SENDRECV
	 * subfunction call completes with AIB return code X'0100' and reason code
	 * X'000C', the response data did not fit in the response area. The application
	 * program can expand the response area and then retrieve the complete response
	 * with the RECEIVE subfunction call.
	 */
	public static enum SubFunction {
	SENDRECV, 
	RECEIVE
	}

	/**
	 * Sets the timeout value, in hundredths of a second, into the AIB to be used by <code>execute</code>
	 * <p>
	 * This is in direct correlation to the AIB field <code>AIBRSFLD</code> The
	 * valid range is 0 - 999999. The system default is 10 seconds.
	 * <p>
	 * This is an input parameter.
	 * 
	 * @param  int
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setTimeout(int timeout) throws DLIException;
	
	/**
	 * Sets the OTMA descriptor name in the AIB to be use by <code>execute</code>
	 * <p>
	 * This is in direct correlation to the AIB field <code>AIBRSNM1</code> This
	 * 8-byte, alphanumeric, left-aligned field must contain the name of the OTMA
	 * descriptor that defines the destination of the IMS call.
	 * <p>
	 * This is an input parameter.
	 * 
	 * @param otmaDescriptor
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setOtmaDescriptor(String otmaDescriptor) throws DLIException;
	
	/**
	 * Sets the SubFunction code in the AIB to be used by <code>execute</code>. This
	 * is in direct correlation to the AIB field <code>AIBSFUNC</code> This is a
	 * required static field in order to issue an ImsCallout request. The default is
	 * <code>SENDRECV</code>
	 * 
	 * <p>
	 * Valid SubFunctions are <code>SENDRECV</code> and <code>RECEIVE</code>
	 * <p>
	 * This is an input parameter.
	 * 
	 * @param subFunction
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setSubFunction(SubFunction subFunction) throws DLIException;
	
	/**
	 * Sets the map name in the AIB to be used by <code>execute</code>. This is in
	 * direct correlation with AIB field <code>AIBUTKN</code>
	 * 
	 * <p>
	 * Valid SubFunctions are <code>SENDRECV</code> and <code>RECEIVE</code>
	 * <p>
	 * This is an optional input parameter.
	 * 
	 * @param mapName
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setMapName(String mapName) throws DLIException;
	
	/**
	 * Sets the response area ByteBuffer to be used by <code>execute</code>.
	 * ByteBuffer used can be either <code>direct</code> or <code>non-direct</code>.
	 * This buffers capacity becomes the static field <code>AIBOAUSE</code>
	 * <p>
	 * This is an input parameter.
	 * <p>
	 * <b> Recommend allocating native ByteBuffer for improved storage efficiency.
	 * Using non-direct ByteBuffers will cause <code>execute</code> to check out
	 * internally managed direct ByteBuffers, which will be copied to the original
	 * set ByteBuffers after <code>execute</code></b>
	 * <p>
	 * If 31-bit native storage is required, Application class includes methods for
	 * allocating and freeing direct 31-bit ByteBuffers.
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param responseArea
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setResponseArea(ByteBuffer responseArea) throws DLIException;

	/**
	 * Sets the request area ByteBuffer to be used by <code>execute</code>.
	 * ByteBuffer to be used can be either <code>direct</code> or
	 * <code>non-direct</code>. This buffers capacity becomes the static field
	 * <code>AIBOALEN</code>
	 * <p>
	 * This is an input parameter.
	 * <p>
	 * <b> Recommend allocating native ByteBuffer for improved storage efficiency.
	 * Using non-direct ByteBuffers will cause <code>execute</code> to check out
	 * internally managed direct ByteBuffers, which will be copied to the original
	 * set ByteBuffers after <code>execute</code></b>
	 * <p>
	 * If 31-bit native storage is required, Application class includes methods for
	 * allocating and freeing direct 31-bit ByteBuffers.
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param requestArea
	 * @return IMSCallout Builder
	 * @throws DLIException
	 * 
	 */
	public IMSCallout setRequestArea(ByteBuffer requestArea) throws DLIException;
	
	/**
	 * Sets the request area ByteBuffer to be used by <code>execute</code>.
	 * ByteBuffer used can be either <code>direct</code> or <code>non-direct</code>.
	 * <p>
	 * requestSize specifies the actual amount of data set into the ByteBuffer with
	 * the intent to efficiently manage storage. Becomes the value for the static
	 * field <code>AIBOALEN</code>
	 * <p>
	 * This is an input parameter.
	 * <p>
	 * <b> Recommend allocating native ByteBuffer for improved storage efficiency.
	 * Using non-direct ByteBuffers will cause <code>execute</code> to check out
	 * internally managed direct ByteBuffers, which will be copied to the original
	 * set ByteBuffers after <code>execute</code></b>
	 * <p>
	 * If 31-bit native storage is required, Application class includes methods for
	 * allocating and freeing direct 31-bit ByteBuffers.
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param requestArea, requestSize
	 * @return IMSCallout builder
	 * @throws DLIException
	 * 
	 */
	public IMSCallout setRequestArea(ByteBuffer requestArea, int requestSize) throws DLIException;

	/**
	 * Sets the request area String to be used by <code>execute</code>. This Strings
	 * length becomes the static field <code>AIBOALEN</code>
	 * <p>
	 * This is an input parameter.
	 * 
	 * @param requestString
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setRequestArea(String requestString) throws DLIException;

	/**
	 * Sets the request area as an encoded String to be used by
	 * <code>execute</code>. This Strings length becomes the static field
	 * <code>AIBOALEN</code>
	 * <p>
	 * This is an input parameter.
	 * 
	 * @param requestString, charset
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setRequestArea(String requestString, Charset charset) throws DLIException;
	
	/**
	 * Sets the control data area String to be used by <code>execute</code>. This
	 * Strings length becomes the static field <code>AIBOPLEN</code>
	 * <p>
	 * This is an optional input parameter for <code>SENDRECV</code>.
	 * 
	 * @param String
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setControlDataArea(String controlDataArea)  throws DLIException ;
	
	/**
	 * Sets the control data area as an encoded String to be used by
	 * <code>execute</code>. This Strings length becomes the static field
	 * <code>AIBOPLEN</code>
	 * <p>
	 * This is an optional input parameter for <code>SENDRECV</code>.
	 * <p>
	 * <b> Recommend allocating native ByteBuffer for improved storage efficiency.
	 * Using non-direct ByteBuffers will cause <code>execute</code> to check out
	 * internally managed direct ByteBuffers, which will be copied to the original
	 * set ByteBuffers after <code>execute</code></b>
	 * <p>
	 * If 31-bit native storage is required, Application class includes methods for
	 * allocating and freeing direct 31-bit ByteBuffers.
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param controlDataArea, charset
	 * @return IMSCallout Builder
	 * @throws DLIException
	 * 
	 */
	public IMSCallout setControlDataArea(String controlDataArea, Charset charset)  throws DLIException ;

	/**
	 * Sets the control data area ByteBuffer to be used by <code>execute</code>.
	 * ByteBuffer used can be either <code>direct</code> or <code>non-direct</code>.
	 * <p>
	 * controlDataAreaSize specifies the actual amount of data set into the
	 * ByteBuffer with the intent to efficiently manage storage. Becomes the value
	 * for the static field <code>AIBOPLEN</code>
	 * <p>
	 * This is an optional input parameter for <code>SENDRECV</code>.
	 * <p>
	 * <b> Recommend allocating native ByteBuffer for improved storage efficiency.
	 * Using non-direct ByteBuffers will cause <code>execute</code> to check out
	 * internally managed direct ByteBuffers, which will be copied to the original
	 * set ByteBuffers after <code>execute</code></b>
	 * <p>
	 * If 31-bit native storage is required, Application class includes methods for
	 * allocating and freeing direct 31-bit ByteBuffers.
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param controlDataArea, controlDataAreaSize
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout setControlDataArea(ByteBuffer controlDataArea, int controlDataAreaSize) throws DLIException;

	/**
	 * Sets the control data area ByteBuffer to be used by
	 * <code>execute</code>. ByteBuffer used can be either <code>direct</code> or
	 * <code>non-direct</code>. This buffers capacity becomes the static field
	 * <code>AIBOPLEN</code>
	 * <p>
	 * This is an optional input parameter for <code>SENDRECV</code>.
	 * <p>
	 * <b> Recommend allocating native ByteBuffer for improved storage efficiency.
	 * Using non-direct ByteBuffers will cause <code>execute</code> to check out
	 * internally managed direct ByteBuffers, which will be copied to the original
	 * set ByteBuffers after <code>execute</code></b>
	 * <p>
	 * If 31-bit native storage is required, Application class includes methods for
	 * allocating and freeing direct 31-bit ByteBuffers.
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param controlDataArea
	 * @return IMSCallout Builder
	 * @throws DLIException
	 * 
	 */
	public IMSCallout setControlDataArea(ByteBuffer controlDataArea) throws DLIException;

	/**
	 * Executes the IMSCallout (ICAL) DL/I request.
	 * 
	 * @return IMSCallout Builder
	 * @throws DLIException
	 */
	public IMSCallout execute() throws DLIException;
	
	/**
	 * Retrieves the static field AIB return code. This is in direct correlation
	 * with AIB field <code>AIBRETRN</code>
	 * <p>
	 * This is an output parameter.
	 * 
	 * @return The return code as an <code>int</code>
	 * @throws DLIException
	 */
	public int getReturnCode() throws DLIException;

	/**
	 * Retrieves the static field AIB reason code. This is in direct correlation
	 * with AIB field <code>AIBREASN</code>
	 * <p>
	 * This is an output parameter.
	 * 
	 * @return The reason code as an <code>int</code>
	 * @throws DLIException
	 */
	public int getReasonCode() throws DLIException;

	/**
	 * Retrieves the static field AIB error code extension.
	 * <p>
	 * This is an output parameter.
	 * 
	 * @return The error code extension as an <code>int</code>
	 * @throws DLIException
	 */
	public int getErrorCodeExtension() throws DLIException;
	
	/**
	 * Retrieves the static field AIB output area used populated by issuing an
	 * <code>execute</code>. This is in direct correlation with AIB field
	 * <code>AIBOAUSE</code>. This field is only retrievable after making a
	 * IMSCallout request
	 * <p>
	 * This is an output parameter.
	 * 
	 * @return The output area used as an <code>int</code>
	 * @throws DLIException
	 */
	public int getOutputAreaUsed() throws DLIException;
	
	/**
	 * Retrieves the static field AIB output area length populated by issuing an
	 * <code>execute</code> and receiving partial data. This is in direct
	 * correlation with AIB field <code>AIBOALEN</code>. This field is only
	 * retrievable after making a IMSCallout request
	 * <p>
	 * This is an output parameter.
	 * 
	 * @return The output area length as an <code>int</code>
	 * @throws DLIException
	 */
	public int getOutputAreaLength() throws DLIException;

	/**
	 * Retrieves the static field for the response area ByteBuffer
	 * 
	 * @return response area as a <code>ByteBuffer</code>
	 */
	public ByteBuffer getResponseArea();

	/**
	 * Retrieves the static field for the request area ByteBuffer
	 * 
	 * @return request area as a <code>ByteBuffer</code>
	 */
	public ByteBuffer getRequestArea();

	/**
	 * Retrieves the static field for the control data area ByteBuffer
	 * 
	 * @return control data area as a <code>ByteBuffer</code>
	 */
	public ByteBuffer getControlDataArea();

	/**
	 * Retrieves the static field for the OTMA Descriptor
	 * 
	 * @return OTMA Descriptor
	 */
	public String getOtmaDescriptor();

	/**
	 * Retrieves the static field for the timeout
	 * 
	 * @return Timeout
	 */
	public int getTimeout();

	/**
	 * Retrieves the static field for the map name
	 * 
	 * @return Map Name
	 */
	public String getMapName();

	/**
	 * Retrieves the default encoding use by the IMSCallout interface
	 * 
	 * @return encoding as <code>Charset</code>
	 */
	public Charset getDefaultEncoding();

	/**
	 * Retrieves the static field for the <code>SubFunction</code>.
	 * 
	 * @return <code>SubFunction</code>
	 */
	public SubFunction getSubFunction();
}
