package com.ibm.ims.dli.tm;

import java.nio.ByteBuffer;

import com.ibm.ims.dli.AIB;
import com.ibm.ims.dli.DLIException;

/**
 * Use this interface to send and receive messages to an IMS message queue or to the Scratch Pad Area (SPA) 
 * for an application. The SPA is used in conversational transactions to store information between one step 
 * of the conversation and the next.
 * <p>For a code example that shows how to write a JMP application to process a conversational transaction, see the topic
 * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_jdrconversationaltrans.htm" target="_blank">Conversational transactions</a>" 
 * in the IMS Application Programming guide documentation.
 * <p>In a non-conversational transaction, it is possible to code the application with a message loop 
 * that retrieves messages until IMS indicates that no more messages exist on the queue. To support 
 * this style of programming, the {@link MessageQueue#getUnique(IOMessage)} method returns 
 * <code>true</code> if there is a message to retrieve from the queue and <code>false</code> if there is not.
 * @see IOMessage
 * @see MessageDestinationSpec
 */
public interface MessageQueue {
	/**
	 * Using this constant value will send the message to the default destination 
	 * of the transaction.
	 * 
	 * The value of this constant is <strong>null</strong>
	 */
    public final static MessageDestinationSpec DEFAULT_DESTINATION = null;

    /**
     * Retrieves the first segment of an input message from IMS. Use this method 
	 * to retrieve the SPA if the program is conversational.
	 * 
	 * @param message
	 *            the <code>IOMessage</code> object to receive the message
	 * @return flag indicating whether a message has been retrieved
	 * @throws TMException
	 *             if the operation fails
     */
    public boolean getUnique(IOMessage message) throws DLIException;
    
    /**
	 * Retrieves the first segment of an input message from IMS. Use this method to
	 * retrieve the SPA if the program is conversational.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param message 
	 *            the <code>ByteBuffer</code> object to receive the message
	 * @return flag indicating whether a message has been retrieved
	 * @throws TMException if the operation fails
	 */
    public boolean getUnique(ByteBuffer buffer) throws DLIException;
    
    /**
     * Retrieves the next segment of the message from IMS. Use this method to retrieve the
	 * remaining segments for a multi-segment message in the program.
	 * 
	 * <p>For a code example that shows how to handle multi-segment messages, see the topic
     * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_jmpappsmultisegmsgs.htm" target="_blank">Handling multi-segment messages</a>" 
     * in the IMS Application Programming guide documentation.
	 * @param message
	 *            the <code>IOMessage</code> object to receive the message
	 * @return <code>true</code> if the message was successfully retrieved, <code>false</code> if no more
	 *         messages
	 * @throws TMException
	 *             if the operation fails
     */
    public boolean getNext(IOMessage message) throws DLIException;
    
    /**
     * Retrieves the next segment of the message from IMS. Use this method to retrieve the
	 * remaining segments for a multi-segment message in the program.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param buffer
	 *            the <code>ByteBuffer</code> object to receive the message
	 * @return <code>true</code> if the message was successfully retrieved, <code>false</code> if no more
	 *         messages
	 * @throws TMException
	 *             if the operation fails
     */
    public boolean getNext(ByteBuffer buffer) throws DLIException;
    
    /**
     * Send an output message to IMS.  Use this method to return the message to the default destination, 
     * perform immediate program switching to an alternate IOPCB, or perform deferred program switching.
	 * <p>The following code example shows how to return the message to the default destination:
	 * <blockquote>
     * <pre>
     *    messageQueue.insert(outputMessage, MessageQueue.DEFAULT_DESTINATION);
     * </pre>
     * </blockquote>
     * <p>The following code example shows how to perform immediate program switching to an alternate IOPCB:
     * <blockquote>
     * <pre>
     *    MessageDestinationSpec mds = new MessageDestinationSpec();
     *    mds.setAlternatePCBName("alternatePCBName");
     *    mds.setDestination("newDestination");
     *    messageQueue.insert(outputMessage, mds);
     * </pre>
     * </blockquote>
     * <p>The following code example shows how to perform deferred program switching:
     * <blockquote>
     * <pre>
     *    spaMessage.setTrancodeName("newTran");
     *    messageQueue.insert(spaMessage, MessageQueue.DEFAULT_DESTINATION);
     * </pre>
     * </blockquote>
     * <p>For more details on program switching in JMP and JBP applications, see the topic
     * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_jmpandjbpprogramswitching.htm" target="_blank">Program switching in JMP and JBP applications</a>" 
     * in the IMS Application Programming guide documentation.
	 * @param message
	 *            the <code>IOMessage</code> object to send the message
	 * @param mds
	 *            the <code>MessageDestinationSpec</code> object that
	 *            specifies the destination to route the message to
	 * @throws TMException
	 *             if the operation fails
     */
  
    public void insert(IOMessage message, MessageDestinationSpec mds) throws DLIException;
	
    /**
     * Send an output message to IMS. Use this method to return to the default destination.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * 
	 * <p>The following code example shows how to return the message to the default destination:
	 * <blockquote>
     * <pre>
     *    ByteBuffer buffer = ByteBuffer.allocate(40);
     *    InputMessage input = new InputMessage(buffer);
     *    input.setLL(10)
     *    input.setZZ(0)
     *    input.setInputData("SAMPLE")
     *    messageQueue.insert(buffer);
     * </pre>
     * </blockquote>
     * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param buffer
	 *            the <code>ByteBuffer</code> object to send the message
	 * @throws TMException
	 *             if the operation fails
     */
	public void insert(ByteBuffer buffer) throws DLIException;
	

	 /**
     * Send an output message to IMS.  Use this method to return the message to the default destination, 
     * perform immediate program switching to an alternate IOPCB, or perform deferred program switching.
	 * <p>The following code example shows how to return the message to the default destination:
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * <blockquote>
     * <pre>
     *    messageQueue.insert(buffer, MessageQueue.DEFAULT_DESTINATION);
     * </pre>
     * </blockquote>
     * <p>The following code example shows how to perform immediate program switching to an alternate IOPCB:
     * <blockquote>
     * <pre>
     *    MessageDestinationSpec mds = new MessageDestinationSpec();
     *    mds.setAlternatePCBName("alternatePCBName");
     *    mds.setDestination("newDestination");
     *    messageQueue.insert(buffer, mds);
     * </pre>
     * </blockquote>
     * <p>For more details on program switching in JMP and JBP applications, see the topic
     * "<a href="http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_jmpandjbpprogramswitching.htm" target="_blank">Program switching in JMP and JBP applications</a>" 
     * in the IMS Application Programming guide documentation.
	 * 
     * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param buffer
	 *            the <code>ByteBuffer</code> object to send the message
	 * @param mds
	 *            the <code>MessageDestinationSpec</code> object that
	 *            specifies the destination to route the message to         
	 * @throws TMException
	 *             if the operation fails
     */
	public void insert(ByteBuffer buffer, MessageDestinationSpec mds) throws DLIException;
	
	/**
	 * Send an output message to IMS. Use this method to perform immediate program
	 * switching to an alternate IOPCB, or perform deferred program switching. To be
	 * used after a change call has set a destination within IMS
	 * 
	 * <p>
	 * Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)} <blockquote>
	 * 
	 * <p>
	 * The following code example shows how to perform immediate program switching
	 * to an alternate IOPCB: <blockquote>
	 * 
	 * <pre>
	 * messageQueue.insert(buffer, "ALTPCB1");
	 * </pre>
	 * 
	 * </blockquote>
	 * <p>
	 * For more details on program switching in JMP and JBP applications, see the
	 * topic "<a href=
	 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_jmpandjbpprogramswitching.htm"
	 * target="_blank">Program switching in JMP and JBP applications</a>" in the IMS
	 * Application Programming guide documentation.
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param buffer the <code>ByteBuffer</code> object to send the message
	 * @param alternatePcbName the name of the alternate PCB to be used for an insert
	 * 
	 * @throws TMException if the operation fails
	 */
	public void insert(ByteBuffer buffer, String alternatePcbName) throws DLIException;
	
	/*
	 * Send an output message to IMS. Use this method to perform immediate program
	 * switching to an alternate IOPCB, or perform deferred program switching. To be
	 * used after a change call has set a destination within IMS
	 * 
	 * <p>
	 * Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)} <blockquote>
	 * 
	 * <p>
	 * The following code example shows how to perform immediate program switching
	 * to an alternate IOPCB: <blockquote>
	 * 
	 * <pre>
	 * messageQueue.insert(buffer, "ALTPCB1", "MODNAME1"));
	 * </pre>
	 * 
	 * </blockquote>
	 * <p>
	 * For more details on program switching in JMP and JBP applications, see the
	 * topic "<a href=
	 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apg/ims_jmpandjbpprogramswitching.htm"
	 * target="_blank">Program switching in JMP and JBP applications</a>" in the IMS
	 * Application Programming guide documentation.
	 * 
	 * @see {@link Application#get31BitDirectByteBuffer(int)}
	 * @see ByteBuffer#allocateDirect(int)
	 * @param buffer the <code>ByteBuffer</code> object to send the message
	 * @param alternatePcbName the name of the alternate PCB to be used for an insert
	 * @param modName the Message Output Descriptor
	 * 
	 * @throws TMException if the operation fails
	 */
	public void insert(ByteBuffer buffer, String alternatePcbName, String modName) throws DLIException;
	
	/**
	 * Change the destination of a message to IMS. Use this method prior to issuing
	 * an insert with the intent of program switching without using the
	 * MessageDestinationSpec class
	 * 
	 * @param destinationName The name of the destination to change to
	 * @param altPCBName
	 * @throws DLIException
	 */
	public void change(String destinationName, String altPCBName) throws DLIException;
	
	/**
	 * A PURG call tells IMS TM that the message built against the specified I/O PCB, 
	 * or alternate PCB (with the ISRT call) is complete.
	 * 
	 * @param alternatePcbName
	 * @throws Exception
	 */
	public void purge(String alternatePcbName) throws DLIException;
	
	/**
	 * Use the PURG call to send output messages to several different terminals. 
	 * A PURG call tells IMS TM that the message built against the specified I/O PCB, 
	 * or alternate PCB (with the ISRT call) is complete. IMS TM collects the message 
	 * segments that have been inserted into one PCB as one message and sends the message 
	 * to the destination specified by the destination name of the alternate PCB listed 
	 * in the PURG call.
     *
     * If you specify an I/O area in the PURG call parameters, 
     * PURG acts as an ISRT call to insert the first segment of the next message. 
     * When you identify the I/O area, you can also specify a MOD name to change the screen format.
	 * 
	 * @param alternatePcbName
	 * @param message
	 * @throws Exception
	 */
	public void purge(String alternatePcbName, IOMessage message) throws DLIException;
	
	/**
	 * Use the PURG call to send output messages to several different terminals. 
	 * A PURG call tells IMS TM that the message built against the specified I/O PCB, 
	 * or alternate PCB (with the ISRT call) is complete. IMS TM collects the message 
	 * segments that have been inserted into one PCB as one message and sends the message 
	 * to the destination specified by the destination name of the alternate PCB listed 
	 * in the PURG call.
     *
     * If you specify an I/O area in the PURG call parameters, 
     * PURG acts as an ISRT call to insert the first segment of the next message. 
     * When you identify the I/O area, you can also specify a MOD name to change the screen format.
	 * 
	 * @param alternatePcbName
	 * @param message
	 * @param modName
	 * @throws DLIException
	 */
	public void purge(String alternatePcbName, IOMessage message, String modName) throws DLIException;
	
    /**
	 * Returns the <code>AIB</code> instance associated with the most recent DL/I call. 
	 * The <code>AIB</code> contains all the data attributes of the IMS application interface block. 
	 * @return the <code>AIB</code> instance associated with the most recent DL/I call.
     */
    public AIB getAIB();
}
