package com.ibm.ims.dli.tm;

import java.io.Serializable;
import java.util.Iterator;

import com.ibm.ims.dli.DLIException;

/**
 * Use this interface during symbolic checkpoint and restart calls to store and 
 * retrieve application state information.
 */
public interface SaveArea extends Serializable {

    /**
     * Retrieves all objects stored inside the <code>SaveArea</code> instance.
     * 
     * @return an <code>Iterator</code> containing the saved objects.
     */
    public Iterator retrieveObjects();
    
    /**
     * Retrieves objects stored inside the <code>SaveArea</code> instance. The objects will
     * be retrieved according to the order that they were stored. 
     * 
     * @param index a 1-based index indicating the order of saved objects.
     * 
     * @return Object a object stored in the <code>SaveArea</code> instance.
     * 
     * @throws TMException  
     */
    public Object getObject(int index) throws TMException;
    
    /**
     * Stores an object in the <code>SaveArea</code> instance.
     * 
     * @param obj the object to save.
     */
    public void addObject(Object obj);
    
//    /**
//     * Sets the limit of the data that the application can checkpoint
//     * 
//     * @param sizeLimit The maximum size of a checkpoint area in bytes
//     */
//    public static void setCheckpointSizeLimit(int sizeLimit);
//    
//    /**
//     * Gets the maximum size of data the application can checkpoint
//     * 
//     * @return The maximum size in bytes
//     */
//    public static int getCheckpointSizeLimit();
    
    /**
     * Checks for any objects saved in the <code>SaveArea</code> instance. 
     * 
     * @return boolean <code>true</code> if no objects are stored in the <code>SaveArea</code> instance
     *         and <code>false</code> otherwise
     */
    public boolean isEmpty();

    /**
     * Returns the number of objects stored in the <code>SaveArea</code> instance.
     * 
     * @return count of objects stored in the <code>SaveArea</code> instance.
     */
    public int size();
    
    /**
     * Use this method to confirm correct repositioning of a PCB after a restart (XRST) call.
	 * Following a restart, IMS attempts to reposition its databases to the positions existing 
	 * at the time of the chosen checkpoint, using information saved in the checkpoint on the IMS log. 
	 * 
	 * @param pcbArray
	 *            a <code>String</code> array of PCB reference or alias names as
	 *            defined in the Java metadata class that provides the database view.
	 *            
	 * @return boolean <code>true</code> if the PCB was correctly repositioned, <code>false</code> otherwise.  
     *    If <code>false</code>, the PCB may not be correctly repositioned, and the application must take 
     *    whatever action is required to ensure valid results before resuming processing.
     */
    public boolean verifyPCBPosition(String [] pcbArray) throws DLIException;
    
    /**
     * Verifies correct repositioning of a PCB after a restart call.
     * 
     * @return boolean <code>true</code> if the PCB was correctly repositioned, <code>false</code> otherwise.  
     *    If <code>false</code>, the PCB may not be correctly repositioned, and the application must take whatever 
     *    action is required to ensure valid results before resuming processing.
     */
    public boolean verifyPCBPosition(String pcbName) throws DLIException;

    /**
     * Returns the list of PCBs that were not reset correctly after a symbolic restart call was issued.
     * 
     * @return the array of PCB names
     */
    public String[] getInvalidPCBs();
    
    
    /**
     * Stores the name of a PCB that was not reset after a symbolic restart call.
     * 
     * @param pcbName the name of the PCB
     */
    public void setInvalidPCBs(String pcbName);
    
    /**
     * Resets the SaveArea instance by removing all application objects. 
     */
    public void clear();

}
