package com.ibm.ims.dli.types;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;

import com.ibm.ims.dli.DLIErrorMessages;
import com.ibm.ims.dli.converters.ConverterFactory;
import com.ibm.ims.dli.converters.PackedDecimalTypeConverter;

/**
 * Provides methods to read and write PackedDecimal values that are stored in an
 * IMS database as numeric representations of a date pattern.
 */
public class PackedDateConverter extends BaseTypeConverter {

	private static final long serialVersionUID = 6567537037659683697L;

	/**
	 * Reads a PackedDecimal value from the database and interprets it as a date
	 * value. The PackedDecimal value will be stored as a numeric representation
	 * of a all numeric date pattern i.e., pattern = YYYYMMdd PackedDecimalValue
	 * = 20110324
	 * 
	 * There are two valid properties for this converter: 1. pattern - This is
	 * the pattern of the date that is stored in the database 2. isSigned - If
	 * the PackedDecimal value is stored as signed or unsigned in the database
	 * 
	 * @return an object that represents the PackedDate value from the database
	 * @throws ConversionException
	 *             if an error occurs during processing
	 */
	public Object readObject(byte[] ioArea, int start, int length, Class objectType, Collection<String> warningStrings)
			throws ConversionException {

		if (objectType == java.sql.Date.class) {
			java.sql.Date result = null;

			// Retrieves the numeric Packed Decimal Value
			boolean isSigned;
			String signProp = this.properties.getProperty("isSigned");
			if (signProp != null && signProp.equalsIgnoreCase("Y")) {
				isSigned = true;
			} else {
				isSigned = false;
			}

			String pattern = this.properties.getProperty("pattern");
			if (!pattern.equals("yyyyMMdd")) {
				throw new ConversionException(
						"This Converter is a sample only and other patterns besides " + pattern + " are invalid.");
			}

			PackedDecimalTypeConverter packedConverter = ConverterFactory.createPackedDecimalConverter(pattern.length(),
					0, isSigned);
			BigDecimal packedDecimalValue = packedConverter.getBigDecimal(ioArea, start, length, warningStrings);

			SimpleDateFormat formatter = new SimpleDateFormat(pattern);
			try {
				result = new java.sql.Date(formatter.parse(packedDecimalValue.toString()).getTime());
			} catch (ParseException e) {
				throw new ConversionException(e.getMessage(), e);
			}

			return result;
		} else if (objectType == byte[].class) {
			byte[] data = new byte[length];
			System.arraycopy(ioArea, start, data, 0, length);

			return data;
		} else if (objectType == java.sql.Timestamp.class) {
			java.sql.Date value = (java.sql.Date) readObject(ioArea, start, length, java.sql.Date.class,
					warningStrings);

			return new java.sql.Timestamp(value.getTime());
		} else if (objectType == java.sql.Time.class) {
			java.sql.Date value = (java.sql.Date) readObject(ioArea, start, length, java.sql.Date.class,
					warningStrings);

			return new java.sql.Time(value.getTime());
		} else if (objectType == String.class) {
			java.sql.Date value = (java.sql.Date) readObject(ioArea, start, length, java.sql.Date.class,
					warningStrings);
			java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat();
			sdf.applyPattern(properties.getProperty("pattern"));

			return new String(sdf.format(value));
		} else {
			throw new ConversionException("Converting from PackedDate to " + objectType + " is not supported.");
		}
	}

	/**
	 * Writes a new PackedDecimal value to an IMS database as a numeric
	 * representation of a date pattern. Use the <code>pattern</code> property
	 * to specify the date pattern. Use the <code>isSigned</code> property to
	 * specify whether the input value is signed or unsigned.
	 * @throws UnsupportedTypeConversion 
	 */
	public void writeObject(byte[] ioArea, int start, int length, Object object, Collection<String> warningStrings)
			throws ConversionException, UnsupportedTypeConversion {

		if (object instanceof java.sql.Date) {
			// Convert from Date to BigDecimal value
			java.sql.Date value = (java.sql.Date) object;
			java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat();
			sdf.applyPattern(properties.getProperty("pattern"));
			String dateString = sdf.format(value);

			BigDecimal bdValue = new BigDecimal(dateString);

			// Write BigDecimal value in the database
			boolean isSigned;
			String signProp = this.properties.getProperty("isSigned");
			if (signProp != null && signProp.equalsIgnoreCase("Y")) {
				isSigned = true;
			} else {
				isSigned = false;
			}
			String pattern = properties.getProperty("pattern");
			PackedDecimalTypeConverter packedConverter = ConverterFactory.createPackedDecimalConverter(pattern.length(),
					0, isSigned);
			packedConverter.setBigDecimal(ioArea, start, length, bdValue, warningStrings);
		} else if (object instanceof byte[]) {
			byte[] value = (byte[]) object;

			if (length != value.length) {
				Object[] inserts = new Object[4];
				inserts[0] = "byte[]";
				inserts[1] = "PackedDate";
				inserts[2] = value.length;
				inserts[3] = length;
				throw new UnsupportedTypeConversion(
						DLIErrorMessages.getIMSBundle().getString("DATA_TRUNCATION_ERROR_WRITE", inserts));
			}

			if (ioArea == null) {
				ioArea = new byte[length];
			}
			System.arraycopy(value, 0, ioArea, start, length);
		} else if (object instanceof java.sql.Timestamp) {
			java.sql.Timestamp value = (java.sql.Timestamp) object;

			writeObject(ioArea, start, length, new java.sql.Date(value.getTime()), warningStrings);
		} else if (object instanceof java.sql.Time) {
			java.sql.Time value = (java.sql.Time) object;

			writeObject(ioArea, start, length, new java.sql.Date(value.getTime()), warningStrings);
		} else if (object instanceof String) {
			String value = (String) object;
			java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat();
			sdf.applyPattern(properties.getProperty("pattern"));

			try {
				writeObject(ioArea, start, length, new java.sql.Date(sdf.parse(value).getTime()), warningStrings);
			} catch (java.text.ParseException e) {
				throw new ConversionException(e.getMessage(), e);
			}
		} else if (object == null) {
			// Check for null because you can't store null values into IMS database.
			throw new ConversionException("Converting from null to PackedDate is not supported.");
		} else {
			throw new ConversionException("Converting from " + object.getClass() + " to PackedDate is not supported.");
		}
	}

}
