package com.ibm.ims.jdbc.xa;

/* (c) Copyright International Business Machines Corporation 2008. All rights reserved. */

import java.io.PrintWriter;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.logging.Logger;

import javax.sql.XAConnection;
import javax.sql.XADataSource;

import com.ibm.ims.dli.IMSConnectionSpec;
import com.ibm.ims.jdbc.ConnectionImpl;
import com.ibm.ims.jdbc.IMSDataSource;

/**
 * The <code>IMSXADataSource</code> is used as a factory for <code>XAConnection</code> objects and 
 * enables your JDBC application to participate in two-phase commit transactions against IMS databases.
 * 
 */
public class IMSXADataSource implements XADataSource {

    /***
     * Constant indicating the driver is pure Java and implements the network protocol for the data source.
     *
     */
    public static final int DRIVER_TYPE_4 = IMSConnectionSpec.DRIVER_TYPE_4;

    private IMSDataSource imsDataSource;
    
    /**
     * The default constructor.
     *
     */
    public IMSXADataSource() {
        super();
        this.imsDataSource = new IMSDataSource();
    }
    
    /**
     * Retrieves the <code>XAConnection</code> object.
     * 
     * @return the <code>XAConnection</code> object, which represents a physical database
     * 		connection that can be used in a distributed transaction.
     */
    public XAConnection getXAConnection() throws SQLException {
        ConnectionImpl connImpl = (ConnectionImpl)imsDataSource.getConnection();
        return new XAConnectionImpl(connImpl);
    }

    /**
     * Retrieves the <code>XAConnection</code> object, using the given user name and password.
     * 
     * @param user the user name
     * @param password the user password
     * @return the <code>XAConnection</code> object, which represents a physical database
     * 		connection that can be used in a distributed transaction.
     */
    public XAConnection getXAConnection(String user, String password) throws SQLException {
        ConnectionImpl connImpl = (ConnectionImpl)imsDataSource.getConnection(user, password);
        return new XAConnectionImpl(connImpl);
    }

    /**
     * Retrieves the log writer for this <code>XADataSource</code> object.
     * <p>
     * The log writer is a character output stream to which all logging and tracing messages
     * for this <code>XADataSource</code> will be printed. This includes messages printed by the methods of 
     * this object,messages printed by methods of other objects manufactured by this object, and so on. 
     * Messages printed to a data source specific log writer are not printed to the log writer associated 
     * with the <code>java.sql.Drivermanager</code>
     * class. When a <code>XADataSource</code> object is created, the log writer is initially null; in other words, the default is for
     * logging to be disabled.
     *
     * @return the log writer for this data source or null if logging is disabled
     */
    public PrintWriter getLogWriter() throws SQLException {
        return imsDataSource.getLogWriter();
    }

    /**
     *  Retrieves the login timeout for this <code>XADataSource</code> object
     *  
     *  @return the login timeout
     */
	public int getLoginTimeout() throws SQLException {
	    return imsDataSource.getLoginTimeout();
    }

    /***
     * Sets the log writer for this <code>XADataSource</code> object to the given <code>java.io.PrintWriter</code> 
     * object. 
     * <p>
     * The log writer is a character output stream to which all logging and tracing messages for this 
     * <code>XADataSource</code> will be printed. 
     * This includes messages printed by the methods of this object, messages printed by methods of other 
     * objects manufactured by this object, and so on. Messages printed to a data source specific log writer are 
     * not printed to the log writer associated with the <code>java.sql.Drivermanager</code> class. 
     * When a <code>XADataSource</code> object is created the log writer is initially <code>null</code>; 
     * in other words, the default is for logging to be disabled. 
     * 
     * @param out the new log writer; to disable logging, set to <code>null</code> 
     */
	public void setLogWriter(PrintWriter out) throws SQLException {
	    imsDataSource.setLogWriter(out);
	}

    /***
     * Sets the maximum time in seconds that this data source will wait while attempting to connect 
     * to a database. A value of zero specifies that the timeout is the default system timeout if there is one; 
     * otherwise, it specifies that there is no timeout. When a <code>XADataSource</code> object is created, the login timeout 
     * is initially zero. 
     * 
     * @param seconds the data source login time limit 
     */
	public void setLoginTimeout(int seconds) throws SQLException {
	    imsDataSource.setLoginTimeout(seconds);
	}

//	public void setDataSourceProperties(IMSDataSource dataSource) {
//		this.dataSource = dataSource;
//	}

    /**
     * Retrieves the URL of the database metadata representing the target IMS database.
     * 
     * @return the metadata URL
     * 
     * @deprecated {@link #getDatabaseName()}
     */
	public String getMetadataURL() {
		return imsDataSource.getMetadataURL();
	}

    /**
     * Sets the location of the database metadata representing the target IMS database.  The metadata URL is the fully 
     * qualified name of the Java metadata class generated by the IMS Explorer for Development.
     * 
     * The URL must begin with class://
     * 
     * @param metadataURL metadata url
     * 
     * @deprecated {@link #setDatabaseName(String)}
     */
	public void setMetadataURL(String metadataURL) {
        imsDataSource.setMetadataURL(metadataURL);
	}
	
    /**
     * Retrieves the database name this DataSource is configured to access.
     * 
     * @return the database name
     */
	public String getDatabaseName() {
	    return imsDataSource.getDatabaseName();
	}
	
    /**
     * Sets the name of the target IMS database to be accessed. 
     * 
     * If the metadata repository is the IMS catalog then the database name is
     * the name of the PSB (program specification block) containing the database(s)
     * to be accessed.
     * 
     * If the metadata repository is the file system then the database name
     * is the fully qualified name of the Java metadata class generated
     * by the IMS Explorer for Development.  In this case the name must begin with class://
     * 
     * @param databaseName
     *            database name
     */
	public void setDatabaseName(String databaseName) {
	    imsDataSource.setDatabaseName(databaseName);
	}

    /**
     * Retrieves the name or IP address of your datastore server (IMS Connect).
     * 
     * @return the name or IP address of the datastore server (IMS Connect)
     */
	public String getDatastoreServer() {
		return imsDataSource.getDatastoreServer();
	}

    /**
     * Sets the name or IP address of the datastore server (IMS Connect).  You can provide either the 
     * host name (for example, dev123.svl.ibm.com) or the IP address  (for example, 192.166.0.2).
     * 
     * @param datastoreServer name or IP address of the datastore server (IMS Connect)
     */
	public void setDatastoreServer(String datastoreServer) {
        imsDataSource.setDatastoreServer(datastoreServer);
	}

    /**
     * Retrieves the description of this <code>XADataSource</code>.
     * @return the description
     */
	public String getDescription() {
		return imsDataSource.getDescription();
	}

    /**
     * Sets the description of this <code>XADataSource</code>.
     * 
     * @param description the description
     */
	public void setDescription(String description) {
        imsDataSource.setDescription(description);
	}

    /**
     * Retrieves the database connectivity method. 
     * 
     * @return the database connectivity method
     */
	public int getDriverType() {
		return imsDataSource.getDriverType();
	}

    /**
     * Sets the type of driver to use to connect to the database.
     * 
     * @param driverType Only IMSXADataSource.DRIVER_TYPE_4 is currently supported.
     */
	public void setDriverType(int driverType) {
        imsDataSource.setDriverType(driverType);
	}

    /**
     * Retrieves the user password.
     *
     * @return user password value for this connection
     */
	public String getPassword() {
		return imsDataSource.getPassword();
	}

    /**
     * Sets the user password.
     *
     * @param password User password value
     */
	public void setPassword(String password) {
        imsDataSource.setPassword(password);
	}

    /**
     * Retrieves the port number.
     * 
     * @return the port number for this connection.
     */
	public int getPortNumber() {
		return imsDataSource.getPortNumber();
	}

    /**
     * Sets the port number to be used to communicate with IMS Connect. The port number is 
     * defined using the DRDAPORT parameter on the ODACCESS statement in the integrated 
     * IMS Connect configuration PROCLIB member.
     * <p>By default, the port number is 8888.
     * 
     * @param portNumber port number
     */
	public void setPortNumber(int portNumber) {
        imsDataSource.setPortNumber(portNumber);
	}

    /**
     * Retrieves name of the IMS datastore this <code>XADataSource</code> is configured to access.
     * 
     * @return the datastore name
     */
	public String getDatastoreName() {
		return imsDataSource.getDatastoreName();
	}

    /**
     * Sets the name of the IMS datastore to access. The datastore parameter must match either the name of the datastore
     * defined to ODBM or be blank.  The datastore name is defined in the ODBM CSLDCxxx PROCLIB member using either the
     * DATASTORE (NAME=name) or ALIAS (NAME=aliasname) parameter.  If the datastore value is left blank (or not supplied), IMS Connect will connect 
     * to any available instance of ODBM as it is assumed all datastores defined to ODBM are data shared.   
     * 
     * @param datastoreName IMS datastore name
     */
	public void setDatastoreName(String datastoreName) {
        imsDataSource.setDatastoreName(datastoreName);
	}

    /**
     * Retrieves the user name.
     *
     * @return the user name for this connection.
     */
	public String getUser() {
		return imsDataSource.getUser();
	}

    /**
     * Sets the user name
     *
     * @param user User name.
     */
	public void setUser(String user) {
        imsDataSource.setUser(user);
	}

	public Logger getParentLogger() throws SQLFeatureNotSupportedException {
		throw new SQLFeatureNotSupportedException();
	}
	
    /**
     * Sets the option to whether to remove the map case fields that don't satisfy the DEPENDINGON 
     * field condition in the WHERE clause from the column's list in the result set.
     * <br><br>
     * When set to <b>true</b>, only the valid map case fields will be returned in the result set.
     * <br><br>
     * When set to <b>false</b> (default) the result set will contain all map case fields.
     * 
     * @param removeInvalidCaseFields
     */
    public void setRemoveInvalidCaseFields(boolean removeInvalidCaseFields) {
    	this.imsDataSource.setRemoveInvalidCaseFields(removeInvalidCaseFields);
    }
    
    /**
     * Returns the option to whether to remove the map case fields that don't satisfy the DEPENDINGON 
     * field condition in the WHERE clause from the column's list in the result set.
     * <br><br>
     * When set to <b>true</b>, only the valid map case fields will be returned in the result set.
     * <br><br>
     * When set to <b>false</b> (default) the result set will contain all map case fields.
     * 
     * @return boolean removeInvalidCaseFields
     */
    public void getRemoveInvalidCaseFields() {
    	this.imsDataSource.getRemoveInvalidCaseFields();
    }
	
	/**
	 * Sets the option to expand or not expand the ResultSet of a column with data
	 * type ARRAY.
	 * <br><br>
	 * When set to <b>true</b> the array result set expands the sub-elements of the
	 * array into 1 or more columns depending on the number of fields defined within
	 * the array.
	 * <br><br>
	 * When set to <b>false</b> (default) the array result set returns a result set
	 * of exactly 2 columns. Column 1 is the index and column 2 is a Struct object
	 * that contains accessors for fields within that array element.
	 * 
	 * @param expandArrayResultSet
	 */
	public void setExpandArrayResultSet(boolean expandArrayResultSet) {
		this.imsDataSource.setExpandArrayResultSet(expandArrayResultSet);
	}
	
	/**
	 * Sets the option to whether to optimize the SSA or not
	 * <br><br>
	 * When set to <b>true</b> the SSA will be optimized to a fully qualified key if 
	 * subfields of that key are provided.
	 * <br><br>
	 * When set to <b>false</b> (default) the SSA will not be optimized.
	 * 
	 * @param ssaOptimization
	 */
	public void setSsaOptimization(boolean ssaOptimization) {
		this.imsDataSource.setSsaOptimization(ssaOptimization);
	}
	
	/**
	 * Returns the option to expand or not expand the ResultSet of a column with data
	 * type ARRAY.
	 * <br><br>
	 * When set to <b>true</b> the array result set expands the sub-elements of the
	 * array into 1 or more columns depending on the number of fields defined within
	 * the array.
	 * <br><br>
	 * When set to <b>false</b> (default) the array result set returns a result set
	 * of exactly 2 columns. Column 1 is the index and column 2 is a Struct object
	 * that contains accessors for fields within that array element.
	 * 
	 * @param expandArrayResultSet
	 */
	public boolean getExpandArrayResultSet() {
		return this.imsDataSource.getExpandArrayResultSet();
	}
	
	/**
	 * Returns the option to whether to optimize the SSA or not
	 * <br><br>
	 * When set to <b>true</b> the SSA will be optimized to a fully qualified key if 
	 * subfields of that key are provided.
	 * <br><br>
	 * When set to <b>false</b> (default) the SSA will not be optimized.
	 * 
	 * @param ssaOptimization
	 */
	public boolean getSsaOptimization() {
		return this.imsDataSource.getSsaOptimization();
	}
}
