/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.ratelimiter;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.ifengxue.http.HttpClientException;
import com.ifengxue.http.annotation.BodyType;
import com.ifengxue.http.annotation.ResponseType;
import com.ifengxue.http.executor.HttpExecutor;
import com.ifengxue.http.executor.Request;
import com.ifengxue.http.proxy.InterceptorAdaptor;
import com.ifengxue.http.ratelimiter.adapter.RateLimiterAdapter;
import com.ifengxue.http.util.AnnotationUtil;
import java.lang.reflect.Method;
import java.time.Duration;
import java.util.concurrent.ExecutionException;
import javax.annotation.Nullable;

/**
 * rate limiter interceptor
 *
 * @see RateLimiter
 */
public class RateLimiterInterceptor extends InterceptorAdaptor {

  private static Cache<Method, RateLimiterHolder> cache;

  static {
    cache = CacheBuilder.newBuilder()
        .maximumSize(1000)
        .expireAfterAccess(Duration.ofMinutes(5))
        .build();
  }

  @Override
  public <T> T beforeRequest(Method method, Request request, BodyType bodyType, ResponseType responseType,
      HttpExecutor executor) throws HttpClientException {
    try {
      RateLimiterHolder rateLimiterHolder = cache.get(method, () -> {
        RateLimiter rateLimiter = AnnotationUtil.findAnnotation(method, RateLimiter.class);
        if (rateLimiter == null) {
          return RateLimiterHolder.EMPTY;
        }
        return new RateLimiterHolder(rateLimiter);
      });
      if (rateLimiterHolder.rateLimiter == null) {
        return null;
      }
      RateLimiterAdapter rateLimiterAdapter = RateLimiterManager.getRateLimiterAdapter(rateLimiterHolder.rateLimiter);
      rateLimiterAdapter.requestPermission(rateLimiterHolder.rateLimiter.timeoutForWaitLimit());
    } catch (ExecutionException e) {
      throw new HttpClientException("load cache error", e.getCause());
    }
    return null;
  }

  private static class RateLimiterHolder {

    private static final RateLimiterHolder EMPTY = new RateLimiterHolder(null);
    @Nullable
    private final RateLimiter rateLimiter;

    private RateLimiterHolder(@Nullable RateLimiter rateLimiter) {
      this.rateLimiter = rateLimiter;
    }
  }
}
