/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.ratelimiter.adapter;

import com.ifengxue.http.ratelimiter.RateLimiterConfig;
import io.github.resilience4j.ratelimiter.RateLimiter;
import java.time.Duration;
import java.time.temporal.ChronoUnit;

public class ResilienceRateLimiterAdapter implements RateLimiterAdapter {

  private final String name;
  private final RateLimiter rateLimiter;
  private final RateLimiterConfig rateLimiterConfig;

  public ResilienceRateLimiterAdapter(String name, RateLimiterConfig rateLimiterConfig) {
    io.github.resilience4j.ratelimiter.RateLimiterConfig config = io.github.resilience4j.ratelimiter.RateLimiterConfig
        .custom()
        .timeoutDuration(Duration.of(rateLimiterConfig.getTimeoutForWaitLimit(), ChronoUnit.MILLIS))
        .limitRefreshPeriod(Duration.of(rateLimiterConfig.getLimitRefreshPeriod(), ChronoUnit.MILLIS))
        .limitForPeriod(rateLimiterConfig.getLimitForPeriod())
        .build();
    rateLimiter = RateLimiter.of(name, config);
    this.name = name;
    this.rateLimiterConfig = rateLimiterConfig;
  }

  @Override
  public void requestPermission(long timeoutForWaitLimit) throws RequestLimitedException, IllegalStateException {
    boolean permission = rateLimiter.getPermission(Duration.of(timeoutForWaitLimit, ChronoUnit.MILLIS));
    if (permission) {
      return;
    }
    if (Thread.currentThread().isInterrupted()) {
      throw new IllegalStateException("Thread was interrupted during permission wait");
    }
    throw new RequestLimitedException("Request limited for limiter: " + name);
  }

  @Override
  public RateLimiterConfig getRateLimiterConfig() {
    return rateLimiterConfig;
  }
}
