/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.codec;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.parser.Feature;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.alibaba.fastjson.serializer.SerializerFeature;
import java.lang.reflect.Type;
import java.util.Map;
import java.util.Optional;

/**
 * fastjson json codec
 */
public class FastjsonJsonCodec implements JsonCodec {

  private final SerializeConfig serializeConfig;
  private final SerializerFeature[] serializerFeatures;
  private final Feature[] features;

  public FastjsonJsonCodec() {
    this(null, null, null);
  }

  public FastjsonJsonCodec(SerializeConfig serializeConfig,
      SerializerFeature[] serializerFeatures, Feature[] features) {
    this.serializeConfig = Optional.ofNullable(serializeConfig).orElseGet(SerializeConfig::getGlobalInstance);
    this.serializerFeatures = Optional.ofNullable(serializerFeatures).orElseGet(() -> new SerializerFeature[0]);
    this.features = Optional.ofNullable(features).orElseGet(() -> new Feature[0]);
  }

  @Override
  public <T> String encode(T obj) {
    return JSON.toJSONString(obj, serializeConfig, serializerFeatures);
  }

  @Override
  @SuppressWarnings("unchecked")
  public <T> Map<String, Object> decode(T obj) {
    Object jsonObj = JSON.toJSON(obj, serializeConfig);
    if (jsonObj instanceof Map) {
      return (Map<String, Object>) jsonObj;
    }
    throw new CodecException(obj.getClass().getName() + " can't be cast to Map");
  }

  @Override
  public <T> T decode(String text, Type type) {
    return JSON.parseObject(text, type, features);
  }

  @Override
  public <T> T decode(String text, Class<T> clazz) {
    return JSON.parseObject(text, clazz, features);
  }
}
