/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.collection;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * value可能为{@link List}的Map，如果某个key仅调用{@link #put(Object, Object)}一次，且value不是{@link Collection}，那么value会保持原始类型
 */
public class MultiValueMap<K> implements MultiMap<K> {

  private Map<K, Object> internalMap;

  public MultiValueMap() {
    internalMap = new HashMap<>();
  }

  public MultiValueMap(int initialCapacity) {
    internalMap = new HashMap<>(initialCapacity);
  }

  @Override
  public int size(K key) {
    if (!containsKey(key)) {
      return 0;
    }
    Object value = get(key);
    if (value instanceof Collection) {
      return ((Collection) value).size();
    }
    if (value.getClass().isArray()) {
      return Array.getLength(value);
    }
    return 1;
  }

  @Override
  public int size() {
    return internalMap.size();
  }

  @Override
  public boolean isEmpty() {
    return internalMap.isEmpty();
  }

  @Override
  public boolean containsKey(Object key) {
    return internalMap.containsKey(key);
  }

  @Override
  public boolean containsValue(Object value) {
    for (Entry<K, Object> entry : internalMap.entrySet()) {
      if ((value instanceof Collection) && value.equals(entry.getValue())) {
        return true;
      }
      if (!(entry.getValue() instanceof Collection) && Objects.equals(entry.getValue(), value)) {
        return true;
      } else if (entry.getValue() instanceof Collection) {
        if (((Collection) entry.getValue()).contains(value)) {
          return true;
        }
      }
    }
    return false;
  }

  @Override
  public Object get(Object key) {
    return internalMap.get(key);
  }

  @Override
  @SuppressWarnings("all")
  public Object put(K key, Object value) {
    if (!internalMap.containsKey(key)) {
      return internalMap.put(key, value);
    }
    Object oldValue = internalMap.get(key);
    if (oldValue instanceof Collection) {
      ((Collection) oldValue).add(value);
      return oldValue;
    } else if (oldValue.getClass().isArray()) {
      int length = Array.getLength(oldValue);
      Object targetArray = Array.newInstance(oldValue.getClass().getComponentType(), length + 1);
      System.arraycopy(oldValue, 0, targetArray, 0, length);
      Array.set(targetArray, length, value);
      internalMap.put(key, targetArray);
      return targetArray;
    } else {
      List<Object> list = new ArrayList<>();
      list.add(oldValue);
      list.add(value);
      return internalMap.put(key, list);
    }
  }

  @Override
  public Object remove(Object key) {
    return internalMap.remove(key);
  }

  @Override
  public void putAll(Map<? extends K, ?> m) {
    m.forEach(this::put);
  }

  @Override
  public void clear() {
    internalMap.clear();
  }

  @Override
  public Set<K> keySet() {
    return internalMap.keySet();
  }

  @Override
  public Collection<Object> values() {
    return internalMap.values();
  }

  @Override
  public Set<Entry<K, Object>> entrySet() {
    return internalMap.entrySet();
  }
}
