/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.contract;

import com.ifengxue.http.annotation.BodyType;
import com.ifengxue.http.annotation.HttpMethod;
import com.ifengxue.http.annotation.ResponseType;
import java.lang.reflect.Type;
import java.util.Map;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * http operations，通过该接口定义的方法可以很容易的发起http请求
 */
public interface HttpOperations {

  // get

  /**
   * 自定义执行get请求并获取<code>json</code>编码的实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param type 返回实体类型
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>type</code>指定的返回值
   */
  default <T> T getForJsonEntity(@Nonnull String url, @Nonnull Type type, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables) {
    return exchange(url, HttpMethod.GET, BodyType.X_WWW_FORM_URLENCODED, ResponseType.APPLICATION_JSON, type,
        httpHeaders, uriVariables, null);
  }

  /**
   * 自定义执行get请求并获取实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param type 返回实体类型
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>type</code>指定的返回值
   */
  default <T> T getForEntity(@Nonnull String url, @Nonnull BodyType bodyType, @Nonnull ResponseType responseType,
      @Nonnull Type type, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables) {
    return exchange(url, HttpMethod.GET, bodyType, responseType, type, httpHeaders, uriVariables, null);
  }

  /**
   * 自定义执行get请求并获取<code>json</code>编码的实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param responseEntityClass 返回实体类
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>type</code>指定的返回值
   */
  default <T> T getForJsonEntity(@Nonnull String url, @Nonnull Class<T> responseEntityClass,
      @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables) {
    return exchange(url, HttpMethod.GET, BodyType.X_WWW_FORM_URLENCODED, ResponseType.APPLICATION_JSON,
        responseEntityClass,
        httpHeaders, uriVariables, null);
  }

  /**
   * 自定义执行get请求并获取实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param responseEntityClass 返回实体类
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>responseEntityClass</code>指定的返回值
   */
  default <T> T getForEntity(@Nonnull String url, @Nonnull BodyType bodyType, @Nonnull ResponseType responseType,
      @Nonnull Class<T> responseEntityClass, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables) {
    return exchange(url, HttpMethod.GET, bodyType, responseType, (Type) responseEntityClass, httpHeaders, uriVariables,
        null);
  }

  /**
   * 自定义执行get请求并获取实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param typeReference 返回实体类型
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>typeReference</code>指定的返回值
   */
  default <T> T getForEntity(@Nonnull String url, @Nonnull BodyType bodyType, @Nonnull ResponseType responseType,
      @Nonnull TypeReference<T> typeReference, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables) {
    return exchange(url, HttpMethod.GET, bodyType, responseType, typeReference.getType(), httpHeaders, uriVariables,
        null);
  }

  // post

  /**
   * 自定义执行post请求并获取<code>json</code>实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param type 返回实体类型
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>type</code>指定的返回值
   */
  default <T> T postForJsonEntity(@Nonnull String url, @Nonnull Type type, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables, @Nullable Object requestBody) {
    return exchange(url, HttpMethod.POST, BodyType.APPLICATION_JSON, ResponseType.APPLICATION_JSON, type, httpHeaders,
        uriVariables, requestBody);
  }

  /**
   * 自定义执行post请求并获取实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param type 返回实体类型
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>type</code>指定的返回值
   */
  default <T> T postForEntity(@Nonnull String url, @Nonnull BodyType bodyType, @Nonnull ResponseType responseType,
      @Nonnull Type type, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables, @Nullable Object requestBody) {
    return exchange(url, HttpMethod.POST, bodyType, responseType, type, httpHeaders, uriVariables, requestBody);
  }

  /**
   * 自定义执行post请求并获取<code>json</code>实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param responseEntityClass 返回实体类
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>type</code>指定的返回值
   */
  default <T> T postForJsonEntity(@Nonnull String url, @Nonnull Class<T> responseEntityClass,
      @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables, @Nullable Object requestBody) {
    return exchange(url, HttpMethod.POST, BodyType.APPLICATION_JSON, ResponseType.APPLICATION_JSON, responseEntityClass,
        httpHeaders, uriVariables, requestBody);
  }

  /**
   * 自定义执行post请求并获取实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param responseEntityClass 返回实体类
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>responseEntityClass</code>指定的返回值
   */
  default <T> T postForEntity(@Nonnull String url, @Nonnull BodyType bodyType, @Nonnull ResponseType responseType,
      @Nonnull Class<T> responseEntityClass, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables, @Nullable Object requestBody) {
    return exchange(url, HttpMethod.POST, bodyType, responseType, (Type) responseEntityClass, httpHeaders, uriVariables,
        requestBody);
  }

  /**
   * 自定义执行post请求并获取实体
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param typeReference 返回实体类型
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @return 由<code>typeReference</code>指定的返回值
   */
  default <T> T postForEntity(@Nonnull String url, @Nonnull BodyType bodyType, @Nonnull ResponseType responseType,
      @Nonnull TypeReference<T> typeReference, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables, @Nullable Object requestBody) {
    return exchange(url, HttpMethod.POST, bodyType, responseType, typeReference.getType(), httpHeaders, uriVariables,
        requestBody);
  }

  // exchange

  /**
   * 自定义执行http请求
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param method 请求方法。如<code>GET, POST</code>等
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param responseEntityClass 返回实体类
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @param requestBody 请求body
   * @return 由<code>responseEntityClass</code>指定的返回值
   */
  default <T> T exchange(@Nonnull String url, @Nonnull HttpMethod method, @Nonnull BodyType bodyType,
      @Nonnull ResponseType responseType, @Nonnull Class<T> responseEntityClass,
      @Nullable Map<String, String> httpHeaders, @Nullable Map<String, Object> uriVariables,
      @Nullable Object requestBody) {
    return exchange(url, method, bodyType, responseType, (Type) responseEntityClass,
        httpHeaders, uriVariables, requestBody);
  }

  /**
   * 自定义执行http请求
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param method 请求方法。如<code>GET, POST</code>等
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param typeReference 返回实体类型
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @param requestBody 请求body
   * @return 由<code>typeReference</code>指定的返回值
   */
  default <T> T exchange(@Nonnull String url, @Nonnull HttpMethod method, @Nonnull BodyType bodyType,
      @Nonnull ResponseType responseType, @Nonnull TypeReference<T> typeReference,
      @Nullable Map<String, String> httpHeaders, @Nullable Map<String, Object> uriVariables,
      @Nullable Object requestBody) {
    return exchange(url, method, bodyType, responseType, typeReference.getType(), httpHeaders, uriVariables,
        requestBody);
  }

  /**
   * 自定义执行http请求
   *
   * @param url 请求地址。如请求地址为<code>https://example.com/foo/bar</code>，则url应为<code>/foo/bar</code>
   * @param method 请求方法。如<code>GET, POST</code>等
   * @param bodyType 请求body类型
   * @param responseType response body类型
   * @param responseEntityType 返回实体类型 {@link TypeReference}
   * @param httpHeaders 请求header
   * @param uriVariables 查询参数
   * @param requestBody 请求body
   * @return 由<code>responseEntityType</code>指定的返回值
   */
  <T> T exchange(@Nonnull String url, @Nonnull HttpMethod method, @Nonnull BodyType bodyType,
      @Nonnull ResponseType responseType, @Nonnull Type responseEntityType, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables, @Nullable Object requestBody);

}
