/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.contract;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.Locale;

/**
 * http response
 */
public interface HttpResponse extends Closeable {

  /**
   * 获取 response 状态码，如<code>200,404</code>等
   */
  int getStatusCode();

  /**
   * 获取 response 原因，如<code>OK, Switching Protocols, Forbidden</code>等
   */
  String getReasonPhrase();

  boolean isChunked();

  /**
   * 获取response body的长度，如果获取失败则返回一个负数
   */
  long getContentLength();

  /**
   * 获取response body
   */
  InputStream getContent() throws IOException, UnsupportedEncodingException;

  /**
   * 获取response body
   *
   * @param charset 字符编码
   */
  String getContent(String charset) throws IOException, UnsupportedEncodingException;

  /**
   * 获取response body
   *
   * @param charset 字符编码
   */
  String getContent(Charset charset) throws IOException;

  /**
   * 设置response body
   */
  void setContent(InputStream inputStream, String mimeType, Charset charset);

  /**
   * 设置response body
   */
  void setContent(String content, String mimeType, Charset charset);

  /**
   * 设置response body
   */
  void setContent(byte[] data, String mimeType, Charset charset);

  /**
   * 获取header value
   *
   * @param headerName header name
   */
  String getHeaderValue(String headerName);

  /**
   * 获取header
   *
   * @param headerName header name
   * @return 可能为null
   */
  Header getFirstHeader(String headerName);

  /**
   * 获取header
   *
   * @param headerName header name
   * @return 所有匹配的header，返回的数组长&ge;0
   */
  Header[] getHeaders(String headerName);

  /**
   * 获取所有的header
   *
   * @return 所有header，返回的数组长&ge;0
   */
  Header[] getAllHeaders();

  /**
   * 获取<code>Content-Type</code>
   */
  String getContentType();

  /**
   * 获取当前response的语言
   *
   * @return 不会为null
   */
  Locale getLocale();

  /**
   * response body transfer to output stream
   *
   * @param outputStream 目标输出流
   */
  void transferTo(OutputStream outputStream) throws IOException;
}
