/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.contract.impl;

import com.ifengxue.http.contract.BasicHeader;
import com.ifengxue.http.contract.Header;
import com.ifengxue.http.contract.HttpResponse;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Locale;
import java.util.Optional;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;

/**
 * apache http client response
 *
 * @see CloseableHttpResponse
 */
public class HttpClientHttpResponse implements HttpResponse {

  private final CloseableHttpResponse httpResponse;

  public HttpClientHttpResponse(CloseableHttpResponse httpResponse) {
    this.httpResponse = httpResponse;
  }

  @Override
  public int getStatusCode() {
    return httpResponse.getStatusLine().getStatusCode();
  }

  @Override
  public String getReasonPhrase() {
    return httpResponse.getStatusLine().getReasonPhrase();
  }

  @Override
  public boolean isChunked() {
    return httpResponse.getEntity().isChunked();
  }

  @Override
  public long getContentLength() {
    return httpResponse.getEntity().getContentLength();
  }

  @Override
  public InputStream getContent() throws IOException, UnsupportedEncodingException {
    return httpResponse.getEntity().getContent();
  }

  @Override
  public String getContent(String charset) throws IOException, UnsupportedEncodingException {
    return EntityUtils.toString(httpResponse.getEntity(), charset);
  }

  @Override
  public String getContent(Charset charset) throws IOException {
    return EntityUtils.toString(httpResponse.getEntity(), charset);
  }

  @Override
  public void setContent(InputStream inputStream, String mimeType, Charset charset) {
    httpResponse.setEntity(new InputStreamEntity(inputStream, ContentType.create(mimeType, charset)));
  }

  @Override
  public void setContent(String content, String mimeType, Charset charset) {
    httpResponse.setEntity(new StringEntity(content, ContentType.create(mimeType, charset)));
  }

  @Override
  public void setContent(byte[] data, String mimeType, Charset charset) {
    httpResponse.setEntity(new ByteArrayEntity(data, ContentType.create(mimeType, charset)));
  }

  @Override
  public String getHeaderValue(String headerName) {
    return httpResponse.getFirstHeader(headerName).getValue();
  }

  @Override
  public Header getFirstHeader(String headerName) {
    return Optional.ofNullable(httpResponse.getFirstHeader(headerName))
        .map(h -> new BasicHeader(h.getName(), h.getValue()))
        .orElse(null);
  }

  @Override
  public Header[] getHeaders(String headerName) {
    return Arrays.stream(httpResponse.getHeaders(headerName))
        .map(h -> new BasicHeader(h.getName(), h.getValue()))
        .toArray(BasicHeader[]::new);
  }

  @Override
  public Header[] getAllHeaders() {
    return Arrays.stream(httpResponse.getAllHeaders())
        .map(h -> new BasicHeader(h.getName(), h.getValue()))
        .toArray(BasicHeader[]::new);
  }

  @Override
  public String getContentType() {
    return httpResponse.getEntity().getContentType().getValue();
  }

  @Override
  public Locale getLocale() {
    return httpResponse.getLocale();
  }

  @Override
  public void transferTo(OutputStream outputStream) throws IOException {
    httpResponse.getEntity().writeTo(outputStream);
  }

  @Override
  public void close() throws IOException {
    httpResponse.close();
  }
}
