/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.executor;

import com.ifengxue.http.HttpClientException;
import com.ifengxue.http.annotation.BodyType;
import com.ifengxue.http.annotation.ResponseType;
import com.ifengxue.http.contract.HttpResponse;
import com.ifengxue.http.contract.impl.HttpClientHttpResponse;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map.Entry;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;

/**
 * multipart/form-data请求执行器
 */
public class FormDataHttpExecutor extends AbstractHttpExecutor {

  @Override
  public HttpResponse execute(Request request, BodyType bodyType,
      ResponseType responseType) throws HttpClientException, IOException {
    MultipartEntityBuilder entityBuilder = MultipartEntityBuilder.create();
    for (Entry<String, Object> entry : request.getParameterMap().entrySet()) {
      String key = entry.getKey();
      Object value = entry.getValue();
      if (value instanceof File) {
        entityBuilder.addBinaryBody(key, (File) value);
        continue;
      }
      if (value instanceof InputStream) {
        entityBuilder.addBinaryBody(key, (InputStream) value);
        continue;
      }
      if (value instanceof byte[]) {
        entityBuilder.addBinaryBody(key, (byte[]) value);
        continue;
      }
      List<NameValuePair> nvps = ParameterUtil.createNameValueParis(key, value);
      for (NameValuePair nvp : nvps) {
        entityBuilder.addTextBody(key, nvp.getValue() == null ? "" : nvp.getValue(),
            ContentType.APPLICATION_FORM_URLENCODED.withCharset(charset));
      }
    }
    RequestBuilder requestBuilder = RequestBuilder.create(request.getMethod())
        .setUri(request.getUrl())
        .setCharset(charset)
        .setConfig(createConfig());
    requestBuilder.setEntity(entityBuilder.build());
    request.getHeaderMap().forEach(requestBuilder::addHeader);
    return new HttpClientHttpResponse(httpClient.execute(requestBuilder.build()));
  }
}
