/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.executor;

import com.ifengxue.http.HttpClientException;
import com.ifengxue.http.annotation.BodyType;
import com.ifengxue.http.util.IOUtil;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

public class HttpExecutorFactory {

  private static CloseableHttpClient httpClient;
  private static ExecutorService threadPool = ForkJoinPool.commonPool();

  /**
   * 获取http executor，如果没设置{@link #httpClient}，则自动创建一个http client
   */
  public HttpExecutor getHttpExecutor(BodyType bodyType) {
    if (httpClient == null) {
      synchronized (HttpExecutorFactory.class) {
        if (httpClient == null) {
          httpClient = HttpClients.createDefault();
        }
      }
    }
    return getHttpExecutor(bodyType, httpClient);
  }

  /**
   * 获取http executor
   *
   * @since 1.4.8
   */
  public HttpExecutor getHttpExecutor(BodyType bodyType, CloseableHttpClient httpClient) {
    HttpExecutor executor;
    switch (bodyType) {
      case FORM_DATA:
        executor = new FormDataHttpExecutor();
        break;
      case APPLICATION_JSON:
        executor = new JsonHttpExecutor();
        break;
      case APPLICATION_XML:
        executor = new XmlHttpExecutor();
        break;
      case X_WWW_FORM_URLENCODED:
        executor = new UrlEncodedHttpExecutor();
        break;
      default:
        throw new HttpClientException("暂不支持的请求类型:" + bodyType);
    }
    executor.setHttpClient(Objects.requireNonNull(httpClient));
    return executor;
  }

  /**
   * 获取线程池
   *
   * @since 1.5.0
   */
  public static ExecutorService getDefaultThreadPool() {
    return Optional.ofNullable(threadPool).orElseThrow(() -> new IllegalStateException("线程池不存在"));
  }

  /**
   * 设置线程池
   *
   * @since 1.5.0
   */
  @SuppressWarnings("unused")
  public static ExecutorService setDefaultThreadPool(ExecutorService threadPool) {
    ExecutorService oldThreadPool = HttpExecutorFactory.threadPool;
    HttpExecutorFactory.threadPool = threadPool;
    return oldThreadPool;
  }

  /**
   * 获取httpClient
   */
  @Deprecated
  public CloseableHttpClient getHttpClient() {
    return Optional.ofNullable(httpClient).orElseThrow(() -> new IllegalStateException("httpClient已经被关闭！"));
  }

  /**
   * 释放httpClient并重新赋值
   */
  @Deprecated
  public synchronized void setHttpClient(CloseableHttpClient httpClient) {
    Objects.requireNonNull(httpClient);
    close();
    HttpExecutorFactory.httpClient = httpClient;
  }

  /**
   * 释放httpClient
   */
  public static void close() {
    try {
      IOUtil.closeQuietly(httpClient);
    } finally {
      httpClient = null;
    }
  }

  /**
   * 释放{@link #httpClient}，并重新设置http client
   *
   * @since 1.4.8
   */
  @SuppressWarnings("unused")
  public synchronized void setDefaultHttpClient(CloseableHttpClient httpClient) {
    Objects.requireNonNull(httpClient);
    close();
    HttpExecutorFactory.httpClient = httpClient;
  }

  /**
   * 获取httpClient
   *
   * @since 1.4.8
   */
  @SuppressWarnings("unused")
  public CloseableHttpClient getDefaultHttClient() {
    return Objects.requireNonNull(httpClient, "没有默认的http client");
  }
}
