/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.parser;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

import com.ifengxue.http.HttpClientException;
import com.ifengxue.http.contract.Header;
import com.ifengxue.http.contract.HttpResponse;
import com.ifengxue.http.util.TypeUtil;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;

/**
 * http header parser
 */
public class HeaderHttpParser implements HttpParser {

  @Override
  @SuppressWarnings("unchecked")
  public <T> T parse(HttpResponse response, Charset charset, Type type) throws HttpClientException {
    if (TypeUtil.isVoidType(type)) {
      throw new UnsupportedTypeException("void is unsupported type");
    }
    Type actualType = TypeUtil.findActualType(type);
    if (actualType == Header[].class) {
      return (T) response.getAllHeaders();
    } else if (actualType instanceof ParameterizedType) {
      ParameterizedType parameterizedType = (ParameterizedType) actualType;
      Type rawType = parameterizedType.getRawType();
      if (!(rawType instanceof Class)) {
        throw new UnsupportedTypeException(type.getTypeName() + " is unsupported type");
      }
      Class<?> rawClass = (Class<?>) rawType;
      if (Collection.class.isAssignableFrom(rawClass)) {
        if (parameterizedType.getActualTypeArguments()[0] != Header.class) {
          throw new UnsupportedTypeException(
              parameterizedType.getActualTypeArguments()[0].getTypeName() + " is unsupported type");
        }
        return (T) Arrays.stream(response.getAllHeaders())
            .collect(toList());
      } else if (Map.class.isAssignableFrom(rawClass)) {
        for (Type actualTypeArgument : parameterizedType.getActualTypeArguments()) {
          if (actualTypeArgument != String.class) {
            throw new UnsupportedTypeException(actualTypeArgument.getTypeName() + " is unsupported type");
          }
        }
        return (T) Arrays.stream(response.getAllHeaders())
            .collect(toMap(Header::getName, Header::getValue));
      } else {
        throw new UnsupportedTypeException(rawType.getTypeName() + " is unsupported type");
      }
    } else {
      throw new UnsupportedTypeException(actualType.getTypeName() + " is unsupported type");
    }
  }
}
