/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.proxy;

import com.ifengxue.http.annotation.BodyType;
import com.ifengxue.http.annotation.HttpMethod;
import com.ifengxue.http.annotation.ResponseType;
import com.ifengxue.http.contract.HttpOperations;
import java.lang.reflect.Type;
import java.util.Map;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import lombok.experimental.Delegate;

/**
 * 一个简单快捷的http invoker
 *
 * @see HttpOperations
 */
public class GenericHttpInvoker implements HttpOperations, HttpClientConfig {

  private final HttpOperations httpOperations;
  @Delegate
  private final HttpClientConfig httpClientConfig;

  public GenericHttpInvoker(HttpOperations httpOperations, HttpClientConfig httpClientConfig) {
    this.httpOperations = httpOperations;
    this.httpClientConfig = httpClientConfig;
  }

  public static GenericHttpInvoker newInstance() {
    Object proxyObject = ProxyBuilder.newBuilder()
        .proxyInterface(Empty.class)
        .build();
    return new GenericHttpInvoker((HttpOperations) proxyObject, (HttpClientConfig) proxyObject);
  }

  /**
   * new instance
   *
   * @param socketTimeout 设置套接字的超时时间（等待数据响应的超时时间），即两个数据包之间可以间隔的最大时间。如果为0则永不超时，如果为负值则使用系统设置，默认为-1
   * @param connectTimeout 设置建立连接所允许的超时时间。如果为0则永不超时，如果为负值则使用系统设置，默认为-1
   */
  public static GenericHttpInvoker newInstance(int socketTimeout, int connectTimeout) {
    Object proxyObject = ProxyBuilder.newBuilder()
        .proxyInterface(Empty.class)
        .socketTimeout(socketTimeout)
        .connectTimeout(connectTimeout)
        .build();
    return new GenericHttpInvoker((HttpOperations) proxyObject, (HttpClientConfig) proxyObject);
  }

  /**
   * new instance
   *
   * @param host host
   * @param socketTimeout 设置套接字的超时时间（等待数据响应的超时时间），即两个数据包之间可以间隔的最大时间。如果为0则永不超时，如果为负值则使用系统设置，默认为-1
   * @param connectTimeout 设置建立连接所允许的超时时间。如果为0则永不超时，如果为负值则使用系统设置，默认为-1
   */
  public static GenericHttpInvoker newInstance(String host, int socketTimeout, int connectTimeout) {
    Object proxyObject = ProxyBuilder.newBuilder()
        .proxyInterface(Empty.class)
        .host(host)
        .socketTimeout(socketTimeout)
        .connectTimeout(connectTimeout)
        .build();
    return new GenericHttpInvoker((HttpOperations) proxyObject, (HttpClientConfig) proxyObject);
  }

  @Override
  public <T> T exchange(@Nonnull String url, @Nonnull HttpMethod method, @Nonnull BodyType bodyType,
      @Nonnull ResponseType responseType, @Nonnull Type responseEntityType, @Nullable Map<String, String> httpHeaders,
      @Nullable Map<String, Object> uriVariables, @Nullable Object requestBody) {
    return httpOperations
        .exchange(url, method, bodyType, responseType, responseEntityType, httpHeaders, uriVariables, requestBody);
  }

  /**
   * empty interface
   */
  private interface Empty {

  }
}
