/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package com.ifengxue.http.proxy;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.NoSuchElementException;
import java.util.concurrent.ExecutorService;
import org.apache.http.impl.client.CloseableHttpClient;

/**
 * http 客户端配置
 */
public interface HttpClientConfig {

  Charset UTF_8 = StandardCharsets.UTF_8;

  /**
   * 添加拦截器
   */
  void addInterceptor(Interceptor interceptor);

  /**
   * 批量移除拦截器
   */
  void removeInterceptors(Interceptor interceptor) throws NoSuchElementException;

  /**
   * 批量移除拦截器
   */
  void removeInterceptors(Class<? extends Interceptor> interceptorClass) throws NoSuchElementException;

  /**
   * 设置编码
   */
  void setCharset(Charset charset);

  /**
   * 设置user-agent
   */
  void setUserAgent(String userAgent);

  /**
   * 添加全局header
   */
  void addHeader(String name, String value);

  /**
   * 添加全局请求参数
   *
   * @param name 参数名称
   * @param value 参数值
   */
  void addParameter(String name, Object value);

  /**
   * 设置替换协议(http/https),主机(host),端口(port)的uri
   *
   * @param host 参数必须是http(s)://host:port，如果是80,443则可以省略
   */
  void setHost(String host);

  /**
   * 设置代理
   *
   * @param host 主机地址
   * @param port 端口
   * @param scheme 协议：如http
   */
  void setProxy(String host, int port, String scheme);

  /**
   * 设置套接字超时时间和建立请求超时时间
   *
   * @param socketTimeout 设置套接字的超时时间（等待数据响应的超时时间），即两个数据包之间可以间隔的最大时间。如果为0则永不超时，如果为负值则使用系统设置，默认为-1
   * @param connectTimeout 设置建立连接所允许的超时时间。如果为0则永不超时，如果为负值则使用系统设置，默认为-1
   */
  void setTimeout(int socketTimeout, int connectTimeout);

  /**
   * 使用自定义的http client
   *
   * @since 1.4.8
   */
  default void setHttpClient(CloseableHttpClient httpClient) {
    throw new UnsupportedOperationException();
  }

  /**
   * 使用自定义的线程池
   *
   * @since 1.5.0
   */
  default void setThreadPool(ExecutorService threadPool) {
    throw new UnsupportedOperationException();
  }

  /**
   * 设置http basic auth
   *
   * @param username 用户名
   * @param password 密码
   */
  default void setBasicAuth(String username, String password) {
    throw new UnsupportedOperationException();
  }

}
