/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.proxy;

import com.ifengxue.http.HttpClientException;
import com.ifengxue.http.annotation.BodyType;
import com.ifengxue.http.annotation.ResponseType;
import com.ifengxue.http.contract.HttpResponse;
import com.ifengxue.http.executor.HttpExecutor;
import com.ifengxue.http.executor.Request;
import com.ifengxue.http.parser.HttpParser;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.Arrays;
import lombok.extern.slf4j.Slf4j;

/**
 * 日志记录拦截器
 */
@Slf4j
public class LoggingInterceptor implements Interceptor {

  private static final String EXECUTE_BEGIN_TIME = "__execute_begin_time";
  private String lineSeparator = System.getProperty("line.separator");

  @Override
  public <T> T beforeRequest(Method method, Request request, BodyType bodyType,
      ResponseType responseType, HttpExecutor executor) throws HttpClientException {
    request.setAttribute(EXECUTE_BEGIN_TIME, System.currentTimeMillis());

    StringBuilder builder = new StringBuilder(1024);
    builder.append(lineSeparator).append("=========================Before Request Start=========================")
        .append(lineSeparator);
    builder.append("MethodName: ").append(method.getName()).append(lineSeparator);
    builder.append("HttpExecutor: ").append(executor.getClass().getName())
        .append(lineSeparator);
    builder.append(request.getMethod()).append(" ").append(request.getUrl()).append(lineSeparator);
    builder.append("BodyType: ").append(bodyType).append(lineSeparator);
    builder.append("ResponseType: ").append(responseType).append(lineSeparator);
    request.getHeaderMap().forEach(
        (name, value) -> builder.append(name).append(": ").append(value).append(lineSeparator));
    request.getParameterMap().forEach(
        (name, value) -> builder.append(name).append("=").append(value).append(lineSeparator));
    builder.append("=========================Before Request End=========================");
    log.info(builder.toString());
    return null;
  }

  @Override
  public <T> T beforeParse(Method method, Request request, BodyType bodyType,
      ResponseType responseType, HttpResponse httpResponse, HttpParser httpParser)
      throws HttpClientException, IOException {
    StringBuilder builder = new StringBuilder(1024);
    builder.append(lineSeparator).append("=========================Before Parse Start=========================")
        .append(lineSeparator);
    builder.append("MethodName: ").append(method.getName()).append(lineSeparator);
    builder.append("HttpParser: ").append(httpParser.getClass().getName())
        .append(lineSeparator);
    builder.append(request.getMethod()).append(" ").append(request.getUrl()).append(lineSeparator);
    builder.append("Status Code: ").append(httpResponse.getStatusCode())
        .append(lineSeparator);
    Arrays.stream(httpResponse.getAllHeaders()).forEach(
        header -> builder.append(header.getName()).append(": ").append(header.getValue())
            .append(lineSeparator));
    long executeBeginTime = request.getAttribute(EXECUTE_BEGIN_TIME);
    long executeEndTime = System.currentTimeMillis();
    builder.append("Execute time: ").append(executeEndTime - executeBeginTime).append(" ms").append(lineSeparator);
    builder.append("=========================Before Parse End=========================");
    log.info(builder.toString());
    return null;
  }
}
