/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.proxy;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import lombok.NonNull;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.impl.client.CloseableHttpClient;

/**
 * proxy builder
 */
public class ProxyBuilder {

  private static final int UNSET_INT_VALUE = -1;
  private ClassLoader classLoader;
  private Class<?> proxyInterface;
  private List<Interceptor> interceptors = new LinkedList<>();
  private Charset charset = StandardCharsets.UTF_8;
  private Map<String, String> headers = new HashMap<>();
  private Map<String, Object> parameters = new HashMap<>();
  private String userAgent;
  private String host;
  private String proxyScheme;
  private String proxyHost;
  private int proxyPort;
  private int socketTimeout = UNSET_INT_VALUE;
  private int connectTimeout = UNSET_INT_VALUE;
  private CloseableHttpClient httpClient;
  private ExecutorService threadPool;
  private UsernamePasswordCredentials basicAuthCredentials;

  public static ProxyBuilder newBuilder() {
    return new ProxyBuilder();
  }

  private ProxyBuilder() {
    classLoader = Thread.currentThread().getContextClassLoader();
    if (classLoader == null) {
      classLoader = getClass().getClassLoader();
    }
  }

  public ProxyBuilder classLoader(@NonNull ClassLoader classLoader) {
    this.classLoader = classLoader;
    return this;
  }

  public ProxyBuilder proxyInterface(@NonNull Class<?> proxyInterface) {
    this.proxyInterface = proxyInterface;
    return this;
  }

  public ProxyBuilder addInterceptor(@NonNull Interceptor interceptor) {
    interceptors.add(interceptor);
    return this;
  }

  public ProxyBuilder addInterceptors(@NonNull List<Interceptor> interceptors) {
    this.interceptors.addAll(interceptors);
    return this;
  }

  public ProxyBuilder charset(@NonNull Charset charset) {
    this.charset = charset;
    return this;
  }

  public ProxyBuilder addHeader(@NonNull String name, @NonNull String value) {
    headers.put(name, value);
    return this;
  }

  public ProxyBuilder addHeaders(@NonNull Map<String, String> headers) {
    this.headers.putAll(headers);
    return this;
  }

  public ProxyBuilder addParameter(@NonNull String name, Object value) {
    parameters.put(name, value);
    return this;
  }

  public ProxyBuilder addParameters(@NonNull Map<String, Object> parameters) {
    this.parameters.putAll(parameters);
    return this;
  }

  public ProxyBuilder userAgent(String userAgent) {
    this.userAgent = userAgent;
    return this;
  }

  public ProxyBuilder host(@NonNull String host) {
    this.host = host;
    return this;
  }

  public ProxyBuilder proxyScheme(@NonNull String proxyScheme) {
    this.proxyScheme = proxyScheme;
    return this;
  }

  public ProxyBuilder proxyHost(@NonNull String proxyHost) {
    this.proxyHost = proxyHost;
    return this;
  }

  public ProxyBuilder proxyPort(int proxyPort) {
    this.proxyPort = proxyPort;
    return this;
  }

  public ProxyBuilder socketTimeout(int socketTimeout) {
    this.socketTimeout = socketTimeout;
    return this;
  }

  public ProxyBuilder connectTimeout(int connectTimeout) {
    this.connectTimeout = connectTimeout;
    return this;
  }

  public ProxyBuilder httpClient(@NonNull CloseableHttpClient httpClient) {
    this.httpClient = httpClient;
    return this;
  }

  public ProxyBuilder threadPool(@NonNull ExecutorService threadPool) {
    this.threadPool = threadPool;
    return this;
  }

  public ProxyBuilder basicAuth(@NonNull String username, String password) {
    this.basicAuthCredentials = new UsernamePasswordCredentials(username, password);
    return this;
  }

  @SuppressWarnings("unchecked")
  public <T> T build() {
    Object proxyObject = RequestInvoker.create(proxyInterface, classLoader);

    // config
    HttpClientConfig httpClientConfig = (HttpClientConfig) proxyObject;
    interceptors.forEach(httpClientConfig::addInterceptor);
    Optional.ofNullable(charset).ifPresent(httpClientConfig::setCharset);
    headers.forEach(httpClientConfig::addHeader);
    parameters.forEach(httpClientConfig::addParameter);
    Optional.ofNullable(userAgent).ifPresent(httpClientConfig::setUserAgent);
    Optional.ofNullable(host).ifPresent(httpClientConfig::setHost);
    Optional.ofNullable(proxyHost).ifPresent(ph -> httpClientConfig.setProxy(proxyHost, proxyPort, proxyScheme));
    if (UNSET_INT_VALUE != socketTimeout || UNSET_INT_VALUE != connectTimeout) {
      httpClientConfig.setTimeout(socketTimeout, connectTimeout);
    }
    if (httpClient != null) {
      httpClientConfig.setHttpClient(httpClient);
    }
    if (threadPool != null) {
      httpClientConfig.setThreadPool(threadPool);
    }
    if (basicAuthCredentials != null) {
      httpClientConfig.setBasicAuth(basicAuthCredentials.getUserName(), basicAuthCredentials.getPassword());
    }
    return (T) proxyObject;
  }
}