/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.util;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;

public enum TypeUtil {
  ;
  private static Set<Type> supportAsyncTypes = new HashSet<>();
  private static Set<Type> builtinTypes = new HashSet<>();

  static {
    supportAsyncTypes.add(Future.class);
    supportAsyncTypes.add(CompletableFuture.class);
    builtinTypes.addAll(supportAsyncTypes);
  }

  /**
   * 检测type是否是void类型，支持的情形为
   * <ul>
   *   <li>void</li>
   *   <li>Void</li>
   *   <li>内置泛型类型为Void，如CompletableFuture&lt;Void&gt;</li>
   * </ul>
   */
  public static boolean isVoidType(Type rawType) {
    return isPrimitiveType(rawType, Void.class);
  }

  /**
   * 解析字符串为基本数据类型的值
   */
  @SuppressWarnings("unchecked")
  public static <T> T resolvePrimitiveTypeValue(String value, Type primitiveType) {
    if (primitiveType == Integer.class || primitiveType == int.class) {
      return (T) Integer.valueOf(value);
    }
    if (primitiveType == Long.class || primitiveType == long.class) {
      return (T) Long.valueOf(value);
    }
    if (primitiveType == Float.class || primitiveType == float.class) {
      return (T) Float.valueOf(value);
    }
    if (primitiveType == Double.class || primitiveType == double.class) {
      return (T) Double.valueOf(value);
    }
    if (primitiveType == Boolean.class || primitiveType == boolean.class) {
      return (T) Boolean.valueOf(value);
    }
    if (primitiveType == Short.class || primitiveType == short.class) {
      return (T) Short.valueOf(value);
    }
    if (primitiveType == Byte.class || primitiveType == byte.class) {
      return (T) Byte.valueOf(value);
    }
    if (primitiveType == Character.class || primitiveType == char.class) {
      if (value.length() != 1) {
        throw new IllegalStateException("can't convert " + value + " to char");
      }
      return (T) Character.valueOf(value.charAt(0));
    }
    throw new IllegalStateException("not primitive type " + primitiveType.getTypeName());
  }

  /**
   * 检测type是否是基本数据类型，支持的情形为
   * <ul>
   *   <li>基本数据类型</li>
   *   <li>基本数据类型的封装类型</li>
   *   <li>内置泛型类型为基本数据类型的封装类型，如CompletableFuture&lt;Integer&gt;</li>
   * </ul>
   *
   * @param rawType 原始类型
   * @see #builtinTypes
   */
  public static boolean isPrimitiveType(Type rawType) {
    return isPrimitiveType(rawType, Integer.class)
        || isPrimitiveType(rawType, Long.class)
        || isPrimitiveType(rawType, Float.class)
        || isPrimitiveType(rawType, Double.class)
        || isPrimitiveType(rawType, Boolean.class)
        || isPrimitiveType(rawType, Short.class)
        || isPrimitiveType(rawType, Byte.class)
        || isPrimitiveType(rawType, Character.class)
        ;
  }

  /**
   * 检测type是否是基本数据类型，支持的情形为
   * <ul>
   *   <li>基本数据类型</li>
   *   <li>基本数据类型的封装类型</li>
   *   <li>内置泛型类型为基本数据类型的封装类型，如CompletableFuture&lt;Integer&gt;</li>
   * </ul>
   *
   * @param rawType 原始类型
   * @param primitiveWrapperType 基本数据类型的封装类型
   * @see #builtinTypes
   */
  public static boolean isPrimitiveType(Type rawType, Class<?> primitiveWrapperType) {
    try {
      Field typeField = primitiveWrapperType.getField("TYPE");
      Class<?> primitiveType = (Class<?>) typeField.get(primitiveWrapperType);
      if (rawType == primitiveType || rawType == primitiveWrapperType) {
        return true;
      }
      return isTypeOf(rawType, primitiveWrapperType);
    } catch (NoSuchFieldException e) {
      throw new IllegalStateException(primitiveWrapperType.getName() + " not a primitive wrapper class.", e);
    } catch (ReflectiveOperationException e) {
      throw new IllegalStateException(e);
    }
  }

  /**
   * 检测当前type是否是异步type
   *
   * @see Future
   * @see CompletableFuture
   */
  public static boolean isAsyncType(Type rawType) {
    if (supportAsyncTypes.contains(rawType)) {
      return true;
    }
    if (rawType instanceof ParameterizedType) {
      return supportAsyncTypes.contains(((ParameterizedType) rawType).getRawType());
    }
    return false;
  }

  /**
   * 查找真实的数据类型
   * <ul>
   *   <li>如果rawType == Class，则真实的数据类型为rawType</li>
   *   <li>如果rawType是泛型类型，且泛型容器为内置数据类型，则真实数据类型为其泛型，否则真实数据类型为rawType</li>
   * </ul>
   */
  public static Type findActualType(Type rawType) {
    if (rawType instanceof Class) {
      return rawType;
    }
    if (rawType instanceof ParameterizedType) {
      ParameterizedType parameterizedType = (ParameterizedType) rawType;
      if (builtinTypes.contains(parameterizedType.getRawType())) {
        return parameterizedType.getActualTypeArguments()[0];
      }
    }
    return rawType;
  }

  /**
   * 检测type是否是指定的数据类型，支持的情形为
   * <ul>
   *   <li>rawType == targetType</li>
   *   <li>rawType为内置泛型类型，targetType为泛型，如CompletableFuture&lt;Integer&gt;</li>
   *   <li>rawType 为 targetType 的子类，如rawType=InputStream.class，targetType=FileInputStream.class</li>
   *   <li>rawType为内置泛型类型，targetType为具体类型的子类，如rawType=CompletableFuture&lt;InputStream&gt;，targetType=FileInputStream.class</li>
   * </ul>
   *
   * @see #builtinTypes
   */
  public static boolean isTypeOf(Type rawType, Type targetType) {
    if (rawType == targetType) {
      return true;
    }
    if (rawType instanceof ParameterizedType) {
      ParameterizedType parameterizedType = (ParameterizedType) rawType;
      if (!builtinTypes.contains(parameterizedType.getRawType())) {
        return false;
      }
      Type type = parameterizedType.getActualTypeArguments()[0];
      if (type == targetType) {
        return true;
      }
      if (type instanceof Class && targetType instanceof Class) {
        return ((Class<?>) type).isAssignableFrom((Class<?>) targetType);
      }
      return false;
    }
    if (rawType instanceof Class && targetType instanceof Class) {
      return ((Class<?>) rawType).isAssignableFrom((Class<?>) targetType);
    }
    return false;
  }
}
