/*
 * Copyright 2019 https://www.ifengxue.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ifengxue.http.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.Properties;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Version {

  /**
   * 默认的版本号名称
   */
  private static final String DEFAULT_VERSION = "1.6.0";
  /**
   * jar包名称正则表达式样例:signature-sdk-1.1.2-SNAPSHOT.jar,signature-sdk-1.1.2-RELEASE.jar,signature-sdk-1.1.2.jar
   */
  private static final String DEFAULT_REGEX = "((\\d+\\.?)+([._-](RELEASE|SNAPSHOT))?).jar";
  /**
   * 版本号
   */
  public static final String VERSION = getVersion();

  public static String getVersion() {
    return getVersion(Version.class, DEFAULT_REGEX, DEFAULT_VERSION);
  }

  public static String getVersion(Class<?> clazz) {
    return getVersion(clazz, DEFAULT_REGEX, DEFAULT_VERSION);
  }

  /**
   * @param clazz 目标类
   * @param regex 从jar文件中匹配版本号 {@link #DEFAULT_REGEX}
   * @param defVer 默认版本号 {@link #DEFAULT_VERSION}
   */
  public static String getVersion(Class<?> clazz, String regex, String defVer) {
    try {
      String version = clazz.getPackage().getImplementationVersion();
      if (isNotEmpty(version)) {
        return version;
      }
      version = getVersionFromMaven(clazz);
      if (isNotEmpty(version)) {
        return version;
      }
      version = getVersionFromJarName(clazz, regex);
      return isNotEmpty(version) ? version : defVer;
    } catch (Exception e) {
      return defVer;
    }
  }

  /**
   * 从pom.properties中读取版本号
   */
  private static String getVersionFromMaven(Class<?> clazz) throws IOException {
    String filePath = clazz.getProtectionDomain().getCodeSource().getLocation().getFile();
    if (isEmpty(filePath)) {
      return null;
    }
    File file = new File(filePath);
    if (!file.exists()) {
      return null;
    }
    InputStream is = null;
    try {
      JarFile jarFile = new JarFile(file);
      Enumeration<JarEntry> entries = jarFile.entries();
      while (entries.hasMoreElements()) {
        JarEntry entry = entries.nextElement();
        String name = entry.getName();
        if (name.startsWith("META-INF/maven") && name.endsWith("pom.properties")) {
          is = clazz.getClassLoader().getResourceAsStream(name);
          Properties prop = new Properties();
          prop.load(is);
          return prop.getProperty("version");
        }
      }
      return null;
    } catch (IOException e) {
      return null;
    } finally {
      if (is != null) {
        is.close();
      }
    }
  }

  /**
   * 从jar文件中使用正则表达式解析版本号
   *
   * @param clazz 类
   * @param regex 正则表达式
   */
  private static String getVersionFromJarName(Class<?> clazz, String regex) {
    if (regex == null) {
      return null;
    }
    String filePath = clazz.getProtectionDomain().getCodeSource().getLocation().getFile();
    if (isEmpty(filePath)) {
      return null;
    }
    File file = new File(filePath);
    if (!file.exists()) {
      return null;
    }
    String filename = file.getName();
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(filename);
    if (matcher.find()) {
      return matcher.group(1);
    }
    return null;
  }

  private static boolean isNotEmpty(String text) {
    return !isEmpty(text);
  }

  private static boolean isEmpty(String text) {
    return text == null || text.trim().isEmpty();
  }
}