/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */

package org.easysearch.client;

import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpPut;
import org.easysearch.action.admin.cluster.health.ClusterHealthRequest;
import org.easysearch.action.admin.cluster.settings.ClusterGetSettingsRequest;
import org.easysearch.action.admin.cluster.settings.ClusterUpdateSettingsRequest;
import org.easysearch.action.support.ActiveShardCount;
import org.easysearch.client.cluster.RemoteInfoRequest;
import org.easysearch.client.indices.ComponentTemplatesExistRequest;
import org.easysearch.client.indices.DeleteComponentTemplateRequest;
import org.easysearch.client.indices.GetComponentTemplatesRequest;
import org.easysearch.client.indices.PutComponentTemplateRequest;
import org.easysearch.common.Strings;

import java.io.IOException;

final class ClusterRequestConverters {

    private ClusterRequestConverters() {}

    static Request clusterPutSettings(ClusterUpdateSettingsRequest clusterUpdateSettingsRequest) throws IOException {
        Request request = new Request(HttpPut.METHOD_NAME, "/_cluster/settings");

        RequestConverters.Params parameters = new RequestConverters.Params();
        parameters.withTimeout(clusterUpdateSettingsRequest.timeout());
        parameters.withMasterTimeout(clusterUpdateSettingsRequest.masterNodeTimeout());
        request.addParameters(parameters.asMap());
        request.setEntity(RequestConverters.createEntity(clusterUpdateSettingsRequest, RequestConverters.REQUEST_BODY_CONTENT_TYPE));
        return request;
    }

    static Request clusterGetSettings(ClusterGetSettingsRequest clusterGetSettingsRequest) throws IOException {
        Request request = new Request(HttpGet.METHOD_NAME, "/_cluster/settings");
        RequestConverters.Params parameters = new RequestConverters.Params();
        parameters.withLocal(clusterGetSettingsRequest.local());
        parameters.withIncludeDefaults(clusterGetSettingsRequest.includeDefaults());
        parameters.withMasterTimeout(clusterGetSettingsRequest.masterNodeTimeout());
        request.addParameters(parameters.asMap());
        return request;
    }

    static Request clusterHealth(ClusterHealthRequest healthRequest) {
        String[] indices = healthRequest.indices() == null ? Strings.EMPTY_ARRAY : healthRequest.indices();
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_cluster/health")
            .addCommaSeparatedPathParts(indices)
            .build();
        Request request = new Request(HttpGet.METHOD_NAME, endpoint);

        RequestConverters.Params params = new RequestConverters.Params()
            .withWaitForStatus(healthRequest.waitForStatus())
            .withWaitForNoRelocatingShards(healthRequest.waitForNoRelocatingShards())
            .withWaitForNoInitializingShards(healthRequest.waitForNoInitializingShards())
            .withWaitForActiveShards(healthRequest.waitForActiveShards(), ActiveShardCount.NONE)
            .withWaitForNodes(healthRequest.waitForNodes())
            .withWaitForEvents(healthRequest.waitForEvents())
            .withTimeout(healthRequest.timeout())
            .withMasterTimeout(healthRequest.masterNodeTimeout())
            .withLocal(healthRequest.local())
            .withLevel(healthRequest.level());
        request.addParameters(params.asMap());
        return request;
    }

    static Request remoteInfo(RemoteInfoRequest remoteInfoRequest) {
        return new Request(HttpGet.METHOD_NAME, "/_remote/info");
    }

    static Request putComponentTemplate(PutComponentTemplateRequest putComponentTemplateRequest) throws IOException {
        String endpoint = new RequestConverters.EndpointBuilder().addPathPartAsIs("_component_template")
            .addPathPart(putComponentTemplateRequest.name()).build();
        Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        RequestConverters.Params params = new RequestConverters.Params();
        params.withMasterTimeout(putComponentTemplateRequest.masterNodeTimeout());
        if (putComponentTemplateRequest.create()) {
            params.putParam("create", Boolean.TRUE.toString());
        }
        if (Strings.hasText(putComponentTemplateRequest.cause())) {
            params.putParam("cause", putComponentTemplateRequest.cause());
        }
        request.addParameters(params.asMap());
        request.setEntity(RequestConverters.createEntity(putComponentTemplateRequest, RequestConverters.REQUEST_BODY_CONTENT_TYPE));
        return request;
    }

    static Request getComponentTemplates(GetComponentTemplatesRequest getComponentTemplatesRequest){
        final String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_component_template")
            .addPathPart(getComponentTemplatesRequest.name())
            .build();
        final Request request = new Request(HttpGet.METHOD_NAME, endpoint);
        final RequestConverters.Params params = new RequestConverters.Params();
        params.withLocal(getComponentTemplatesRequest.isLocal());
        params.withMasterTimeout(getComponentTemplatesRequest.getMasterNodeTimeout());
        request.addParameters(params.asMap());
        return request;
    }

    static Request componentTemplatesExist(ComponentTemplatesExistRequest componentTemplatesRequest) {
        final String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_component_template")
            .addPathPart(componentTemplatesRequest.name())
            .build();
        final Request request = new Request(HttpHead.METHOD_NAME, endpoint);
        final RequestConverters.Params params = new RequestConverters.Params();
        params.withLocal(componentTemplatesRequest.isLocal());
        params.withMasterTimeout(componentTemplatesRequest.getMasterNodeTimeout());
        request.addParameters(params.asMap());
        return request;
    }

    static Request deleteComponentTemplate(DeleteComponentTemplateRequest deleteComponentTemplateRequest) {
        String name = deleteComponentTemplateRequest.getName();
        String endpoint = new RequestConverters.EndpointBuilder().addPathPartAsIs("_component_template").addPathPart(name).build();
        Request request = new Request(HttpDelete.METHOD_NAME, endpoint);
        RequestConverters.Params params = new RequestConverters.Params();
        params.withMasterTimeout(deleteComponentTemplateRequest.masterNodeTimeout());
        request.addParameters(params.asMap());
        return request;
    }
}
