/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */
package org.easysearch.client;

import org.easysearch.EasysearchException;
import org.easysearch.action.support.nodes.BaseNodesResponse;
import org.easysearch.common.Nullable;
import org.easysearch.common.ParseField;
import org.easysearch.common.xcontent.ConstructingObjectParser;
import org.easysearch.common.xcontent.ToXContent;
import org.easysearch.common.xcontent.XContentBuilder;
import org.easysearch.common.xcontent.XContentParser;
import org.easysearch.rest.action.RestActions;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * A utility class to parse the Nodes Header returned by
 * {@link RestActions#buildNodesHeader(XContentBuilder, ToXContent.Params, BaseNodesResponse)}.
 */
public final class NodesResponseHeader {

    public static final ParseField TOTAL = new ParseField("total");
    public static final ParseField SUCCESSFUL = new ParseField("successful");
    public static final ParseField FAILED = new ParseField("failed");
    public static final ParseField FAILURES = new ParseField("failures");

    @SuppressWarnings("unchecked")
    public static final ConstructingObjectParser<NodesResponseHeader, Void> PARSER =
        new ConstructingObjectParser<>("nodes_response_header", true,
            (a) -> {
                int i = 0;
                int total = (Integer) a[i++];
                int successful = (Integer) a[i++];
                int failed = (Integer) a[i++];
                List<EasysearchException> failures = (List<EasysearchException>) a[i++];
                return new NodesResponseHeader(total, successful, failed, failures);
            });

    static {
        PARSER.declareInt(ConstructingObjectParser.constructorArg(), TOTAL);
        PARSER.declareInt(ConstructingObjectParser.constructorArg(), SUCCESSFUL);
        PARSER.declareInt(ConstructingObjectParser.constructorArg(), FAILED);
        PARSER.declareObjectArray(ConstructingObjectParser.optionalConstructorArg(),
            (p, c) -> EasysearchException.fromXContent(p), FAILURES);
    }

    private final int total;
    private final int successful;
    private final int failed;
    private final List<EasysearchException> failures;

    public NodesResponseHeader(int total, int successful, int failed, @Nullable List<EasysearchException> failures) {
        this.total = total;
        this.successful = successful;
        this.failed = failed;
        this.failures = failures == null ? Collections.emptyList() : failures;
    }

    public static NodesResponseHeader fromXContent(XContentParser parser, Void context) throws IOException {
        return PARSER.parse(parser, context);
    }

    /** the total number of nodes that the operation was carried on */
    public int getTotal() {
        return total;
    }

    /** the number of nodes that the operation has failed on */
    public int getFailed() {
        return failed;
    }

    /** the number of nodes that the operation was successful on */
    public int getSuccessful() {
        return successful;
    }

    /**
     * Get the failed node exceptions.
     *
     * @return Never {@code null}. Can be empty.
     */
    public List<EasysearchException> getFailures() {
        return failures;
    }

    /**
     * Determine if there are any node failures in {@link #failures}.
     *
     * @return {@code true} if {@link #failures} contains at least 1 exception.
     */
    public boolean hasFailures() {
        return failures.isEmpty() == false;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        NodesResponseHeader that = (NodesResponseHeader) o;
        return total == that.total &&
            successful == that.successful &&
            failed == that.failed &&
            Objects.equals(failures, that.failures);
    }

    @Override
    public int hashCode() {
        return Objects.hash(total, successful, failed, failures);
    }

}
