/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */

package org.easysearch.client;

import org.apache.http.HttpEntity;
import org.apache.http.util.EntityUtils;
import org.easysearch.EasysearchException;
import org.easysearch.EasysearchStatusException;
import org.easysearch.action.ActionListener;
import org.easysearch.action.ActionRequest;
import org.easysearch.action.ActionRequestValidationException;
import org.easysearch.action.admin.cluster.node.tasks.list.ListTasksResponse;
import org.easysearch.action.admin.cluster.storedscripts.DeleteStoredScriptRequest;
import org.easysearch.action.admin.cluster.storedscripts.GetStoredScriptRequest;
import org.easysearch.action.admin.cluster.storedscripts.GetStoredScriptResponse;
import org.easysearch.action.admin.cluster.storedscripts.PutStoredScriptRequest;
import org.easysearch.action.bulk.BulkRequest;
import org.easysearch.action.bulk.BulkResponse;
import org.easysearch.action.delete.DeleteRequest;
import org.easysearch.action.delete.DeleteResponse;
import org.easysearch.action.explain.ExplainRequest;
import org.easysearch.action.explain.ExplainResponse;
import org.easysearch.action.fieldcaps.FieldCapabilitiesRequest;
import org.easysearch.action.fieldcaps.FieldCapabilitiesResponse;
import org.easysearch.action.get.GetRequest;
import org.easysearch.action.get.GetResponse;
import org.easysearch.action.get.MultiGetRequest;
import org.easysearch.action.get.MultiGetResponse;
import org.easysearch.action.index.IndexRequest;
import org.easysearch.action.index.IndexResponse;
import org.easysearch.action.search.ClearScrollRequest;
import org.easysearch.action.search.ClearScrollResponse;
import org.easysearch.action.search.MultiSearchRequest;
import org.easysearch.action.search.MultiSearchResponse;
import org.easysearch.action.search.SearchRequest;
import org.easysearch.action.search.SearchResponse;
import org.easysearch.action.search.SearchScrollRequest;
import org.easysearch.action.support.master.AcknowledgedResponse;
import org.easysearch.action.update.UpdateRequest;
import org.easysearch.action.update.UpdateResponse;
//import org.easysearch.client.analytics.InferencePipelineAggregationBuilder;
//import org.easysearch.client.analytics.ParsedInference;
import org.easysearch.client.analytics.ParsedStringStats;
import org.easysearch.client.analytics.ParsedTopMetrics;
import org.easysearch.client.analytics.StringStatsAggregationBuilder;
import org.easysearch.client.analytics.TopMetricsAggregationBuilder;
import org.easysearch.client.core.CountRequest;
import org.easysearch.client.core.CountResponse;
import org.easysearch.client.core.GetSourceRequest;
import org.easysearch.client.core.GetSourceResponse;
import org.easysearch.client.core.MainRequest;
import org.easysearch.client.core.MainResponse;
import org.easysearch.client.core.MultiTermVectorsRequest;
import org.easysearch.client.core.MultiTermVectorsResponse;
import org.easysearch.client.core.TermVectorsRequest;
import org.easysearch.client.core.TermVectorsResponse;
import org.easysearch.client.tasks.TaskSubmissionResponse;
import org.easysearch.common.CheckedConsumer;
import org.easysearch.common.CheckedFunction;
import org.easysearch.common.ParseField;
import org.easysearch.common.Strings;
import org.easysearch.common.xcontent.*;
import org.easysearch.index.rankeval.RankEvalRequest;
import org.easysearch.index.rankeval.RankEvalResponse;
import org.easysearch.index.reindex.BulkByScrollResponse;
import org.easysearch.index.reindex.DeleteByQueryRequest;
import org.easysearch.index.reindex.ReindexRequest;
import org.easysearch.index.reindex.UpdateByQueryRequest;
import org.easysearch.plugins.spi.NamedXContentProvider;
import org.easysearch.rest.BytesRestResponse;
import org.easysearch.rest.RestStatus;
import org.easysearch.script.mustache.MultiSearchTemplateRequest;
import org.easysearch.script.mustache.MultiSearchTemplateResponse;
import org.easysearch.script.mustache.SearchTemplateRequest;
import org.easysearch.script.mustache.SearchTemplateResponse;
import org.easysearch.search.aggregations.Aggregation;
import org.easysearch.search.aggregations.bucket.adjacency.AdjacencyMatrixAggregationBuilder;
import org.easysearch.search.aggregations.bucket.adjacency.ParsedAdjacencyMatrix;
import org.easysearch.search.aggregations.bucket.composite.CompositeAggregationBuilder;
import org.easysearch.search.aggregations.bucket.composite.ParsedComposite;
import org.easysearch.search.aggregations.bucket.filter.FilterAggregationBuilder;
import org.easysearch.search.aggregations.bucket.filter.FiltersAggregationBuilder;
import org.easysearch.search.aggregations.bucket.filter.ParsedFilter;
import org.easysearch.search.aggregations.bucket.filter.ParsedFilters;
import org.easysearch.search.aggregations.bucket.geogrid.GeoHashGridAggregationBuilder;
import org.easysearch.search.aggregations.bucket.geogrid.GeoTileGridAggregationBuilder;
import org.easysearch.search.aggregations.bucket.geogrid.ParsedGeoHashGrid;
import org.easysearch.search.aggregations.bucket.geogrid.ParsedGeoTileGrid;
import org.easysearch.search.aggregations.bucket.global.GlobalAggregationBuilder;
import org.easysearch.search.aggregations.bucket.global.ParsedGlobal;
import org.easysearch.search.aggregations.bucket.histogram.AutoDateHistogramAggregationBuilder;
import org.easysearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.easysearch.search.aggregations.bucket.histogram.HistogramAggregationBuilder;
import org.easysearch.search.aggregations.bucket.histogram.ParsedAutoDateHistogram;
import org.easysearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.easysearch.search.aggregations.bucket.histogram.ParsedHistogram;
import org.easysearch.search.aggregations.bucket.histogram.ParsedVariableWidthHistogram;
import org.easysearch.search.aggregations.bucket.histogram.VariableWidthHistogramAggregationBuilder;
import org.easysearch.search.aggregations.bucket.missing.MissingAggregationBuilder;
import org.easysearch.search.aggregations.bucket.missing.ParsedMissing;
import org.easysearch.search.aggregations.bucket.nested.NestedAggregationBuilder;
import org.easysearch.search.aggregations.bucket.nested.ParsedNested;
import org.easysearch.search.aggregations.bucket.nested.ParsedReverseNested;
import org.easysearch.search.aggregations.bucket.nested.ReverseNestedAggregationBuilder;
import org.easysearch.search.aggregations.bucket.range.DateRangeAggregationBuilder;
import org.easysearch.search.aggregations.bucket.range.GeoDistanceAggregationBuilder;
import org.easysearch.search.aggregations.bucket.range.IpRangeAggregationBuilder;
import org.easysearch.search.aggregations.bucket.range.ParsedBinaryRange;
import org.easysearch.search.aggregations.bucket.range.ParsedDateRange;
import org.easysearch.search.aggregations.bucket.range.ParsedGeoDistance;
import org.easysearch.search.aggregations.bucket.range.ParsedRange;
import org.easysearch.search.aggregations.bucket.range.RangeAggregationBuilder;
import org.easysearch.search.aggregations.bucket.sampler.InternalSampler;
import org.easysearch.search.aggregations.bucket.sampler.ParsedSampler;
import org.easysearch.search.aggregations.bucket.terms.LongRareTerms;
import org.easysearch.search.aggregations.bucket.terms.ParsedLongRareTerms;
import org.easysearch.search.aggregations.bucket.terms.ParsedSignificantLongTerms;
import org.easysearch.search.aggregations.bucket.terms.ParsedSignificantStringTerms;
import org.easysearch.search.aggregations.bucket.terms.ParsedStringRareTerms;
import org.easysearch.search.aggregations.bucket.terms.SignificantLongTerms;
import org.easysearch.search.aggregations.bucket.terms.SignificantStringTerms;
import org.easysearch.search.aggregations.bucket.terms.DoubleTerms;
import org.easysearch.search.aggregations.bucket.terms.LongTerms;
import org.easysearch.search.aggregations.bucket.terms.ParsedDoubleTerms;
import org.easysearch.search.aggregations.bucket.terms.ParsedLongTerms;
import org.easysearch.search.aggregations.bucket.terms.ParsedStringTerms;
import org.easysearch.search.aggregations.bucket.terms.StringRareTerms;
import org.easysearch.search.aggregations.bucket.terms.StringTerms;
import org.easysearch.search.aggregations.metrics.AvgAggregationBuilder;
import org.easysearch.search.aggregations.metrics.CardinalityAggregationBuilder;
import org.easysearch.search.aggregations.metrics.ExtendedStatsAggregationBuilder;
import org.easysearch.search.aggregations.metrics.GeoBoundsAggregationBuilder;
import org.easysearch.search.aggregations.metrics.GeoCentroidAggregationBuilder;
import org.easysearch.search.aggregations.metrics.InternalHDRPercentileRanks;
import org.easysearch.search.aggregations.metrics.InternalHDRPercentiles;
import org.easysearch.search.aggregations.metrics.InternalTDigestPercentileRanks;
import org.easysearch.search.aggregations.metrics.InternalTDigestPercentiles;
import org.easysearch.search.aggregations.metrics.MaxAggregationBuilder;
import org.easysearch.search.aggregations.metrics.MedianAbsoluteDeviationAggregationBuilder;
import org.easysearch.search.aggregations.metrics.MinAggregationBuilder;
import org.easysearch.search.aggregations.metrics.ParsedAvg;
import org.easysearch.search.aggregations.metrics.ParsedCardinality;
import org.easysearch.search.aggregations.metrics.ParsedExtendedStats;
import org.easysearch.search.aggregations.metrics.ParsedGeoBounds;
import org.easysearch.search.aggregations.metrics.ParsedGeoCentroid;
import org.easysearch.search.aggregations.metrics.ParsedHDRPercentileRanks;
import org.easysearch.search.aggregations.metrics.ParsedHDRPercentiles;
import org.easysearch.search.aggregations.metrics.ParsedMax;
import org.easysearch.search.aggregations.metrics.ParsedMedianAbsoluteDeviation;
import org.easysearch.search.aggregations.metrics.ParsedMin;
import org.easysearch.search.aggregations.metrics.ParsedScriptedMetric;
import org.easysearch.search.aggregations.metrics.ParsedStats;
import org.easysearch.search.aggregations.metrics.ParsedSum;
import org.easysearch.search.aggregations.metrics.ParsedTDigestPercentileRanks;
import org.easysearch.search.aggregations.metrics.ParsedTDigestPercentiles;
import org.easysearch.search.aggregations.metrics.ParsedTopHits;
import org.easysearch.search.aggregations.metrics.ParsedValueCount;
import org.easysearch.search.aggregations.metrics.ParsedWeightedAvg;
import org.easysearch.search.aggregations.metrics.ScriptedMetricAggregationBuilder;
import org.easysearch.search.aggregations.metrics.StatsAggregationBuilder;
import org.easysearch.search.aggregations.metrics.SumAggregationBuilder;
import org.easysearch.search.aggregations.metrics.TopHitsAggregationBuilder;
import org.easysearch.search.aggregations.metrics.ValueCountAggregationBuilder;
import org.easysearch.search.aggregations.metrics.WeightedAvgAggregationBuilder;
import org.easysearch.search.aggregations.pipeline.DerivativePipelineAggregationBuilder;
import org.easysearch.search.aggregations.pipeline.ExtendedStatsBucketPipelineAggregationBuilder;
import org.easysearch.search.aggregations.pipeline.InternalBucketMetricValue;
import org.easysearch.search.aggregations.pipeline.InternalSimpleValue;
import org.easysearch.search.aggregations.pipeline.ParsedBucketMetricValue;
import org.easysearch.search.aggregations.pipeline.ParsedDerivative;
import org.easysearch.search.aggregations.pipeline.ParsedExtendedStatsBucket;
import org.easysearch.search.aggregations.pipeline.ParsedPercentilesBucket;
import org.easysearch.search.aggregations.pipeline.ParsedSimpleValue;
import org.easysearch.search.aggregations.pipeline.ParsedStatsBucket;
import org.easysearch.search.aggregations.pipeline.PercentilesBucketPipelineAggregationBuilder;
import org.easysearch.search.aggregations.pipeline.StatsBucketPipelineAggregationBuilder;
import org.easysearch.search.suggest.Suggest;
import org.easysearch.search.suggest.completion.CompletionSuggestion;
import org.easysearch.search.suggest.completion.CompletionSuggestionBuilder;
import org.easysearch.search.suggest.phrase.PhraseSuggestion;
import org.easysearch.search.suggest.phrase.PhraseSuggestionBuilder;
import org.easysearch.search.suggest.term.TermSuggestion;
import org.easysearch.search.suggest.term.TermSuggestionBuilder;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;

/**
 * High level REST client that wraps an instance of the low level {@link RestClient} and allows to build requests and read responses. The
 * {@link RestClient} instance is internally built based on the provided {@link RestClientBuilder} and it gets closed automatically when
 * closing the {@link RestHighLevelClient} instance that wraps it.
 * <p>
 *
 * In case an already existing instance of a low-level REST client needs to be provided, this class can be subclassed and the
 * {@link #RestHighLevelClient(RestClient, CheckedConsumer, List)} constructor can be used.
 * <p>
 *
 * This class can also be sub-classed to expose additional client methods that make use of endpoints added to Easysearch through plugins,
 * or to add support for custom response sections, again added to Easysearch through plugins.
 * <p>
 *
 * The majority of the methods in this class come in two flavors, a blocking and an asynchronous version (e.g.
 * {@link #search(SearchRequest, RequestOptions)} and {@link #searchAsync(SearchRequest, RequestOptions, ActionListener)}, where the later
 * takes an implementation of an {@link ActionListener} as an argument that needs to implement methods that handle successful responses and
 * failure scenarios. Most of the blocking calls can throw an {@link IOException} or an unchecked {@link EasysearchException} in the
 * following cases:
 *
 * <ul>
 * <li>an {@link IOException} is usually thrown in case of failing to parse the REST response in the high-level REST client, the request
 * times out or similar cases where there is no response coming back from the Easysearch server</li>
 * <li>an {@link EasysearchException} is usually thrown in case where the server returns a 4xx or 5xx error code. The high-level client
 * then tries to parse the response body error details into a generic EasysearchException and suppresses the original
 * {@link ResponseException}</li>
 * </ul>
 *
 */
public class RestHighLevelClient implements Closeable {

    private final RestClient client;
    private final NamedXContentRegistry registry;
    private final CheckedConsumer<RestClient, IOException> doClose;

    private final IndicesClient indicesClient = new IndicesClient(this);
    private final ClusterClient clusterClient = new ClusterClient(this);
    private final IngestClient ingestClient = new IngestClient(this);
    private final SnapshotClient snapshotClient = new SnapshotClient(this);
    private final TasksClient tasksClient = new TasksClient(this);
    private final SecurityClient securityClient = new SecurityClient(this);
    //private final IndexLifecycleClient ilmClient = new IndexLifecycleClient(this);
    //private final CcrClient ccrClient = null;
    //private final AsyncSearchClient asyncSearchClient = new AsyncSearchClient(this);

    /**
     * Creates a {@link RestHighLevelClient} given the low level {@link RestClientBuilder} that allows to build the
     * {@link RestClient} to be used to perform requests.
     */
    public RestHighLevelClient(RestClientBuilder restClientBuilder) {
        this(restClientBuilder, Collections.emptyList());
    }

    /**
     * Creates a {@link RestHighLevelClient} given the low level {@link RestClientBuilder} that allows to build the
     * {@link RestClient} to be used to perform requests and parsers for custom response sections added to Easysearch through plugins.
     */
    protected RestHighLevelClient(RestClientBuilder restClientBuilder, List<NamedXContentRegistry.Entry> namedXContentEntries) {
        this(restClientBuilder.build(), RestClient::close, namedXContentEntries);
    }

    /**
     * Creates a {@link RestHighLevelClient} given the low level {@link RestClient} that it should use to perform requests and
     * a list of entries that allow to parse custom response sections added to Easysearch through plugins.
     * This constructor can be called by subclasses in case an externally created low-level REST client needs to be provided.
     * The consumer argument allows to control what needs to be done when the {@link #close()} method is called.
     * Also subclasses can provide parsers for custom response sections added to Easysearch through plugins.
     */
    protected RestHighLevelClient(RestClient restClient, CheckedConsumer<RestClient, IOException> doClose,
                                  List<NamedXContentRegistry.Entry> namedXContentEntries) {
        this.client = Objects.requireNonNull(restClient, "restClient must not be null");
        this.doClose = Objects.requireNonNull(doClose, "doClose consumer must not be null");
        this.registry = new NamedXContentRegistry(
                Stream.of(getDefaultNamedXContents().stream(), getProvidedNamedXContents().stream(), namedXContentEntries.stream())
                    .flatMap(Function.identity()).collect(toList()));
    }

    /**
     * Returns the low-level client that the current high-level client instance is using to perform requests
     */
    public final RestClient getLowLevelClient() {
        return client;
    }

    @Override
    public final void close() throws IOException {
        doClose.accept(client);
    }

    /**
     * Provides an {@link IndicesClient} which can be used to access the Indices API.
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/indices.html">Indices API on elastic.co</a>
     */
    public final IndicesClient indices() {
        return indicesClient;
    }

    /**
     * Provides a {@link ClusterClient} which can be used to access the Cluster API.
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/cluster.html">Cluster API on elastic.co</a>
     */
    public final ClusterClient cluster() {
        return clusterClient;
    }

    /**
     * Provides a {@link IngestClient} which can be used to access the Ingest API.
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/ingest.html">Ingest API on elastic.co</a>
     */
    public final IngestClient ingest() {
        return ingestClient;
    }

    /**
     * Provides a {@link SnapshotClient} which can be used to access the Snapshot API.
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/modules-snapshots.html">Snapshot API on elastic.co</a>
     */
    public final SnapshotClient snapshot() {
        return snapshotClient;
    }


//    public final CcrClient ccr() {
//        return ccrClient;
//    }

    /**
     * Provides a {@link TasksClient} which can be used to access the Tasks API.
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/tasks.html">Task Management API on elastic.co</a>
     */
    public final TasksClient tasks() {
        return tasksClient;
    }


//    public IndexLifecycleClient indexLifecycle() {
//        return ilmClient;
//    }


//    public AsyncSearchClient asyncSearch() {
//        return asyncSearchClient;
//    }


    /**
     * Provide Security APIs.
     *
     * @return the client wrapper for making Security API calls
     */
    public SecurityClient security() {
        return securityClient;
    }

    /**
     * Executes a bulk request using the Bulk API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-bulk.html">Bulk API on elastic.co</a>
     * @param bulkRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final BulkResponse bulk(BulkRequest bulkRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(bulkRequest, RequestConverters::bulk, options, BulkResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously executes a bulk request using the Bulk API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-bulk.html">Bulk API on elastic.co</a>
     * @param bulkRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable bulkAsync(BulkRequest bulkRequest, RequestOptions options, ActionListener<BulkResponse> listener) {
        return performRequestAsyncAndParseEntity(bulkRequest, RequestConverters::bulk, options,
            BulkResponse::fromXContent, listener, emptySet());
    }

    /**
     * Executes a reindex request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-reindex.html">Reindex API on elastic.co</a>
     * @param reindexRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final BulkByScrollResponse reindex(ReindexRequest reindexRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(
            reindexRequest, RequestConverters::reindex, options, BulkByScrollResponse::fromXContent, singleton(409)
        );
    }

    /**
     * Submits a reindex task.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-reindex.html">Reindex API on elastic.co</a>
     * @param reindexRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the submission response
     */
    public final TaskSubmissionResponse submitReindexTask(ReindexRequest reindexRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(
            reindexRequest, RequestConverters::submitReindex, options, TaskSubmissionResponse::fromXContent, emptySet()
        );
    }

    /**
     * Asynchronously executes a reindex request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-reindex.html">Reindex API on elastic.co</a>
     * @param reindexRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable reindexAsync(ReindexRequest reindexRequest, RequestOptions options,
                                          ActionListener<BulkByScrollResponse> listener) {
        return performRequestAsyncAndParseEntity(
            reindexRequest, RequestConverters::reindex, options, BulkByScrollResponse::fromXContent, listener, singleton(409)
        );
    }

    /**
     * Executes a update by query request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-update-by-query.html">
     *     Update By Query API on elastic.co</a>
     * @param updateByQueryRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final BulkByScrollResponse updateByQuery(UpdateByQueryRequest updateByQueryRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(
            updateByQueryRequest, RequestConverters::updateByQuery, options, BulkByScrollResponse::fromXContent, singleton(409)
        );
    }

    /**
     * Submits a update by query task.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-update-by-query.html">
     *     Update By Query API on elastic.co</a>
     * @param updateByQueryRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the submission response
     */
    public final TaskSubmissionResponse submitUpdateByQueryTask(UpdateByQueryRequest updateByQueryRequest,
                                                                RequestOptions options) throws IOException {
        return performRequestAndParseEntity(
            updateByQueryRequest, RequestConverters::submitUpdateByQuery, options, TaskSubmissionResponse::fromXContent, emptySet()
        );
    }

    /**
     * Asynchronously executes an update by query request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-update-by-query.html">
     *     Update By Query API on elastic.co</a>
     * @param updateByQueryRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable updateByQueryAsync(UpdateByQueryRequest updateByQueryRequest, RequestOptions options,
                                                ActionListener<BulkByScrollResponse> listener) {
        return performRequestAsyncAndParseEntity(
            updateByQueryRequest, RequestConverters::updateByQuery, options, BulkByScrollResponse::fromXContent, listener, singleton(409)
        );
    }

    /**
     * Executes a delete by query request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-delete-by-query.html">
     *     Delete By Query API on elastic.co</a>
     * @param deleteByQueryRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final BulkByScrollResponse deleteByQuery(DeleteByQueryRequest deleteByQueryRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(
            deleteByQueryRequest, RequestConverters::deleteByQuery, options, BulkByScrollResponse::fromXContent, singleton(409)
        );
    }

    /**
     * Submits a delete by query task
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-delete-by-query.html">
     *      Delete By Query API on elastic.co</a>
     * @param deleteByQueryRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the submission response
     */
    public final TaskSubmissionResponse submitDeleteByQueryTask(DeleteByQueryRequest deleteByQueryRequest,
                                                                RequestOptions options) throws IOException {
        return performRequestAndParseEntity(
            deleteByQueryRequest, RequestConverters::submitDeleteByQuery, options, TaskSubmissionResponse::fromXContent, emptySet()
        );
    }

    /**
     * Asynchronously executes a delete by query request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-delete-by-query.html">
     *     Delete By Query API on elastic.co</a>
     * @param deleteByQueryRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable deleteByQueryAsync(DeleteByQueryRequest deleteByQueryRequest, RequestOptions options,
                                                ActionListener<BulkByScrollResponse> listener) {
        return performRequestAsyncAndParseEntity(
            deleteByQueryRequest, RequestConverters::deleteByQuery, options, BulkByScrollResponse::fromXContent, listener, singleton(409)
        );
    }

    /**
     * Executes a delete by query rethrottle request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-delete-by-query.html">
     *     Delete By Query API on elastic.co</a>
     * @param rethrottleRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final ListTasksResponse deleteByQueryRethrottle(RethrottleRequest rethrottleRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(rethrottleRequest, RequestConverters::rethrottleDeleteByQuery, options,
                ListTasksResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously execute an delete by query rethrottle request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-delete-by-query.html">
     *     Delete By Query API on elastic.co</a>
     * @param rethrottleRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable deleteByQueryRethrottleAsync(RethrottleRequest rethrottleRequest, RequestOptions options,
                                                          ActionListener<ListTasksResponse> listener) {
        return performRequestAsyncAndParseEntity(rethrottleRequest, RequestConverters::rethrottleDeleteByQuery, options,
                ListTasksResponse::fromXContent, listener, emptySet());
    }

    /**
     * Executes a update by query rethrottle request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-update-by-query.html">
     *     Update By Query API on elastic.co</a>
     * @param rethrottleRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final ListTasksResponse updateByQueryRethrottle(RethrottleRequest rethrottleRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(rethrottleRequest, RequestConverters::rethrottleUpdateByQuery, options,
                ListTasksResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously execute an update by query rethrottle request.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-update-by-query.html">
     *     Update By Query API on elastic.co</a>
     * @param rethrottleRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable updateByQueryRethrottleAsync(RethrottleRequest rethrottleRequest, RequestOptions options,
                                                          ActionListener<ListTasksResponse> listener) {
        return performRequestAsyncAndParseEntity(rethrottleRequest, RequestConverters::rethrottleUpdateByQuery, options,
                ListTasksResponse::fromXContent, listener, emptySet());
    }

    /**
     * Executes a reindex rethrottling request.
     * See the <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-reindex.html#docs-reindex-rethrottle">
     * Reindex rethrottling API on elastic.co</a>
     *
     * @param rethrottleRequest the request
     * @param options           the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final ListTasksResponse reindexRethrottle(RethrottleRequest rethrottleRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(rethrottleRequest, RequestConverters::rethrottleReindex, options,
                ListTasksResponse::fromXContent, emptySet());
    }

    /**
     * Executes a reindex rethrottling request.
     * See the <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-reindex.html#docs-reindex-rethrottle">
     * Reindex rethrottling API on elastic.co</a>
     * @param rethrottleRequest the request
     * @param options           the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener          the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable reindexRethrottleAsync(RethrottleRequest rethrottleRequest, RequestOptions options,
                                                    ActionListener<ListTasksResponse> listener) {
        return performRequestAsyncAndParseEntity(rethrottleRequest,
            RequestConverters::rethrottleReindex, options, ListTasksResponse::fromXContent, listener, emptySet());
    }

    /**
     * Pings the remote Easysearch cluster and returns true if the ping succeeded, false otherwise
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return <code>true</code> if the ping succeeded, false otherwise
     */
    public final boolean ping(RequestOptions options) throws IOException {
        return performRequest(new MainRequest(), (request) -> RequestConverters.ping(), options, RestHighLevelClient::convertExistsResponse,
                emptySet());
    }

    /**
     * Get the cluster info otherwise provided when sending an HTTP request to '/'
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final MainResponse info(RequestOptions options) throws IOException {
        return performRequestAndParseEntity(new MainRequest(), (request) -> RequestConverters.info(), options,
                MainResponse::fromXContent, emptySet());
    }

    /**
     * Retrieves a document by id using the Get API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html">Get API on elastic.co</a>
     * @param getRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final GetResponse get(GetRequest getRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(getRequest, RequestConverters::get, options, GetResponse::fromXContent, singleton(404));
    }

    /**
     * Asynchronously retrieves a document by id using the Get API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html">Get API on elastic.co</a>
     * @param getRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable getAsync(GetRequest getRequest, RequestOptions options, ActionListener<GetResponse> listener) {
        return performRequestAsyncAndParseEntity(getRequest, RequestConverters::get, options, GetResponse::fromXContent, listener,
                singleton(404));
    }

    /**
     * Retrieves multiple documents by id using the Multi Get API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-multi-get.html">Multi Get API on elastic.co</a>
     * @param multiGetRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     * @deprecated use {@link #mget(MultiGetRequest, RequestOptions)} instead
     */
    @Deprecated
    public final MultiGetResponse multiGet(MultiGetRequest multiGetRequest, RequestOptions options) throws IOException {
        return mget(multiGetRequest, options);
    }


    /**
     * Retrieves multiple documents by id using the Multi Get API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-multi-get.html">Multi Get API on elastic.co</a>
     * @param multiGetRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final MultiGetResponse mget(MultiGetRequest multiGetRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(multiGetRequest, RequestConverters::multiGet, options, MultiGetResponse::fromXContent,
                singleton(404));
    }

    /**
     * Asynchronously retrieves multiple documents by id using the Multi Get API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-multi-get.html">Multi Get API on elastic.co</a>
     * @param multiGetRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @deprecated use {@link #mgetAsync(MultiGetRequest, RequestOptions, ActionListener)} instead
     * @return cancellable that may be used to cancel the request
     */
    @Deprecated
    public final Cancellable multiGetAsync(MultiGetRequest multiGetRequest, RequestOptions options,
                                           ActionListener<MultiGetResponse> listener) {
        return mgetAsync(multiGetRequest, options, listener);
    }

    /**
     * Asynchronously retrieves multiple documents by id using the Multi Get API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-multi-get.html">Multi Get API on elastic.co</a>
     * @param multiGetRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable mgetAsync(MultiGetRequest multiGetRequest, RequestOptions options,
                                       ActionListener<MultiGetResponse> listener) {
        return performRequestAsyncAndParseEntity(multiGetRequest, RequestConverters::multiGet, options,
            MultiGetResponse::fromXContent, listener, singleton(404));
    }

    /**
     * Checks for the existence of a document. Returns true if it exists, false otherwise.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html">Get API on elastic.co</a>
     * @param getRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return <code>true</code> if the document exists, <code>false</code> otherwise
     */
    public final boolean exists(GetRequest getRequest, RequestOptions options) throws IOException {
        return performRequest(getRequest, RequestConverters::exists, options, RestHighLevelClient::convertExistsResponse, emptySet());
    }

    /**
     * Asynchronously checks for the existence of a document. Returns true if it exists, false otherwise.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html">Get API on elastic.co</a>
     * @param getRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable existsAsync(GetRequest getRequest, RequestOptions options, ActionListener<Boolean> listener) {
        return performRequestAsync(getRequest, RequestConverters::exists, options, RestHighLevelClient::convertExistsResponse, listener,
                emptySet());
    }

    /**
     * Checks for the existence of a document with a "_source" field. Returns true if it exists, false otherwise.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html#_source">Source exists API
     * on elastic.co</a>
     * @param getRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return <code>true</code> if the document and _source field exists, <code>false</code> otherwise
     * @deprecated use {@link #existsSource(GetSourceRequest, RequestOptions)} instead
     */
    @Deprecated
    public boolean existsSource(GetRequest getRequest, RequestOptions options) throws IOException {
        GetSourceRequest getSourceRequest = GetSourceRequest.from(getRequest);
        return performRequest(getSourceRequest, RequestConverters::sourceExists, options,
            RestHighLevelClient::convertExistsResponse, emptySet());
    }

    /**
     * Asynchronously checks for the existence of a document with a "_source" field. Returns true if it exists, false otherwise.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html#_source">Source exists API
     * on elastic.co</a>
     * @param getRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     * @deprecated use {@link #existsSourceAsync(GetSourceRequest, RequestOptions, ActionListener)} instead
     */
    @Deprecated
    public final Cancellable existsSourceAsync(GetRequest getRequest, RequestOptions options, ActionListener<Boolean> listener) {
        GetSourceRequest getSourceRequest = GetSourceRequest.from(getRequest);
        return performRequestAsync(getSourceRequest, RequestConverters::sourceExists, options,
            RestHighLevelClient::convertExistsResponse, listener, emptySet());
    }

    /**
     * Checks for the existence of a document with a "_source" field. Returns true if it exists, false otherwise.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html#_source">Source exists API
     * on elastic.co</a>
     * @param getSourceRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return <code>true</code> if the document and _source field exists, <code>false</code> otherwise
     */
    public boolean existsSource(GetSourceRequest getSourceRequest, RequestOptions options) throws IOException {
        return performRequest(getSourceRequest, RequestConverters::sourceExists, options,
            RestHighLevelClient::convertExistsResponse, emptySet());
    }

    /**
     * Asynchronously checks for the existence of a document with a "_source" field. Returns true if it exists, false otherwise.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html#_source">Source exists API
     * on elastic.co</a>
     * @param getSourceRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable existsSourceAsync(GetSourceRequest getSourceRequest, RequestOptions options,
                                               ActionListener<Boolean> listener) {
        return performRequestAsync(getSourceRequest, RequestConverters::sourceExists, options,
            RestHighLevelClient::convertExistsResponse, listener, emptySet());
    }

    /**
     * Retrieves the source field only of a document using GetSource API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html#_source">Get Source API
     * on elastic.co</a>
     * @param getSourceRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public GetSourceResponse getSource(GetSourceRequest getSourceRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(getSourceRequest, RequestConverters::getSource, options,
            GetSourceResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously retrieves the source field only of a document using GetSource API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-get.html#_source">Get Source API
     * on elastic.co</a>
     * @param getSourceRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable getSourceAsync(GetSourceRequest getSourceRequest, RequestOptions options,
                                            ActionListener<GetSourceResponse> listener) {
        return performRequestAsyncAndParseEntity(getSourceRequest, RequestConverters::getSource, options,
            GetSourceResponse::fromXContent, listener, emptySet());
    }

    /**
     * Index a document using the Index API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-index_.html">Index API on elastic.co</a>
     * @param indexRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final IndexResponse index(IndexRequest indexRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(indexRequest, RequestConverters::index, options,
            IndexResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously index a document using the Index API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-index_.html">Index API on elastic.co</a>
     * @param indexRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable indexAsync(IndexRequest indexRequest, RequestOptions options, ActionListener<IndexResponse> listener) {
        return performRequestAsyncAndParseEntity(indexRequest, RequestConverters::index, options, IndexResponse::fromXContent, listener,
                emptySet());
    }

    /**
     * Executes a count request using the Count API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-count.html">Count API on elastic.co</a>
     * @param countRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final CountResponse count(CountRequest countRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(countRequest, RequestConverters::count, options, CountResponse::fromXContent,
        emptySet());
    }

    /**
     * Asynchronously executes a count request using the Count API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-count.html">Count API on elastic.co</a>
     * @param countRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable countAsync(CountRequest countRequest, RequestOptions options, ActionListener<CountResponse> listener) {
        return performRequestAsyncAndParseEntity(countRequest, RequestConverters::count,  options,CountResponse::fromXContent,
            listener, emptySet());
    }

    /**
     * Updates a document using the Update API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-update.html">Update API on elastic.co</a>
     * @param updateRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final UpdateResponse update(UpdateRequest updateRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(updateRequest, RequestConverters::update, options, UpdateResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously updates a document using the Update API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-update.html">Update API on elastic.co</a>
     * @param updateRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable updateAsync(UpdateRequest updateRequest, RequestOptions options, ActionListener<UpdateResponse> listener) {
        return performRequestAsyncAndParseEntity(updateRequest, RequestConverters::update, options, UpdateResponse::fromXContent, listener,
                emptySet());
    }

    /**
     * Deletes a document by id using the Delete API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-delete.html">Delete API on elastic.co</a>
     * @param deleteRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final DeleteResponse delete(DeleteRequest deleteRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(deleteRequest, RequestConverters::delete, options, DeleteResponse::fromXContent,
                singleton(404));
    }

    /**
     * Asynchronously deletes a document by id using the Delete API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-delete.html">Delete API on elastic.co</a>
     * @param deleteRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable deleteAsync(DeleteRequest deleteRequest, RequestOptions options, ActionListener<DeleteResponse> listener) {
        return performRequestAsyncAndParseEntity(deleteRequest, RequestConverters::delete, options, DeleteResponse::fromXContent, listener,
            Collections.singleton(404));
    }

    /**
     * Executes a search request using the Search API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-search.html">Search API on elastic.co</a>
     * @param searchRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final SearchResponse search(SearchRequest searchRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(
                searchRequest,
                r -> RequestConverters.search(r, "_search"),
                options,
                SearchResponse::fromXContent,
                emptySet());
    }

    /**
     * Asynchronously executes a search using the Search API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-search.html">Search API on elastic.co</a>
     * @param searchRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable searchAsync(SearchRequest searchRequest, RequestOptions options, ActionListener<SearchResponse> listener) {
        return performRequestAsyncAndParseEntity(
                searchRequest,
                r -> RequestConverters.search(r, "_search"),
                options,
                SearchResponse::fromXContent,
                listener,
                emptySet());
    }

    /**
     * Executes a multi search using the msearch API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-multi-search.html">Multi search API on
     * elastic.co</a>
     * @param multiSearchRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     * @deprecated use {@link #msearch(MultiSearchRequest, RequestOptions)} instead
     */
    @Deprecated
    public final MultiSearchResponse multiSearch(MultiSearchRequest multiSearchRequest, RequestOptions options) throws IOException {
        return msearch(multiSearchRequest, options);
    }

    /**
     * Executes a multi search using the msearch API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-multi-search.html">Multi search API on
     * elastic.co</a>
     * @param multiSearchRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final MultiSearchResponse msearch(MultiSearchRequest multiSearchRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(multiSearchRequest, RequestConverters::multiSearch, options, MultiSearchResponse::fromXContext,
                emptySet());
    }

    /**
     * Asynchronously executes a multi search using the msearch API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-multi-search.html">Multi search API on
     * elastic.co</a>
     * @param searchRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @deprecated use {@link #msearchAsync(MultiSearchRequest, RequestOptions, ActionListener)} instead
     * @return cancellable that may be used to cancel the request
     */
    @Deprecated
    public final Cancellable multiSearchAsync(MultiSearchRequest searchRequest, RequestOptions options,
                                              ActionListener<MultiSearchResponse> listener) {
        return msearchAsync(searchRequest, options, listener);
    }

    /**
     * Asynchronously executes a multi search using the msearch API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-multi-search.html">Multi search API on
     * elastic.co</a>
     * @param searchRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable msearchAsync(MultiSearchRequest searchRequest, RequestOptions options,
                                          ActionListener<MultiSearchResponse> listener) {
        return performRequestAsyncAndParseEntity(searchRequest, RequestConverters::multiSearch, options, MultiSearchResponse::fromXContext,
                listener, emptySet());
    }

    /**
     * Executes a search using the Search Scroll API.
     * See <a
     * href="https://www.elastic.co/guide/en/easysearch/reference/master/search-request-body.html#request-body-search-scroll">Search
     * Scroll API on elastic.co</a>
     * @param searchScrollRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     * @deprecated use {@link #scroll(SearchScrollRequest, RequestOptions)} instead
     */
    @Deprecated
    public final SearchResponse searchScroll(SearchScrollRequest searchScrollRequest, RequestOptions options) throws IOException {
        return scroll(searchScrollRequest, options);
    }

    /**
     * Executes a search using the Search Scroll API.
     * See <a
     * href="https://www.elastic.co/guide/en/easysearch/reference/master/search-request-body.html#request-body-search-scroll">Search
     * Scroll API on elastic.co</a>
     * @param searchScrollRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final SearchResponse scroll(SearchScrollRequest searchScrollRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(searchScrollRequest, RequestConverters::searchScroll, options, SearchResponse::fromXContent,
                emptySet());
    }

    /**
     * Asynchronously executes a search using the Search Scroll API.
     * See <a
     * href="https://www.elastic.co/guide/en/easysearch/reference/master/search-request-body.html#request-body-search-scroll">Search
     * Scroll API on elastic.co</a>
     * @param searchScrollRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @deprecated use {@link #scrollAsync(SearchScrollRequest, RequestOptions, ActionListener)} instead
     * @return cancellable that may be used to cancel the request
     */
    @Deprecated
    public final Cancellable searchScrollAsync(SearchScrollRequest searchScrollRequest, RequestOptions options,
                                               ActionListener<SearchResponse> listener) {
        return scrollAsync(searchScrollRequest, options, listener);
    }

    /**
     * Asynchronously executes a search using the Search Scroll API.
     * See <a
     * href="https://www.elastic.co/guide/en/easysearch/reference/master/search-request-body.html#request-body-search-scroll">Search
     * Scroll API on elastic.co</a>
     * @param searchScrollRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable scrollAsync(SearchScrollRequest searchScrollRequest, RequestOptions options,
                                         ActionListener<SearchResponse> listener) {
        return performRequestAsyncAndParseEntity(searchScrollRequest, RequestConverters::searchScroll,
            options, SearchResponse::fromXContent, listener, emptySet());
    }

    /**
     * Clears one or more scroll ids using the Clear Scroll API.
     * See <a
     * href="https://www.elastic.co/guide/en/easysearch/reference/master/search-request-body.html#_clear_scroll_api">
     * Clear Scroll API on elastic.co</a>
     * @param clearScrollRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final ClearScrollResponse clearScroll(ClearScrollRequest clearScrollRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(clearScrollRequest, RequestConverters::clearScroll, options, ClearScrollResponse::fromXContent,
                emptySet());
    }

    /**
     * Asynchronously clears one or more scroll ids using the Clear Scroll API.
     * See <a
     * href="https://www.elastic.co/guide/en/easysearch/reference/master/search-request-body.html#_clear_scroll_api">
     * Clear Scroll API on elastic.co</a>
     * @param clearScrollRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable clearScrollAsync(ClearScrollRequest clearScrollRequest, RequestOptions options,
                                              ActionListener<ClearScrollResponse> listener) {
        return performRequestAsyncAndParseEntity(clearScrollRequest, RequestConverters::clearScroll,
            options, ClearScrollResponse::fromXContent, listener, emptySet());
    }

    /**
     * Executes a request using the Search Template API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-template.html">Search Template API
     * on elastic.co</a>.
     * @param searchTemplateRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final SearchTemplateResponse searchTemplate(SearchTemplateRequest searchTemplateRequest,
                                                       RequestOptions options) throws IOException {
        return performRequestAndParseEntity(searchTemplateRequest, RequestConverters::searchTemplate, options,
            SearchTemplateResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously executes a request using the Search Template API.
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-template.html">Search Template API
     * on elastic.co</a>.
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable searchTemplateAsync(SearchTemplateRequest searchTemplateRequest, RequestOptions options,
                                                 ActionListener<SearchTemplateResponse> listener) {
        return performRequestAsyncAndParseEntity(searchTemplateRequest, RequestConverters::searchTemplate, options,
            SearchTemplateResponse::fromXContent, listener, emptySet());
    }

    /**
     * Executes a request using the Explain API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-explain.html">Explain API on elastic.co</a>
     * @param explainRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final ExplainResponse explain(ExplainRequest explainRequest, RequestOptions options) throws IOException {
        return performRequest(explainRequest, RequestConverters::explain, options,
            response -> {
                CheckedFunction<XContentParser, ExplainResponse, IOException> entityParser =
                    parser -> ExplainResponse.fromXContent(parser, convertExistsResponse(response));
                return parseEntity(response.getEntity(), entityParser);
            },
            singleton(404));
    }

    /**
     * Asynchronously executes a request using the Explain API.
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-explain.html">Explain API on elastic.co</a>
     * @param explainRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable explainAsync(ExplainRequest explainRequest, RequestOptions options, ActionListener<ExplainResponse> listener) {
        return performRequestAsync(explainRequest, RequestConverters::explain, options,
            response -> {
                CheckedFunction<XContentParser, ExplainResponse, IOException> entityParser =
                    parser -> ExplainResponse.fromXContent(parser, convertExistsResponse(response));
                return parseEntity(response.getEntity(), entityParser);
            },
            listener, singleton(404));
    }


    /**
     * Calls the Term Vectors API
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-termvectors.html">Term Vectors API on
     * elastic.co</a>
     *
     * @param request   the request
     * @param options   the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     */
    public final TermVectorsResponse termvectors(TermVectorsRequest request, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(request, RequestConverters::termVectors, options, TermVectorsResponse::fromXContent,
            emptySet());
    }

    /**
     * Asynchronously calls the Term Vectors API
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-termvectors.html">Term Vectors API on
     * elastic.co</a>
     * @param request the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable termvectorsAsync(TermVectorsRequest request, RequestOptions options,
                                              ActionListener<TermVectorsResponse> listener) {
        return performRequestAsyncAndParseEntity(request, RequestConverters::termVectors, options,
            TermVectorsResponse::fromXContent, listener,
            emptySet());
    }


    /**
     * Calls the Multi Term Vectors API
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-multi-termvectors.html">Multi Term Vectors API
     * on elastic.co</a>
     *
     * @param request   the request
     * @param options   the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     */
    public final MultiTermVectorsResponse mtermvectors(MultiTermVectorsRequest request, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(
            request, RequestConverters::mtermVectors, options, MultiTermVectorsResponse::fromXContent, emptySet());
    }


    /**
     * Asynchronously calls the Multi Term Vectors API
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/docs-multi-termvectors.html">Multi Term Vectors API
     * on elastic.co</a>
     * @param request the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable mtermvectorsAsync(MultiTermVectorsRequest request, RequestOptions options,
                                               ActionListener<MultiTermVectorsResponse> listener) {
        return performRequestAsyncAndParseEntity(
            request, RequestConverters::mtermVectors, options, MultiTermVectorsResponse::fromXContent, listener, emptySet());
    }


    /**
     * Executes a request using the Ranking Evaluation API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-rank-eval.html">Ranking Evaluation API
     * on elastic.co</a>
     * @param rankEvalRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final RankEvalResponse rankEval(RankEvalRequest rankEvalRequest, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(rankEvalRequest, RequestConverters::rankEval, options, RankEvalResponse::fromXContent,
                emptySet());
    }


    /**
     * Executes a request using the Multi Search Template API.
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/multi-search-template.html">Multi Search Template API
     * on elastic.co</a>.
     */
    public final MultiSearchTemplateResponse msearchTemplate(MultiSearchTemplateRequest multiSearchTemplateRequest,
                                                             RequestOptions options) throws IOException {
        return performRequestAndParseEntity(multiSearchTemplateRequest, RequestConverters::multiSearchTemplate,
                options, MultiSearchTemplateResponse::fromXContext, emptySet());
    }

    /**
     * Asynchronously executes a request using the Multi Search Template API
     *
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/multi-search-template.html">Multi Search Template API
     * on elastic.co</a>.
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable msearchTemplateAsync(MultiSearchTemplateRequest multiSearchTemplateRequest,
                                                  RequestOptions options,
                                                  ActionListener<MultiSearchTemplateResponse> listener) {
        return performRequestAsyncAndParseEntity(multiSearchTemplateRequest, RequestConverters::multiSearchTemplate,
            options, MultiSearchTemplateResponse::fromXContext, listener, emptySet());
    }

    /**
     * Asynchronously executes a request using the Ranking Evaluation API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-rank-eval.html">Ranking Evaluation API
     * on elastic.co</a>
     * @param rankEvalRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable rankEvalAsync(RankEvalRequest rankEvalRequest, RequestOptions options,
                                           ActionListener<RankEvalResponse> listener) {
        return performRequestAsyncAndParseEntity(rankEvalRequest, RequestConverters::rankEval, options,
            RankEvalResponse::fromXContent, listener,
                emptySet());
    }

    /**
     * Executes a request using the Field Capabilities API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-field-caps.html">Field Capabilities API
     * on elastic.co</a>.
     * @param fieldCapabilitiesRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public final FieldCapabilitiesResponse fieldCaps(FieldCapabilitiesRequest fieldCapabilitiesRequest,
                                                     RequestOptions options) throws IOException {
        return performRequestAndParseEntity(fieldCapabilitiesRequest, RequestConverters::fieldCaps, options,
            FieldCapabilitiesResponse::fromXContent, emptySet());
    }

    /**
     * Get stored script by id.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/modules-scripting-using.html">
     *     How to use scripts on elastic.co</a>
     * @param request the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public GetStoredScriptResponse getScript(GetStoredScriptRequest request, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(request, RequestConverters::getScript, options,
            GetStoredScriptResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously get stored script by id.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/modules-scripting-using.html">
     *     How to use scripts on elastic.co</a>
     * @param request the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public Cancellable getScriptAsync(GetStoredScriptRequest request, RequestOptions options,
                                      ActionListener<GetStoredScriptResponse> listener) {
        return performRequestAsyncAndParseEntity(request, RequestConverters::getScript, options,
            GetStoredScriptResponse::fromXContent, listener, emptySet());
    }

    /**
     * Delete stored script by id.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/modules-scripting-using.html">
     *     How to use scripts on elastic.co</a>
     * @param request the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public AcknowledgedResponse deleteScript(DeleteStoredScriptRequest request, RequestOptions options) throws IOException {
        return performRequestAndParseEntity(request, RequestConverters::deleteScript, options,
            AcknowledgedResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously delete stored script by id.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/modules-scripting-using.html">
     *     How to use scripts on elastic.co</a>
     * @param request the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public Cancellable deleteScriptAsync(DeleteStoredScriptRequest request, RequestOptions options,
                                         ActionListener<AcknowledgedResponse> listener) {
        return performRequestAsyncAndParseEntity(request, RequestConverters::deleteScript, options,
            AcknowledgedResponse::fromXContent, listener, emptySet());
    }

    /**
     * Puts an stored script using the Scripting API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/modules-scripting-using.html"> Scripting API
     * on elastic.co</a>
     * @param putStoredScriptRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response
     */
    public AcknowledgedResponse putScript(PutStoredScriptRequest putStoredScriptRequest,
                                             RequestOptions options) throws IOException {
        return performRequestAndParseEntity(putStoredScriptRequest, RequestConverters::putScript, options,
            AcknowledgedResponse::fromXContent, emptySet());
    }

    /**
     * Asynchronously puts an stored script using the Scripting API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/modules-scripting-using.html"> Scripting API
     * on elastic.co</a>
     * @param putStoredScriptRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public Cancellable putScriptAsync(PutStoredScriptRequest putStoredScriptRequest, RequestOptions options,
                                      ActionListener<AcknowledgedResponse> listener) {
        return performRequestAsyncAndParseEntity(putStoredScriptRequest, RequestConverters::putScript, options,
            AcknowledgedResponse::fromXContent, listener, emptySet());
    }

    /**
     * Asynchronously executes a request using the Field Capabilities API.
     * See <a href="https://www.elastic.co/guide/en/easysearch/reference/current/search-field-caps.html">Field Capabilities API
     * on elastic.co</a>.
     * @param fieldCapabilitiesRequest the request
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @param listener the listener to be notified upon request completion
     * @return cancellable that may be used to cancel the request
     */
    public final Cancellable fieldCapsAsync(FieldCapabilitiesRequest fieldCapabilitiesRequest, RequestOptions options,
                                            ActionListener<FieldCapabilitiesResponse> listener) {
        return performRequestAsyncAndParseEntity(fieldCapabilitiesRequest, RequestConverters::fieldCaps, options,
            FieldCapabilitiesResponse::fromXContent, listener, emptySet());
    }

    /**
     * @deprecated If creating a new HLRC ReST API call, consider creating new actions instead of reusing server actions. The Validation
     * layer has been added to the ReST client, and requests should extend {@link Validatable} instead of {@link ActionRequest}.
     */
    @Deprecated
    protected final <Req extends ActionRequest, Resp> Resp performRequestAndParseEntity(Req request,
                                                                            CheckedFunction<Req, Request, IOException> requestConverter,
                                                                            RequestOptions options,
                                                                            CheckedFunction<XContentParser, Resp, IOException> entityParser,
                                                                            Set<Integer> ignores) throws IOException {
        return performRequest(request, requestConverter, options,
                response -> parseEntity(response.getEntity(), entityParser), ignores);
    }

    /**
     * Defines a helper method for performing a request and then parsing the returned entity using the provided entityParser.
     */
    protected final <Req extends Validatable, Resp> Resp performRequestAndParseEntity(Req request,
                                                                  CheckedFunction<Req, Request, IOException> requestConverter,
                                                                  RequestOptions options,
                                                                  CheckedFunction<XContentParser, Resp, IOException> entityParser,
                                                                  Set<Integer> ignores) throws IOException {
        return performRequest(request, requestConverter, options,
                response -> parseEntity(response.getEntity(), entityParser), ignores);
    }

    /**
     * @deprecated If creating a new HLRC ReST API call, consider creating new actions instead of reusing server actions. The Validation
     * layer has been added to the ReST client, and requests should extend {@link Validatable} instead of {@link ActionRequest}.
     */
    @Deprecated
    protected final <Req extends ActionRequest, Resp> Resp performRequest(Req request,
                                                                           CheckedFunction<Req, Request, IOException> requestConverter,
                                                                           RequestOptions options,
                                                                           CheckedFunction<Response, Resp, IOException> responseConverter,
                                                                           Set<Integer> ignores) throws IOException {
        ActionRequestValidationException validationException = request.validate();
        if (validationException != null && validationException.validationErrors().isEmpty() == false) {
            throw validationException;
        }
        return internalPerformRequest(request, requestConverter, options, responseConverter, ignores);
    }

    /**
     * Defines a helper method for performing a request.
     */
    protected final <Req extends Validatable, Resp> Resp performRequest(Req request,
                                                             CheckedFunction<Req, Request, IOException> requestConverter,
                                                             RequestOptions options,
                                                             CheckedFunction<Response, Resp, IOException> responseConverter,
                                                             Set<Integer> ignores) throws IOException {
        Optional<ValidationException> validationException = request.validate();
        if (validationException != null && validationException.isPresent()) {
            throw validationException.get();
        }
        return internalPerformRequest(request, requestConverter, options, responseConverter, ignores);
    }

    /**
     * Provides common functionality for performing a request.
     */
    private <Req, Resp> Resp internalPerformRequest(Req request,
                                            CheckedFunction<Req, Request, IOException> requestConverter,
                                            RequestOptions options,
                                            CheckedFunction<Response, Resp, IOException> responseConverter,
                                            Set<Integer> ignores) throws IOException {
        Request req = requestConverter.apply(request);
        req.setOptions(options);
        Response response;
        try {
            response = client.performRequest(req);
        } catch (ResponseException e) {
            if (ignores.contains(e.getResponse().getStatusLine().getStatusCode())) {
                try {
                    return responseConverter.apply(e.getResponse());
                } catch (Exception innerException) {
                    // the exception is ignored as we now try to parse the response as an error.
                    // this covers cases like get where 404 can either be a valid document not found response,
                    // or an error for which parsing is completely different. We try to consider the 404 response as a valid one
                    // first. If parsing of the response breaks, we fall back to parsing it as an error.
                    throw parseResponseException(e);
                }
            }
            throw parseResponseException(e);
        }

        try {
            return responseConverter.apply(response);
        } catch(Exception e) {
            throw new IOException("Unable to parse response body for " + response, e);
        }
    }

    /**
     * Defines a helper method for requests that can 404 and in which case will return an empty Optional
     * otherwise tries to parse the response body
     */
    protected final <Req extends Validatable, Resp> Optional<Resp> performRequestAndParseOptionalEntity(Req request,
                                                                  CheckedFunction<Req, Request, IOException> requestConverter,
                                                                  RequestOptions options,
                                                                  CheckedFunction<XContentParser, Resp, IOException> entityParser
                                                                  ) throws IOException {
        Optional<ValidationException> validationException = request.validate();
        if (validationException != null && validationException.isPresent()) {
            throw validationException.get();
        }
        Request req = requestConverter.apply(request);
        req.setOptions(options);
        Response response;
        try {
            response = client.performRequest(req);
        } catch (ResponseException e) {
            if (RestStatus.NOT_FOUND.getStatus() == e.getResponse().getStatusLine().getStatusCode()) {
                return Optional.empty();
            }
            throw parseResponseException(e);
        }

        try {
            return Optional.of(parseEntity(response.getEntity(), entityParser));
        } catch (Exception e) {
            throw new IOException("Unable to parse response body for " + response, e);
        }
    }

    /**
     * @deprecated If creating a new HLRC ReST API call, consider creating new actions instead of reusing server actions. The Validation
     * layer has been added to the ReST client, and requests should extend {@link Validatable} instead of {@link ActionRequest}.
     * @return Cancellable instance that may be used to cancel the request
     */
    @Deprecated
    protected final <Req extends ActionRequest, Resp> Cancellable performRequestAsyncAndParseEntity(Req request,
                CheckedFunction<Req, Request, IOException> requestConverter,
                RequestOptions options,
                CheckedFunction<XContentParser, Resp, IOException> entityParser,
                ActionListener<Resp> listener, Set<Integer> ignores) {
        return performRequestAsync(request, requestConverter, options,
                response -> parseEntity(response.getEntity(), entityParser), listener, ignores);
    }

    /**
     * Defines a helper method for asynchronously performing a request.
     * @return Cancellable instance that may be used to cancel the request
     */
    protected final <Req extends Validatable, Resp> Cancellable performRequestAsyncAndParseEntity(Req request,
                CheckedFunction<Req, Request, IOException> requestConverter,
                RequestOptions options,
                CheckedFunction<XContentParser, Resp, IOException> entityParser,
                ActionListener<Resp> listener, Set<Integer> ignores) {
        return performRequestAsync(request, requestConverter, options,
                response -> parseEntity(response.getEntity(), entityParser), listener, ignores);
    }


    /**
     * @deprecated If creating a new HLRC ReST API call, consider creating new actions instead of reusing server actions. The Validation
     * layer has been added to the ReST client, and requests should extend {@link Validatable} instead of {@link ActionRequest}.
     * @return Cancellable instance that may be used to cancel the request
     */
    @Deprecated
    protected final <Req extends ActionRequest, Resp> Cancellable performRequestAsync(Req request,
                CheckedFunction<Req, Request, IOException> requestConverter,
                RequestOptions options,
                CheckedFunction<Response, Resp, IOException> responseConverter,
                ActionListener<Resp> listener, Set<Integer> ignores) {
        ActionRequestValidationException validationException = request.validate();
        if (validationException != null && validationException.validationErrors().isEmpty() == false) {
            listener.onFailure(validationException);
            return Cancellable.NO_OP;
        }
        return internalPerformRequestAsync(request, requestConverter, options, responseConverter, listener, ignores);
    }

    /**
     * Defines a helper method for asynchronously performing a request.
     * @return Cancellable instance that may be used to cancel the request
     */
    protected final <Req extends Validatable, Resp> Cancellable performRequestAsync(Req request,
                CheckedFunction<Req, Request, IOException> requestConverter,
                RequestOptions options,
                CheckedFunction<Response, Resp, IOException> responseConverter,
                ActionListener<Resp> listener, Set<Integer> ignores) {
        Optional<ValidationException> validationException = request.validate();
        if (validationException != null && validationException.isPresent()) {
            listener.onFailure(validationException.get());
            return Cancellable.NO_OP;
        }
        return internalPerformRequestAsync(request, requestConverter, options, responseConverter, listener, ignores);
    }

    /**
     * Provides common functionality for asynchronously performing a request.
     * @return Cancellable instance that may be used to cancel the request
     */
    private <Req, Resp> Cancellable internalPerformRequestAsync(Req request,
                 CheckedFunction<Req, Request, IOException> requestConverter,
                 RequestOptions options,
                 CheckedFunction<Response, Resp, IOException> responseConverter,
                 ActionListener<Resp> listener, Set<Integer> ignores) {
        Request req;
        try {
            req = requestConverter.apply(request);
        } catch (Exception e) {
            listener.onFailure(e);
            return Cancellable.NO_OP;
        }
        req.setOptions(options);

        ResponseListener responseListener = wrapResponseListener(responseConverter, listener, ignores);
        return client.performRequestAsync(req, responseListener);
    }


    final <Resp> ResponseListener wrapResponseListener(CheckedFunction<Response, Resp, IOException> responseConverter,
                                                        ActionListener<Resp> actionListener, Set<Integer> ignores) {
        return new ResponseListener() {
            @Override
            public void onSuccess(Response response) {
                try {
                    actionListener.onResponse(responseConverter.apply(response));
                } catch(Exception e) {
                    IOException ioe = new IOException("Unable to parse response body for " + response, e);
                    onFailure(ioe);
                }
            }

            @Override
            public void onFailure(Exception exception) {
                if (exception instanceof ResponseException) {
                    ResponseException responseException = (ResponseException) exception;
                    Response response = responseException.getResponse();
                    if (ignores.contains(response.getStatusLine().getStatusCode())) {
                        try {
                            actionListener.onResponse(responseConverter.apply(response));
                        } catch (Exception innerException) {
                            // the exception is ignored as we now try to parse the response as an error.
                            // this covers cases like get where 404 can either be a valid document not found response,
                            // or an error for which parsing is completely different. We try to consider the 404 response as a valid one
                            // first. If parsing of the response breaks, we fall back to parsing it as an error.
                            actionListener.onFailure(parseResponseException(responseException));
                        }
                    } else {
                        actionListener.onFailure(parseResponseException(responseException));
                    }
                } else {
                    actionListener.onFailure(exception);
                }
            }
        };
    }

    /**
     * Asynchronous request which returns empty {@link Optional}s in the case of 404s or parses entity into an Optional
     * @return Cancellable instance that may be used to cancel the request
     */
    protected final <Req extends Validatable, Resp> Cancellable performRequestAsyncAndParseOptionalEntity(Req request,
              CheckedFunction<Req, Request, IOException> requestConverter,
              RequestOptions options,
              CheckedFunction<XContentParser, Resp, IOException> entityParser,
              ActionListener<Optional<Resp>> listener) {
        Optional<ValidationException> validationException = request.validate();
        if (validationException != null && validationException.isPresent()) {
            listener.onFailure(validationException.get());
            return Cancellable.NO_OP;
        }
        Request req;
        try {
            req = requestConverter.apply(request);
        } catch (Exception e) {
            listener.onFailure(e);
            return Cancellable.NO_OP;
        }
        req.setOptions(options);
        ResponseListener responseListener = wrapResponseListener404sOptional(response -> parseEntity(response.getEntity(),
                entityParser), listener);
        return client.performRequestAsync(req, responseListener);
    }

    final <Resp> ResponseListener wrapResponseListener404sOptional(CheckedFunction<Response, Resp, IOException> responseConverter,
            ActionListener<Optional<Resp>> actionListener) {
        return new ResponseListener() {
            @Override
            public void onSuccess(Response response) {
                try {
                    actionListener.onResponse(Optional.of(responseConverter.apply(response)));
                } catch (Exception e) {
                    IOException ioe = new IOException("Unable to parse response body for " + response, e);
                    onFailure(ioe);
                }
            }

            @Override
            public void onFailure(Exception exception) {
                if (exception instanceof ResponseException) {
                    ResponseException responseException = (ResponseException) exception;
                    Response response = responseException.getResponse();
                    if (RestStatus.NOT_FOUND.getStatus() == response.getStatusLine().getStatusCode()) {
                            actionListener.onResponse(Optional.empty());
                    } else {
                        actionListener.onFailure(parseResponseException(responseException));
                    }
                } else {
                    actionListener.onFailure(exception);
                }
            }
        };
    }

    /**
     * Converts a {@link ResponseException} obtained from the low level REST client into an {@link EasysearchException}.
     * If a response body was returned, tries to parse it as an error returned from Easysearch.
     * If no response body was returned or anything goes wrong while parsing the error, returns a new {@link EasysearchStatusException}
     * that wraps the original {@link ResponseException}. The potential exception obtained while parsing is added to the returned
     * exception as a suppressed exception. This method is guaranteed to not throw any exception eventually thrown while parsing.
     */
    protected final EasysearchStatusException parseResponseException(ResponseException responseException) {
        Response response = responseException.getResponse();
        HttpEntity entity = response.getEntity();
        EasysearchStatusException easysearchException;
        RestStatus restStatus = RestStatus.fromCode(response.getStatusLine().getStatusCode());

        if (entity == null) {
            easysearchException = new EasysearchStatusException(
                    responseException.getMessage(), restStatus, responseException);
        } else {
            try {
                easysearchException = parseEntity(entity, BytesRestResponse::errorFromXContent);
                easysearchException.addSuppressed(responseException);
            } catch (Exception e) {
                easysearchException = new EasysearchStatusException("Unable to parse response body", restStatus, responseException);
                easysearchException.addSuppressed(e);
            }
        }
        return easysearchException;
    }

    protected final <Resp> Resp parseEntity(final HttpEntity entity,
                                      final CheckedFunction<XContentParser, Resp, IOException> entityParser) throws IOException {
        if (entity == null) {
            throw new IllegalStateException("Response body expected but not returned");
        }
        if (entity.getContentType() == null) {
            throw new IllegalStateException("Easysearch didn't return the [Content-Type] header, unable to parse response body");
        }
        XContentType xContentType = XContentType.fromMediaTypeOrFormat(entity.getContentType().getValue());
        if (xContentType == null) {
            throw new IllegalStateException("Unsupported Content-Type: " + entity.getContentType().getValue());
        }
        try (XContentParser parser = xContentType.xContent().createParser(registry, DEPRECATION_HANDLER, entity.getContent())) {
            return entityParser.apply(parser);
        }
    }

    protected static boolean convertExistsResponse(Response response) {
        return response.getStatusLine().getStatusCode() == 200;
    }

    /**
     * Ignores deprecation warnings. This is appropriate because it is only
     * used to parse responses from Easysearch. Any deprecation warnings
     * emitted there just mean that you are talking to an old version of
     * Easysearch. There isn't anything you can do about the deprecation.
     */
    private static final DeprecationHandler DEPRECATION_HANDLER = DeprecationHandler.IGNORE_DEPRECATIONS;

    static List<NamedXContentRegistry.Entry> getDefaultNamedXContents() {
        Map<String, ContextParser<Object, ? extends Aggregation>> map = new HashMap<>();
        map.put(CardinalityAggregationBuilder.NAME, (p, c) -> ParsedCardinality.fromXContent(p, (String) c));
        map.put(InternalHDRPercentiles.NAME, (p, c) -> ParsedHDRPercentiles.fromXContent(p, (String) c));
        map.put(InternalHDRPercentileRanks.NAME, (p, c) -> ParsedHDRPercentileRanks.fromXContent(p, (String) c));
        map.put(InternalTDigestPercentiles.NAME, (p, c) -> ParsedTDigestPercentiles.fromXContent(p, (String) c));
        map.put(InternalTDigestPercentileRanks.NAME, (p, c) -> ParsedTDigestPercentileRanks.fromXContent(p, (String) c));
        map.put(PercentilesBucketPipelineAggregationBuilder.NAME, (p, c) -> ParsedPercentilesBucket.fromXContent(p, (String) c));
        map.put(MedianAbsoluteDeviationAggregationBuilder.NAME, (p, c) -> ParsedMedianAbsoluteDeviation.fromXContent(p, (String) c));
        map.put(MinAggregationBuilder.NAME, (p, c) -> ParsedMin.fromXContent(p, (String) c));
        map.put(MaxAggregationBuilder.NAME, (p, c) -> ParsedMax.fromXContent(p, (String) c));
        map.put(SumAggregationBuilder.NAME, (p, c) -> ParsedSum.fromXContent(p, (String) c));
        map.put(AvgAggregationBuilder.NAME, (p, c) -> ParsedAvg.fromXContent(p, (String) c));
        map.put(WeightedAvgAggregationBuilder.NAME, (p, c) -> ParsedWeightedAvg.fromXContent(p, (String) c));
        map.put(ValueCountAggregationBuilder.NAME, (p, c) -> ParsedValueCount.fromXContent(p, (String) c));
        map.put(InternalSimpleValue.NAME, (p, c) -> ParsedSimpleValue.fromXContent(p, (String) c));
        map.put(DerivativePipelineAggregationBuilder.NAME, (p, c) -> ParsedDerivative.fromXContent(p, (String) c));
        map.put(InternalBucketMetricValue.NAME, (p, c) -> ParsedBucketMetricValue.fromXContent(p, (String) c));
        map.put(StatsAggregationBuilder.NAME, (p, c) -> ParsedStats.fromXContent(p, (String) c));
        map.put(StatsBucketPipelineAggregationBuilder.NAME, (p, c) -> ParsedStatsBucket.fromXContent(p, (String) c));
        map.put(ExtendedStatsAggregationBuilder.NAME, (p, c) -> ParsedExtendedStats.fromXContent(p, (String) c));
        map.put(ExtendedStatsBucketPipelineAggregationBuilder.NAME,
                (p, c) -> ParsedExtendedStatsBucket.fromXContent(p, (String) c));
        map.put(GeoBoundsAggregationBuilder.NAME, (p, c) -> ParsedGeoBounds.fromXContent(p, (String) c));
        map.put(GeoCentroidAggregationBuilder.NAME, (p, c) -> ParsedGeoCentroid.fromXContent(p, (String) c));
        map.put(HistogramAggregationBuilder.NAME, (p, c) -> ParsedHistogram.fromXContent(p, (String) c));
        map.put(DateHistogramAggregationBuilder.NAME, (p, c) -> ParsedDateHistogram.fromXContent(p, (String) c));
        map.put(AutoDateHistogramAggregationBuilder.NAME, (p, c) -> ParsedAutoDateHistogram.fromXContent(p, (String) c));
        map.put(VariableWidthHistogramAggregationBuilder.NAME,
            (p, c) -> ParsedVariableWidthHistogram.fromXContent(p, (String) c));
        map.put(StringTerms.NAME, (p, c) -> ParsedStringTerms.fromXContent(p, (String) c));
        map.put(LongTerms.NAME, (p, c) -> ParsedLongTerms.fromXContent(p, (String) c));
        map.put(DoubleTerms.NAME, (p, c) -> ParsedDoubleTerms.fromXContent(p, (String) c));
        map.put(LongRareTerms.NAME, (p, c) -> ParsedLongRareTerms.fromXContent(p, (String) c));
        map.put(StringRareTerms.NAME, (p, c) -> ParsedStringRareTerms.fromXContent(p, (String) c));
        map.put(MissingAggregationBuilder.NAME, (p, c) -> ParsedMissing.fromXContent(p, (String) c));
        map.put(NestedAggregationBuilder.NAME, (p, c) -> ParsedNested.fromXContent(p, (String) c));
        map.put(ReverseNestedAggregationBuilder.NAME, (p, c) -> ParsedReverseNested.fromXContent(p, (String) c));
        map.put(GlobalAggregationBuilder.NAME, (p, c) -> ParsedGlobal.fromXContent(p, (String) c));
        map.put(FilterAggregationBuilder.NAME, (p, c) -> ParsedFilter.fromXContent(p, (String) c));
        map.put(InternalSampler.PARSER_NAME, (p, c) -> ParsedSampler.fromXContent(p, (String) c));
        map.put(GeoHashGridAggregationBuilder.NAME, (p, c) -> ParsedGeoHashGrid.fromXContent(p, (String) c));
        map.put(GeoTileGridAggregationBuilder.NAME, (p, c) -> ParsedGeoTileGrid.fromXContent(p, (String) c));
        map.put(RangeAggregationBuilder.NAME, (p, c) -> ParsedRange.fromXContent(p, (String) c));
        map.put(DateRangeAggregationBuilder.NAME, (p, c) -> ParsedDateRange.fromXContent(p, (String) c));
        map.put(GeoDistanceAggregationBuilder.NAME, (p, c) -> ParsedGeoDistance.fromXContent(p, (String) c));
        map.put(FiltersAggregationBuilder.NAME, (p, c) -> ParsedFilters.fromXContent(p, (String) c));
        map.put(AdjacencyMatrixAggregationBuilder.NAME, (p, c) -> ParsedAdjacencyMatrix.fromXContent(p, (String) c));
        map.put(SignificantLongTerms.NAME, (p, c) -> ParsedSignificantLongTerms.fromXContent(p, (String) c));
        map.put(SignificantStringTerms.NAME, (p, c) -> ParsedSignificantStringTerms.fromXContent(p, (String) c));
        map.put(ScriptedMetricAggregationBuilder.NAME, (p, c) -> ParsedScriptedMetric.fromXContent(p, (String) c));
        map.put(IpRangeAggregationBuilder.NAME, (p, c) -> ParsedBinaryRange.fromXContent(p, (String) c));
        map.put(TopHitsAggregationBuilder.NAME, (p, c) -> ParsedTopHits.fromXContent(p, (String) c));
        map.put(CompositeAggregationBuilder.NAME, (p, c) -> ParsedComposite.fromXContent(p, (String) c));
        map.put(StringStatsAggregationBuilder.NAME, (p, c) -> ParsedStringStats.PARSER.parse(p, (String) c));
        map.put(TopMetricsAggregationBuilder.NAME, (p, c) -> ParsedTopMetrics.PARSER.parse(p, (String) c));
        //map.put(InferencePipelineAggregationBuilder.NAME, (p, c) -> ParsedInference.fromXContent(p, (String ) (c)));
        List<NamedXContentRegistry.Entry> entries = map.entrySet().stream()
                .map(entry -> new NamedXContentRegistry.Entry(Aggregation.class, new ParseField(entry.getKey()), entry.getValue()))
                .collect(Collectors.toList());
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(TermSuggestionBuilder.SUGGESTION_NAME),
                (parser, context) -> TermSuggestion.fromXContent(parser, (String)context)));
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(PhraseSuggestionBuilder.SUGGESTION_NAME),
                (parser, context) -> PhraseSuggestion.fromXContent(parser, (String)context)));
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(CompletionSuggestionBuilder.SUGGESTION_NAME),
                (parser, context) -> CompletionSuggestion.fromXContent(parser, (String)context)));
        return entries;
    }

    /**
     * Loads and returns the {@link NamedXContentRegistry.Entry} parsers provided by plugins.
     */
    static List<NamedXContentRegistry.Entry> getProvidedNamedXContents() {
        List<NamedXContentRegistry.Entry> entries = new ArrayList<>();
        for (NamedXContentProvider service : ServiceLoader.load(NamedXContentProvider.class)) {
            entries.addAll(service.getNamedXContentParsers());
        }
        return entries;
    }
}
