/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */

package org.easysearch.client;

import org.easysearch.action.ActionListener;
import org.easysearch.client.security.*;

import java.io.IOException;

import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;

/**
 * A wrapper for the {@link RestHighLevelClient} that provides methods for accessing the Security APIs.
 */
public final class SecurityClient {

    private final RestHighLevelClient restHighLevelClient;

    SecurityClient(RestHighLevelClient restHighLevelClient) {
        this.restHighLevelClient = restHighLevelClient;
    }


    public GetAccountResponse getAccount() throws IOException {
        GetAccountRequest getAccountRequest = new GetAccountRequest();
        RequestOptions options = RequestOptions.DEFAULT;
        return restHighLevelClient.performRequestAndParseEntity(
            getAccountRequest, SecurityRequestConverters::getAccount, options,
            GetAccountResponse::fromXContent, emptySet());
    }

    /**
     * Get a user, or list of users, in the native realm synchronously.
     * @param request the request with the user's name
     * @param options the request options (e.g., headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the get users call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public GetUsersResponse getUsers(GetUsersRequest request, RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::getUsers, options,
            GetUsersResponse::fromXContent, emptySet());
    }

    /**
     * Create/update a user in the native realm synchronously.
     * @param request the request with the user's information
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the put user call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public PutUserResponse putUser(PutUserRequest request, RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::putUser, options,
            PutUserResponse::fromXContent, emptySet());
    }

    /**
     * Removes user from the native realm synchronously.
     * @param request the request with the user to delete
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the delete user call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public DeleteUserResponse deleteUser(DeleteUserRequest request, RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::deleteUser, options,
            DeleteUserResponse::fromXContent, singleton(404));
    }

    /**
     * Create/Update a role mapping.
     * @param request the request with the role mapping information
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the put role mapping call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public PutRoleMappingResponse putRoleMapping(final PutRoleMappingRequest request, final RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::putRoleMapping, options,
                PutRoleMappingResponse::fromXContent, emptySet());
    }

    /**
     * Synchronously get role mapping(s).
     *
     * @param request {@link GetRoleMappingsRequest} with role mapping name(s).
     * If no role mapping name is provided then retrieves all role mappings.
     * @param options the request options (e.g. headers), use
     * {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the get role mapping call
     * @throws IOException in case there is a problem sending the request or
     * parsing back the response
     */
    public GetRoleMappingsResponse getRoleMappings(final GetRoleMappingsRequest request,
                                                   final RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::getRoleMappings,
            options, GetRoleMappingsResponse::fromXContent, emptySet());
    }

    /**
     * Synchronously retrieve the X.509 certificates that are used to encrypt communications in an Easysearch cluster.
     *
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the get certificates call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public GetSslCertificatesResponse getSslCertificates(RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(GetSslCertificatesRequest.INSTANCE, GetSslCertificatesRequest::getRequest,
            options, GetSslCertificatesResponse::fromXContent, emptySet());
    }

    /**
     * Change the password of a user of a native realm or built-in user synchronously.
     *
     * @param request the request with the user's new password
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return {@code true} if the request succeeded (the new password was set)
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public boolean changePassword(ChangePasswordRequest request, RequestOptions options) throws IOException {
        return restHighLevelClient.performRequest(request, SecurityRequestConverters::changePassword, options,
            RestHighLevelClient::convertExistsResponse, emptySet());
    }

    /**
     * Delete a role mapping.
     *
     * @param request the request with the role mapping name to be deleted.
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the delete role mapping call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public DeleteRoleMappingResponse deleteRoleMapping(DeleteRoleMappingRequest request, RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::deleteRoleMapping, options,
            DeleteRoleMappingResponse::fromXContent, singleton(404));
    }

    /**
     * Retrieves roles from the native roles store.
     *
     * @param request the request with the roles to get
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the get roles call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public GetRolesResponse getRoles(final GetRolesRequest request, final RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::getRoles, options,
            GetRolesResponse::fromXContent, emptySet());
    }

    /**
     * Create or update a role in the native roles store.
     *
     * @param request the request containing the role to create or update
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the put role call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public PutRoleResponse putRole(final PutRoleRequest request, final RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::putRole, options,
            PutRoleResponse::fromXContent, emptySet());
    }

    /**
     * Removes role.
     *
     * @param request the request with the role to delete
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the delete role call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public DeleteRoleResponse deleteRole(DeleteRoleRequest request, RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::deleteRole, options,
            DeleteRoleResponse::fromXContent, singleton(404));
    }

    /**
     * Synchronously get privilege(s).
     *
     * @param request {@link GetPrivilegesRequest} with the privilege name.
     * @param options the request options (e.g. headers), use
     *                {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the get privileges call
     * @throws IOException in case there is a problem sending the request or
     *                     parsing back the response
     */
    public GetPrivilegesResponse getPrivileges(final GetPrivilegesRequest request, final RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::getPrivileges,
            options, GetPrivilegesResponse::fromXContent, emptySet());
    }


    /**
     * Create or update privilege.
     *
     * @param request the request to create or update privileges
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the create or update privileges call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public PutPrivilegeResponse putPrivileges(final PutPrivilegeRequest request, final RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::putPrivileges, options,
                PutPrivilegeResponse::fromXContent, emptySet());
    }

    /**
     * Removes privilege
     *
     * @param request the request with the privilege to delete
     * @param options the request options (e.g. headers), use {@link RequestOptions#DEFAULT} if nothing needs to be customized
     * @return the response from the delete privilege call
     * @throws IOException in case there is a problem sending the request or parsing back the response
     */
    public DeletePrivilegeResponse deletePrivilege(DeletePrivilegesRequest request, RequestOptions options) throws IOException {
        return restHighLevelClient.performRequestAndParseEntity(request, SecurityRequestConverters::deletePrivileges, options,
            DeletePrivilegeResponse::fromXContent, singleton(404));
    }
}
