/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */

package org.easysearch.client;

import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.easysearch.client.security.*;
import org.easysearch.common.Strings;

import java.io.IOException;

import static org.easysearch.client.RequestConverters.REQUEST_BODY_CONTENT_TYPE;
import static org.easysearch.client.RequestConverters.createEntity;

final class SecurityRequestConverters {

    private SecurityRequestConverters() {}

    static Request changePassword(ChangePasswordRequest changePasswordRequest) throws IOException {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/account")
            .build();
        Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        request.setEntity(createEntity(changePasswordRequest, REQUEST_BODY_CONTENT_TYPE));
        RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }

    static Request getUsers(GetUsersRequest getUsersRequest) {
        RequestConverters.EndpointBuilder builder = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/user");
        if (getUsersRequest.getUsernames().size() > 0) {
            builder.addPathPart(Strings.collectionToCommaDelimitedString(getUsersRequest.getUsernames()));
        }
        return new Request(HttpGet.METHOD_NAME, builder.build());
    }

    static Request getAccount(GetAccountRequest getAccountRequest) {
        RequestConverters.EndpointBuilder builder = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/account");
        return new Request(HttpGet.METHOD_NAME, builder.build());
    }

    static Request putUser(PutUserRequest putUserRequest) throws IOException {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/user")
            .addPathPart(putUserRequest.getUser().getName())
            .build();
        Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        request.setEntity(createEntity(putUserRequest, REQUEST_BODY_CONTENT_TYPE));
        RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }

    static Request deleteUser(DeleteUserRequest deleteUserRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security", "user")
            .addPathPart(deleteUserRequest.getName())
            .build();
        Request request = new Request(HttpDelete.METHOD_NAME, endpoint);
        RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }

    static Request putRoleMapping(final PutRoleMappingRequest putRoleMappingRequest) throws IOException {
        final String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/role_mapping")
            .addPathPart(putRoleMappingRequest.getRoleMapping().getName())
            .build();
        final Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        request.setEntity(createEntity(putRoleMappingRequest, REQUEST_BODY_CONTENT_TYPE));
        final RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }

    static Request getRoleMappings(final GetRoleMappingsRequest getRoleMappingRequest) throws IOException {
        RequestConverters.EndpointBuilder builder = new RequestConverters.EndpointBuilder();
        builder.addPathPartAsIs("_security/role_mapping");
        if (getRoleMappingRequest.getRoleMappingNames().size() > 0) {
            builder.addPathPart(Strings.collectionToCommaDelimitedString(getRoleMappingRequest.getRoleMappingNames()));
        }
        return new Request(HttpGet.METHOD_NAME, builder.build());
    }

    static Request deleteRoleMapping(DeleteRoleMappingRequest deleteRoleMappingRequest) {
        final String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/role_mapping")
            .addPathPart(deleteRoleMappingRequest.getName())
            .build();
        final Request request = new Request(HttpDelete.METHOD_NAME, endpoint);
        final RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }

    static Request deleteRole(DeleteRoleRequest deleteRoleRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/role")
            .addPathPart(deleteRoleRequest.getName())
            .build();
        Request request = new Request(HttpDelete.METHOD_NAME, endpoint);
        RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }

    static Request getRoles(GetRolesRequest getRolesRequest) {
        RequestConverters.EndpointBuilder builder = new RequestConverters.EndpointBuilder();
        builder.addPathPartAsIs("_security/role");
        if (getRolesRequest.getRoleNames().size() > 0) {
            builder.addPathPart(Strings.collectionToCommaDelimitedString(getRolesRequest.getRoleNames()));
        }
        return new Request(HttpGet.METHOD_NAME, builder.build());
    }

    static Request getPrivileges(GetPrivilegesRequest getPrivilegesRequest) {
        RequestConverters.EndpointBuilder builder = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/privilege");
        if (getPrivilegesRequest.getPrivilegeNames().size() > 0) {
            builder.addPathPart(Strings.collectionToCommaDelimitedString(getPrivilegesRequest.getPrivilegeNames()));
        }
        return new Request(HttpGet.METHOD_NAME, builder.build());
    }

    static Request putPrivileges(final PutPrivilegeRequest putPrivilegesRequest) throws IOException {
         final String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/privilege")
            .addPathPart(putPrivilegesRequest.getPrivilege().getName())
            .build();
        Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        request.setEntity(createEntity(putPrivilegesRequest, REQUEST_BODY_CONTENT_TYPE));
        RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }

    static Request deletePrivileges(DeletePrivilegesRequest deletePrivilegeRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/privilege")
            .addPathPart(deletePrivilegeRequest.getPrivilege())
            .build();
        Request request = new Request(HttpDelete.METHOD_NAME, endpoint);
        RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }

    static Request putRole(final PutRoleRequest putRoleRequest) throws IOException {
        final String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_security/role")
            .addPathPart(putRoleRequest.getRole().getName())
            .build();
        final Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        request.setEntity(createEntity(putRoleRequest, REQUEST_BODY_CONTENT_TYPE));
        final RequestConverters.Params params = new RequestConverters.Params();
        request.addParameters(params.asMap());
        return request;
    }
}
