/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */
package org.easysearch.client.indices;

import org.easysearch.action.admin.indices.open.OpenIndexResponse;
import org.easysearch.action.support.ActiveShardCount;
import org.easysearch.action.support.IndicesOptions;
import org.easysearch.client.TimedRequest;

import java.util.Objects;

/**
 * Request for the _unfreeze index API
 */
public final class UnfreezeIndexRequest extends TimedRequest {

    private final String[] indices;
    private IndicesOptions indicesOptions;
    private ActiveShardCount waitForActiveShards;

    /**
     * Creates a new unfreeze index request
     * @param indices the index to unfreeze
     */
    public UnfreezeIndexRequest(String... indices) {
        this.indices = Objects.requireNonNull(indices);
    }

    /**
     * Returns the indices to unfreeze
     */
    public String[] getIndices() {
        return indices;
    }

    /**
     * Specifies what type of requested indices to ignore and how to deal with wildcard expressions.
     * For example indices that don't exist.
     *
     * @return the current behaviour when it comes to index names and wildcard indices expressions
     */
    public IndicesOptions indicesOptions() {
        return indicesOptions;
    }

    /**
     * Specifies what type of requested indices to ignore and how to deal with wildcard expressions.
     * For example indices that don't exist.
     *
     * @param indicesOptions the desired behaviour regarding indices to ignore and wildcard indices expressions
     */
    public void setIndicesOptions(IndicesOptions indicesOptions) {
        this.indicesOptions = indicesOptions;
    }

    /**
     * Returns the wait for active shard cound or null if the default should be used
     */
    public ActiveShardCount getWaitForActiveShards() {
        return waitForActiveShards;
    }

    /**
     * Sets the number of shard copies that should be active for indices opening to return.
     * Defaults to {@link ActiveShardCount#DEFAULT}, which will wait for one shard copy
     * (the primary) to become active. Set this value to {@link ActiveShardCount#ALL} to
     * wait for all shards (primary and all replicas) to be active before returning.
     * Otherwise, use {@link ActiveShardCount#from(int)} to set this value to any
     * non-negative integer, up to the number of copies per shard (number of replicas + 1),
     * to wait for the desired amount of shard copies to become active before returning.
     * Indices opening will only wait up until the timeout value for the number of shard copies
     * to be active before returning.  Check {@link OpenIndexResponse#isShardsAcknowledged()} to
     * determine if the requisite shard copies were all started before returning or timing out.
     *
     * @param waitForActiveShards number of active shard copies to wait on
     */
    public void setWaitForActiveShards(ActiveShardCount waitForActiveShards) {
        this.waitForActiveShards = waitForActiveShards;
    }

}
