/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */
package org.easysearch.client.rollup;

import org.easysearch.action.support.IndicesOptions;
import org.easysearch.client.Validatable;
import org.easysearch.common.Strings;
import org.easysearch.common.xcontent.ToXContentObject;
import org.easysearch.common.xcontent.XContentBuilder;

import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;

public class GetRollupIndexCapsRequest implements Validatable, ToXContentObject {
    private static final String INDICES = "indices";
    private static final String INDICES_OPTIONS = "indices_options";

    private String[] indices;
    private IndicesOptions options;

    public GetRollupIndexCapsRequest(final String... indices) {
        this(indices, IndicesOptions.STRICT_EXPAND_OPEN_FORBID_CLOSED);
    }

    public GetRollupIndexCapsRequest(final String[] indices, final IndicesOptions options) {
        if (indices == null || indices.length == 0) {
            throw new IllegalArgumentException("[indices] must not be null or empty");
        }
        for (String index : indices) {
            if (Strings.isNullOrEmpty(index)) {
                throw new IllegalArgumentException("[index] must not be null or empty");
            }
        }
        this.indices = indices;
        this.options = Objects.requireNonNull(options);
    }

    public IndicesOptions indicesOptions() {
        return options;
    }

    public String[] indices() {
        return indices;
    }

    @Override
    public XContentBuilder toXContent(XContentBuilder builder, Params params) throws IOException {
        builder.startObject();
        {
            builder.array(INDICES, indices);
            builder.startObject(INDICES_OPTIONS);
            {
                options.toXContent(builder, params);
            }
            builder.endObject();
        }
        builder.endObject();
        return builder;
    }

    @Override
    public int hashCode() {
        return Objects.hash(Arrays.hashCode(indices), options);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        GetRollupIndexCapsRequest other = (GetRollupIndexCapsRequest) obj;
        return Arrays.equals(indices, other.indices)
            && Objects.equals(options, other.options);
    }
}
