/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */
package org.easysearch.client.rollup.job.config;

import org.easysearch.client.Validatable;
import org.easysearch.client.ValidationException;
import org.easysearch.common.ParseField;
import org.easysearch.common.xcontent.ConstructingObjectParser;
import org.easysearch.common.xcontent.ToXContentObject;
import org.easysearch.common.xcontent.XContentBuilder;
import org.easysearch.common.xcontent.XContentParser;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import static org.easysearch.common.xcontent.ConstructingObjectParser.constructorArg;

/**
 * The configuration object for the histograms in the rollup config
 *
 * {
 *     "groups": [
 *        "terms": {
 *            "fields" : [ "foo", "bar" ]
 *        }
 *     ]
 * }
 */
public class TermsGroupConfig implements Validatable, ToXContentObject {

    static final String NAME = "terms";
    private static final String FIELDS = "fields";

    private static final ConstructingObjectParser<TermsGroupConfig, Void> PARSER;
    static {
        PARSER = new ConstructingObjectParser<>(NAME, true, args -> {
            @SuppressWarnings("unchecked") List<String> fields = (List<String>) args[0];
            return new TermsGroupConfig(fields != null ? fields.toArray(new String[fields.size()]) : null);
        });
        PARSER.declareStringArray(constructorArg(), new ParseField(FIELDS));
    }

    private final String[] fields;

    public TermsGroupConfig(final String... fields) {
        this.fields = fields;
    }

    @Override
    public Optional<ValidationException> validate() {
        final ValidationException validationException = new ValidationException();
        if (fields == null || fields.length == 0) {
            validationException.addValidationError("Fields must have at least one value");
        }
        if (validationException.validationErrors().isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(validationException);
    }

    /**
     * @return the names of the fields. Never {@code null}.
     */
    public String[] getFields() {
        return fields;
    }

    @Override
    public XContentBuilder toXContent(XContentBuilder builder, Params params) throws IOException {
        builder.startObject();
        {
            builder.field(FIELDS, fields);
        }
        return builder.endObject();
    }

    @Override
    public boolean equals(final Object other) {
        if (this == other) {
            return true;
        }
        if (other == null || getClass() != other.getClass()) {
            return false;
        }
        final TermsGroupConfig that = (TermsGroupConfig) other;
        return Arrays.equals(fields, that.fields);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(fields);
    }

    public static TermsGroupConfig fromXContent(final XContentParser parser) throws IOException {
        return PARSER.parse(parser, null);
    }
}
