/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */

package org.easysearch.client.security;

import org.easysearch.client.Validatable;
import org.easysearch.client.ValidationException;
import org.easysearch.client.security.user.User;
import org.easysearch.common.CharArrays;
import org.easysearch.common.Nullable;
import org.easysearch.common.xcontent.ToXContentObject;
import org.easysearch.common.xcontent.XContentBuilder;

import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;

/**
 * Request object to create or update a user in the native realm.
 */
public final class PutUserRequest implements Validatable, ToXContentObject {

    private final User user;
    private final @Nullable char[] password;

    /**
     * Create or update a user in the native realm, with the user's new or updated password specified in plaintext.
     * @param user the user to be created or updated
     * @param password the password of the user. The password array is not modified by this class.
     *                 It is the responsibility of the caller to clear the password after receiving
     *                 a response.
     */
    public static PutUserRequest withPassword(User user, char[] password) {
        return new PutUserRequest(user, password);
    }


    /**
     * Update an existing user in the native realm without modifying their password.
     * @param user the user to be created or updated
     */
    public static PutUserRequest updateUser(User user) {
        return new PutUserRequest(user, null);
    }



    /**
     * Creates a new request that is used to create or update a user in the native realm.
     * @param user the user to be created or updated
     * @param password the password of the user. The password array is not modified by this class.
     *                 It is the responsibility of the caller to clear the password after receiving
     *                 a response.
     */
    private PutUserRequest(User user, @Nullable char[] password) {
        this.user = Objects.requireNonNull(user, "user is required, cannot be null");
        this.password = password;
    }

    public User getUser() {
        return user;
    }

    public @Nullable char[] getPassword() {
        return password;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        final PutUserRequest that = (PutUserRequest) o;
        return Objects.equals(user, that.user)
                && Arrays.equals(password, that.password);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(user);
        result = 31 * result + Arrays.hashCode(password);
        return result;
    }

    @Override
    public Optional<ValidationException> validate() {
        if (user.getAttributes() != null && user.getAttributes().keySet().stream().anyMatch(s -> s.startsWith("_"))) {
            ValidationException validationException = new ValidationException();
            validationException.addValidationError("user metadata keys may not start with [_]");
            return Optional.of(validationException);
        }
        return Optional.empty();
    }

    @Override
    public XContentBuilder toXContent(XContentBuilder builder, Params params) throws IOException {
        builder.startObject();
        if (password != null) {
            charField(builder, "password", password);
        }

        builder.field("roles", user.getRoles());
        builder.field("external_roles", user.getExternal_roles());
        if (user.getHash() != null) {
            builder.field("hash", user.getHash());
        }
        builder.field("attributes", user.getAttributes());
        return builder.endObject();
    }

    private void charField(XContentBuilder builder, String fieldName, char[] chars) throws IOException {
        byte[] charBytes = CharArrays.toUtf8Bytes(chars);
        try {
            builder.field(fieldName).utf8Value(charBytes, 0, charBytes.length);
        } finally {
            Arrays.fill(charBytes, (byte) 0);
        }
    }
}
