/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */

package org.easysearch.client.security.user;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * A user to be utilized with security APIs.
 * Can be an existing authenticated user or it can be a new user to be enrolled to the native realm.
 */
public final class User {

    @JsonIgnore
    private final String name;
    private String hash;
     private boolean reserved;
    private boolean hidden;
    @JsonProperty(value = "static")
    private boolean _static;
    private final List<String> roles;
    private final List<String> external_roles;
    private final Map<String, Object> attributes;

    public User(String username, List<String> roles, List<String> external_roles, Map<String, Object> attributes) {
        this(username, null, roles, external_roles, attributes, false, false, false);
    }

    public User(String name, String hash, List<String> roles, List<String> external_roles, Map<String, Object> attributes, boolean reserved, boolean hidden, boolean _static) {
        this.name = Objects.requireNonNull(name, "`name` is required, cannot be null");
        this.roles = Objects.requireNonNull(roles, "`roles` is required, cannot be null. Pass an empty Collection instead.");
        this.hash = hash;
        this.external_roles = external_roles;
        this.attributes = attributes;
        this.reserved = reserved;
        this.hidden = hidden;
        this._static = _static;
    }

    public String getName() {
        return name;
    }

    /**
     * @return The roles this user is associated with. The roles are
     * identified by their unique names and each represents as
     * set of permissions. Can never be {@code null}.
     */
    public List<String> getRoles() {
        return this.roles;
    }

    public List<String> getExternal_roles() {
        return external_roles;
    }

    public Map<String, Object> getAttributes() {
        return attributes;
    }

    public boolean isReserved() {
        return reserved;
    }

    public boolean isHidden() {
        return hidden;
    }

    public boolean is_static() {
        return _static;
    }

    public String getHash() {
        return hash;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        User user = (User) o;
        return reserved == user.reserved && hidden == user.hidden
            && _static == user._static && name.equals(user.name) && hash.equals(hash)
            && roles.equals(user.roles) && Objects.equals(external_roles, user.external_roles)
            && Objects.equals(attributes, user.attributes);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, hash, roles, external_roles, attributes, reserved, hidden, _static);
    }
}
