/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */

package org.easysearch.client.security.user.parser;

import org.easysearch.client.security.user.Index;
import org.easysearch.client.security.user.Role;
import org.easysearch.common.xcontent.XContentParser;
import org.easysearch.common.xcontent.XContentParserUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class RoleJsonParser {
    public static Set<Role> parseRoles(XContentParser jsonParser) throws IOException, IllegalArgumentException {
        XContentParserUtils.ensureExpectedToken(XContentParser.Token.START_OBJECT, jsonParser.nextToken(), jsonParser);
        Set<Role> roles = new HashSet<>();
        XContentParser.Token token;
        while ((token = jsonParser.nextToken()) != XContentParser.Token.END_OBJECT) {
            XContentParserUtils.ensureExpectedToken(XContentParser.Token.FIELD_NAME, token, jsonParser);

            String name = jsonParser.currentName();
            jsonParser.nextToken();

            String description = null;
            List<String> cluster = null;
            List<Index> indices = null;
            boolean reserved = false;
            boolean hidden = false;
            boolean isStatic = false;

            while (jsonParser.nextToken() != XContentParser.Token.END_OBJECT) {
                String fieldName = jsonParser.currentName();
                jsonParser.nextToken();

                switch (fieldName) {
                    case "description":
                        description = jsonParser.text();
                        break;
                    case "cluster":
                        cluster = parseStringArray(jsonParser);
                        break;
                    case "indices":
                        indices = parseIndicesArray(jsonParser);
                        break;
                    case "reserved":
                        reserved = jsonParser.booleanValue();
                        break;
                    case "hidden":
                        hidden = jsonParser.booleanValue();
                        break;
                    case "static":
                        isStatic = jsonParser.booleanValue();
                        break;
                    default:
                        //logger.warn("Unexpected field '{}' encountered", fieldName);
                        break;
                }
            }

            Role role = new Role(reserved, hidden, isStatic, name, cluster, indices, description);
            roles.add(role);
        }
        return roles;
    }

    private static List<String> parseStringArray(XContentParser jsonParser) throws IOException {
        List<String> stringList = new ArrayList<>();
        while (jsonParser.nextToken() != XContentParser.Token.END_ARRAY) {
            stringList.add(jsonParser.text());
        }
        return stringList;
    }

    private static List<Index> parseIndicesArray(XContentParser jsonParser) throws IOException {
        List<Index> indices = new ArrayList<>();
        XContentParser.Token token;
        XContentParserUtils.ensureExpectedToken(XContentParser.Token.START_ARRAY, jsonParser.currentToken(), jsonParser);
        while ((token = jsonParser.nextToken()) != XContentParser.Token.END_ARRAY) {
            XContentParserUtils.ensureExpectedToken(XContentParser.Token.START_OBJECT, token, jsonParser);
            Index index = Index.fromXContent(jsonParser);
            indices.add(index);
        }
        return indices;
    }
}
