/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */
package org.easysearch.client.tasks;

import org.easysearch.client.Validatable;
import org.easysearch.client.ValidationException;
import org.easysearch.common.unit.TimeValue;

import java.util.Objects;
import java.util.Optional;

public class GetTaskRequest implements Validatable {
    private final String nodeId;
    private final long taskId;
    private boolean waitForCompletion = false;
    private TimeValue timeout = null;

    public GetTaskRequest(String nodeId, long taskId) {
        this.nodeId = nodeId;
        this.taskId = taskId;
    }

    public String getNodeId() {
        return nodeId;
    }

    public long getTaskId() {
        return taskId;
    }

    /**
     * Should this request wait for all found tasks to complete?
     */
    public boolean getWaitForCompletion() {
        return waitForCompletion;
    }

    /**
     * Should this request wait for all found tasks to complete?
     */
    public GetTaskRequest setWaitForCompletion(boolean waitForCompletion) {
        this.waitForCompletion = waitForCompletion;
        return this;
    }

    /**
     * Timeout to wait for any async actions this request must take. It must take anywhere from 0 to 2.
     */
    public TimeValue getTimeout() {
        return timeout;
    }

    /**
     * Timeout to wait for any async actions this request must take.
     */
    public GetTaskRequest setTimeout(TimeValue timeout) {
        this.timeout = timeout;
        return this;
    }

    @Override
    public Optional<ValidationException> validate() {
        final ValidationException validationException = new ValidationException();
        if (timeout != null && !waitForCompletion) {
            validationException.addValidationError("Timeout settings are only accepted if waitForCompletion is also set");
        }
        if (validationException.validationErrors().isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(validationException);
    }

    @Override
    public int hashCode() {
        return Objects.hash(nodeId, taskId, waitForCompletion, timeout);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        GetTaskRequest other = (GetTaskRequest) obj;
        return Objects.equals(nodeId, other.nodeId) &&
                taskId == other.taskId &&
                waitForCompletion == other.waitForCompletion &&
                Objects.equals(timeout, other.timeout);
    }
}
