/* Copyright © INFINI Ltd. All rights reserved.
 * Web: https://infinilabs.com
 * Email: hello#infini.ltd */
package org.easysearch.client.tasks;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * Client side counterpart of server side version.
 *
 * {@link org.easysearch.action.admin.cluster.node.tasks.list.TaskGroup}
 */
public class TaskGroup {

    private final TaskInfo task;

    @Override
    public String toString() {
        return "TaskGroup{" +
            "task=" + task +
            ", childTasks=" + childTasks +
            '}';
    }

    private final List<TaskGroup> childTasks = new ArrayList<>();

    public TaskGroup(TaskInfo task, List<TaskGroup> childTasks) {
        this.task = task;
        this.childTasks.addAll(childTasks);
    }

    public static TaskGroup.Builder builder(TaskInfo taskInfo) {
        return new TaskGroup.Builder(taskInfo);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof TaskGroup)) return false;
        TaskGroup taskGroup = (TaskGroup) o;
        return Objects.equals(task, taskGroup.task) &&
            Objects.equals(getChildTasks(), taskGroup.getChildTasks());
    }

    @Override
    public int hashCode() {
        return Objects.hash(task, getChildTasks());
    }

    public static class Builder {
        private TaskInfo taskInfo;
        private List<TaskGroup.Builder> childTasks;

        private Builder(TaskInfo taskInfo) {
            this.taskInfo = taskInfo;
            childTasks = new ArrayList<>();
        }

        public void addGroup(TaskGroup.Builder builder) {
            childTasks.add(builder);
        }

        public TaskInfo getTaskInfo() {
            return taskInfo;
        }

        public TaskGroup build() {
            return new TaskGroup(
                taskInfo,
                childTasks.stream().map(TaskGroup.Builder::build).collect(Collectors.toList())
            );
        }
    }

    public TaskInfo getTaskInfo() {
        return task;
    }

    public List<TaskGroup> getChildTasks() {
        return childTasks;
    }
}

