package com.itv.scalapact.shared.pact

import argonaut.Argonaut._
import argonaut._
import com.itv.scalapact.shared._

object PactReader extends IPactReader {

  def jsonStringToPact(json: String): Either[String, Pact] = {
    val brokenPact: Option[(PactActor, PactActor, List[(Option[Interaction], Option[String], Option[String])])] = for {
      provider     <- JsonBodySpecialCaseHelper.extractPactActor("provider")(json)
      consumer     <- JsonBodySpecialCaseHelper.extractPactActor("consumer")(json)
      interactions <- JsonBodySpecialCaseHelper.extractInteractions(json)
    } yield (provider, consumer, interactions)

    brokenPact.map { bp =>
      val interactions = bp._3.collect {
        case (Some(i), r1, r2) =>
          i.copy(
            request = i.request.copy(body = r1),
            response = i.response.copy(body = r2)
          )
      }

      Pact(
        provider = bp._1,
        consumer = bp._2,
        interactions = interactions
          .map(i => i.copy(providerState = i.providerState.orElse(i.provider_state)))
          .map(i => i.copy(provider_state = None))
      )

    } match {
      case Some(pact) => Right(pact)
      case None       => Left(s"Could not read pact from json: $json")
    }
  }

}

object JsonBodySpecialCaseHelper {

  import PactImplicits._

  val extractPactActor: String => String => Option[PactActor] = field =>
    json =>
      json.parseOption
        .flatMap { j =>
          (j.hcursor --\ field).focus
        }
        .flatMap(p => p.toString.decodeOption[PactActor])

  val extractInteractions: String => Option[List[(Option[Interaction], Option[String], Option[String])]] = json => {

    val interations =
      json.parseOption
        .flatMap { j =>
          (j.hcursor --\ "interactions").focus.flatMap(_.array)
        }

    val makeOptionalBody: Json => Option[String] = {
      case body: Json if body.isString =>
        body.string.map(_.toString)

      case body =>
        Option(body.toString)
    }

    interations.map { is =>
      is.map { i =>
        val minusRequestBody =
          (i.hcursor --\ "request" --\ "body").delete.undo match {
            case ok @ Some(s) => ok
            case None         => Option(i)
          }

        val minusResponseBody = minusRequestBody.flatMap { ii =>
          (ii.hcursor --\ "response" --\ "body").delete.undo match {
            case ok @ Some(s) => ok
            case None         => minusRequestBody // There wasn't a body, but there was still an interaction.
          }
        }

        val requestBody = (i.hcursor --\ "request" --\ "body").focus
          .flatMap { makeOptionalBody }

        val responseBody = (i.hcursor --\ "response" --\ "body").focus
          .flatMap { makeOptionalBody }

        (minusResponseBody.flatMap(p => p.toString.decodeOption[Interaction]), requestBody, responseBody)
      }
    }
  }

}
